/******************************************************************************
  Copyright (c) 2012 by Turku PET Centre

  File:        mask.c
  Description: Contains functions for extracting TACs from image data
               based mask image.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Version history:
  2012-12-12 VO
      First created.
  2012-12-26 VO
      Added function imgMaskPixelTACs().

******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "include/idi.h"
/*****************************************************************************/

/*****************************************************************************/
/** Calculate TAC as weighted average of voxels in specified image data
 *  with relative weights given in a mask image.
\return Returns 0 if successful.
 */
int imgMaskTAC(
  /** Pointer to allocated image from which the weighted TAC is calculated. */
  IMG *img,
  /** Pointer to mask image; x, y, and z dimensions must be the same as in
   *  the image to which the mask is applied. */
  IMG *mask,
  /** Pointer to an array where weighted pixel averages are written;
   *  it must be allocated for size >= dimt. */
  double *tac,
  /** Verbose level; set to <=0 to prevent all prints to stdout */
  int verbose
) {
  if(verbose>0) printf("imgMaskTAC()\n");

  int xi, yi, zi, fi;
  double w;

  if(img->status<IMG_STATUS_OCCUPIED) return(1);
  if(mask->status<IMG_STATUS_OCCUPIED) return(2);
  if(mask->dimz!=img->dimz) return(3);
  if(mask->dimy!=img->dimy) return(4);
  if(mask->dimx!=img->dimx) return(5);
  if(img->dimt<1 || mask->dimt<1) return(6);
  if(tac==NULL) return(7);

  /* Initiate TAC */
  for(fi=0; fi<img->dimt; fi++) tac[fi]=0.0;
  /* Add weighted sum to TAC */
  w=0.0;
  for(zi=0; zi<mask->dimz; zi++)
    for(yi=0; yi<mask->dimy; yi++)
      for(xi=0; xi<mask->dimx; xi++) if(mask->m[zi][yi][xi][0]>0.0) {
        for(fi=0; fi<img->dimt; fi++)
          tac[fi]+=mask->m[zi][yi][xi][0]*img->m[zi][yi][xi][fi];
        w+=mask->m[zi][yi][xi][0];
      }
  /* Divide sum TAC by sum weights */
  for(fi=0; fi<img->dimt; fi++) tac[fi]/=w;
  if(verbose>1) printf("mask_sum := %g\n", w);
  if(w<=0.0 && verbose>0)
    fprintf(stderr, "Warning: zero mask applied to image.\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Extract TACs of every image voxel which has a value > thrs in the mask
 *  image.
 *  Mask value is saved as region 'size' in DFT.
 *  Mainly for testing IDI methods.
\return Returns 0 if successful.
 */
int imgMaskPixelTACs(
  /** Pointer to allocated image from which the weighted TAC is calculated. */
  IMG *img,
  /** Pointer to mask image; x, y, and z dimensions must be the same as in
   *  the image to which the mask is applied. */
  IMG *mask,
  /** Mask threshold: pixels with mask values above this are included in TACs */
  double thrs,
  /** Pointer to initiated DFT struct where pixel TACs will be written;
   *  any previous contents are deleted. */
  DFT *dft,
  /** Verbose level; set to <=0 to prevent all prints to stdout */
  int verbose
) {
  if(verbose>0) printf("imgMaskPixelTACs()\n");

  int xi, yi, zi, fi, ri, nr, ret;

  if(img->status<IMG_STATUS_OCCUPIED) return(1);
  if(mask->status<IMG_STATUS_OCCUPIED) return(2);
  if(mask->dimz!=img->dimz) return(3);
  if(mask->dimy!=img->dimy) return(4);
  if(mask->dimx!=img->dimx) return(5);
  if(img->dimt<1 || mask->dimt<1) return(6);
  if(dft==NULL) return(7);

  /* Delete any previous TAC data; do it first so that caller can check
     the nr of extracted TACs inside DFT */
  dftEmpty(dft);

  /* Calculate the nr of voxels to extract */
  nr=0;
  for(zi=0; zi<mask->dimz; zi++)
    for(yi=0; yi<mask->dimy; yi++)
      for(xi=0; xi<mask->dimx; xi++)
        if(mask->m[zi][yi][xi][0]>thrs) nr++;
  if(verbose>1) printf("mask_pixel_nr := %d\n", nr);
  /* Return, if no voxels were marked in mask image */
  if(nr==0) return(0);

  /* Allocate memory for pixel TACs */
  ret=dftAllocateWithIMG(dft, nr, img);
  if(ret!=0) {
    if(verbose>0)
      fprintf(stderr, "Error: cannot allocate memory for %d pixel TACs.\n", nr);
    return(10);
  }
  strcpy(dft->studynr, img->studyNr);

  /* Save the pixel TACs in DFT */
  ri=0;
  for(zi=0; zi<mask->dimz; zi++)
    for(yi=0; yi<mask->dimy; yi++)
      for(xi=0; xi<mask->dimx; xi++) if(mask->m[zi][yi][xi][0]>thrs)
  {
    sprintf(dft->voi[ri].voiname, "%d", 1+xi);
    sprintf(dft->voi[ri].hemisphere, "%d", 1+yi);
    sprintf(dft->voi[ri].place, "%d", 1+zi);
    sprintf(dft->voi[ri].name, "%d %d %d", 1+xi, 1+yi, 1+zi);
    for(fi=0; fi<img->dimt; fi++)
      dft->voi[ri].y[fi]=img->m[zi][yi][xi][fi];
    dft->voi[ri].size=mask->m[zi][yi][xi][0];
    ri++;
  }
  dft->voiNr=ri;

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
