/******************************************************************************
  Copyright (c) 2012 by Turku PET Centre

  File:        recovery.c
  Description: Contains the functions for computing recovery errors for vessel.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Version history:
  2012-10-24 Vesa Oikonen
    First created.
  2012-12-02 VO
    Added idiSimulateTubeImg().
  2012-12-04 VO
    Added function idiSimulateTubeImgPlane().
    Function idiSimulateTubeVol() calculates the value not in the middle of
    pixel, but as an average of four samples inside the pixel.
    Function idiSimulateTubeImg() simulate the circle without using sqrt(),
    and also sets pixel value as an average of four samples inside the pixel.
  2012-12-12 VO
    Vessel center points are given not as pixels but as mm from upper left
    corner of image in idiSimulateTubeVol(), idiSimulateTubeImg(), and
    idiSimulateTubeImgPlane().
  2012-12-20 VO
    Vessel simulation functions moved into a new source file vessel.c.
     


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "imgfilter.h"
#include "img.h"
#include "vol.h"
/*****************************************************************************/
#include "include/idi.h"
/*****************************************************************************/

/*****************************************************************************/
/** Calculate the traditional recovery coefficient (RC) of peak value of
 *  a circular radioactive object in 2D PET image, assuming that the object
 *  length in 3D is relatively long, and assuming that activity of background
 *  is zero.
 *  
 *  True object activity A0 can be calculated using RC, backgroud activity Abkg,
 *  and measured peak activity Aexp using equation
 *       A0 = Abkg + (Aexp - Abkg)/RC
 *  , or if Abkg=0,
 *       A0 = Aexp/RC.
 *  
 *  References: Germano et al. JNM 1992; 33: 613-620 and
 *  Brix et al. Nuklearmedizin 2002;41:184-190.
\return Returns the recovery coefficient [0-1].
 */
double rcPeakPET(
  /** Full-width half-maximum value */
  double FWHM,
  /** Radius of the object */
  double R
) {
  double s, rc;
  s=FWHM/2.354820; // 2*sqrt(2*ln(2))
  rc=1.0-exp(-R*R/(2.0*s*s));
  return(rc);
}
/*****************************************************************************/

/*****************************************************************************/
