/******************************************************************************
  Copyright (c) 2012 by Turku PET Centre

  File:        vessel.c
  Description: Contains the functions for simulating image of vessel (tube).

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Version history:
  2012-12-20 Vesa Oikonen
      First created, contents moved from other source files in this library.
      Bug fix in idiSimulateTubeImgPlane() and idiSimulateTubeImg(): Gaussian
      smoothing is done only to the specified image plane.
    
     


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "imgfilter.h"
#include "img.h"
#include "vol.h"
/*****************************************************************************/
#include "include/idi.h"
/*****************************************************************************/

/*****************************************************************************/
/** Simulate image volume surrounding a circular object extending across
 *  image planes (bar or vessel). Based on Germano et al. JNM 1992; 33: 613-620.
 *  This equation can not be used to estimate recovery coefficient or the true
 *  activity concentration inside the vessel, but only to fit the radius
 *  of the vessel (see Germano), which then can be used to calculate the RC.
 *  
 *  To simulate the circular vessel correctly in 2D image matrix use
 *  the equations in Brix et al. Nuklearmedizin 2002;41:184-190 instead;
 *  however, that would require numerical solution to double integrals,
 *  which may be either slow or error-prone for fitting purposes.
 *
 *  In this version, the activity is calculated as an average of four samples
 *  inside the pixel.
 *  
\return Returns 0 if successful, or <> 0 in case of an error.
 */
int idiSimulateTubeVol(
  /** Pointer to allocated image volume; volume must contain pixel sizes
   *  and dimensions */
  VOL *vol,
  /** Plane index [0..dimz-1] */
  int zi,
  /** X distance of circle center (mm) from the upper left corner of the image 
   */
  double cx,
  /** Y distance of circle center (mm) from the upper left corner of the image 
   */
  double cy, 
  /** Radius of vessel (mm) */
  double r,
  /** FWHM (mm) */
  double FWHM,
  /** Background activity */
  double cbkg,
  /** Vessel activity; can be higher or lower than background */
  double cblo
) {
  double s, dx, dy, d, v;
  int xi, yi;

  if(vol->status!=IMG_STATUS_OCCUPIED) return(1);
  if(zi<0 || zi>=vol->dimz) return(2);
  if(vol->sizey<=0.0 || vol->sizex<=0.0) return(3);

  s=FWHM/2.354820; // 2*sqrt(2*ln(2))
  v=cblo-cbkg;
  for(yi=0; yi<vol->dimy; yi++) {
    dy=(0.5+(double)yi)*vol->sizey - cy;
    for(xi=0; xi<vol->dimx; xi++) {
      dx=(0.5+(double)xi)*vol->sizex - cx;
      vol->v[zi][yi][xi]=0.0;
      d=hypot(dx-0.25*vol->sizex, dy-0.25*vol->sizey);
      vol->v[zi][yi][xi]+=
        (v/2.0)*( erf((d+r)/(M_SQRT2*s)) - erf((d-r)/(M_SQRT2*s)) );
      d=hypot(dx-0.25*vol->sizex, dy+0.25*vol->sizey);
      vol->v[zi][yi][xi]+=
        (v/2.0)*( erf((d+r)/(M_SQRT2*s)) - erf((d-r)/(M_SQRT2*s)) );
      d=hypot(dx+0.25*vol->sizex, dy-0.25*vol->sizey);
      vol->v[zi][yi][xi]+=
        (v/2.0)*( erf((d+r)/(M_SQRT2*s)) - erf((d-r)/(M_SQRT2*s)) );
      d=hypot(dx+0.25*vol->sizex, dy+0.25*vol->sizey);
      vol->v[zi][yi][xi]+=
        (v/2.0)*( erf((d+r)/(M_SQRT2*s)) - erf((d-r)/(M_SQRT2*s)) );
      vol->v[zi][yi][xi]*=0.25;
      vol->v[zi][yi][xi]+= cbkg;
    }
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Simulate dynamic image surrounding a circular object extending across
 *  image planes (bar or vessel). Based on 2D Gaussian smoothing.
 *     
\return Returns 0 if successful, or <> 0 in case of an error.
 */
int idiSimulateTubeImg(
  /** Pointer to allocated dynamic image; volume must contain pixel sizes
   *  and dimensions, and the same time frames as the TAC data */
  IMG *img,
  /** Plane index [0..dimz-1] */
  int zi,
  /** X distance of circle center (mm) from the upper left corner of the image 
   */
  double cx,
  /** Y distance of circle center (mm) from the upper left corner of the image 
   */
  double cy, 
  /** Radius of vessel (mm) */
  double r,
  /** FWHM (mm) */
  double FWHM,
  /** Array of Background activities; size must be the same as img frame nr */
  double *cbkg,
  /** Array of Vessel activities; can be higher or lower than background;
   *  size must be the same as img frame nr */
  double *cblo
) {
  double s, dx1, dy1, dx2, dy2, d, d1, d2, d3, d4, r2;
  int xi, yi, fi, n, ret;

  if(img->status!=IMG_STATUS_OCCUPIED) return(1);
  if(zi<0 || zi>=img->dimz) return(2);
  if(img->sizey<=0.0 || img->sizex<=0.0) return(3);

  /* Simulate the circle */
  r2=r*r;
  for(yi=0; yi<img->dimy; yi++) {
    dy1=dy2=(0.5+(double)yi)*img->sizey - cy;
    dy1-=0.25*img->sizey; dy2+=0.25*img->sizey;
    for(xi=0; xi<img->dimx; xi++) {
      dx1=dx2=(0.5+(double)xi)*img->sizex - cx;
      dx1-=0.25*img->sizex; dx2+=0.25*img->sizex;
      d1=dx1*dx1+dy1*dy1; d2=dx1*dx1+dy2*dy2;
      d3=dx2*dx2+dy1*dy1; d4=dx2*dx2+dy2*dy2;
      n=0; if(d1<=r2) n++; if(d2<=r2) n++; if(d3<=r2) n++; if(d4<=r2) n++;
      for(fi=0; fi<img->dimt; fi++)
        img->m[zi][yi][xi][fi]=
          0.25*((double)n*cblo[fi]+(double)(4-n)*cbkg[fi]);
    }
  }

  /* Convert FWHM in mm to filter SD in pixels */
  d=0.5*(img->sizex+img->sizey);
  s=(FWHM/2.354820)/d; // printf("filter_sd_in_pixels := %g\n", s);
  /* Apply Gaussian filter */
  ret=imgGaussianFilter(img, zi, -1, s, 0, 1, 0, NULL);
  if(ret!=0) return(100+ret);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Simulate image surrounding a circular object extending across
 *  image planes (bar or vessel). This function applies currently two
 *  methods to simulate the spill-over and spill-in effects:
 *  
 *  Method 1. is based on Germano et al. JNM 1992; 33: 613-620.
 *  This equation can not be used to estimate recovery coefficient or the true
 *  activity concentration inside the vessel, but only to fit the radius
 *  of the vessel (see Germano), which then can be used to calculate the RC.
 *  
 *  Method 2. is based on 2D Gaussian smoothing, which is much slower,
 *  but may be more correct in case of small vessels.
 *  
 *  To simulate the circular vessel correctly in 2D image matrix use
 *  the equations in Brix et al. Nuklearmedizin 2002;41:184-190 instead;
 *  however, that would require numerical solution to double integrals,
 *  which may be either slow or error-prone for fitting purposes.
 *     
\return Returns 0 if successful, or <> 0 in case of an error.
 */
int idiSimulateTubeImgPlane(
  /** Simulation method: 0=Germano, 1=Gaussian2D */
  int simmet,
  /** Pointer to allocated dynamic image; volume must contain pixel sizes
   *  and dimensions, and the same time frames as the TAC data */
  IMG *img,
  /** Plane index [0..dimz-1] */
  int zi,
  /** X distance of circle center (mm) from the upper left corner of the image 
   */
  double cx,
  /** Y distance of circle center (mm) from the upper left corner of the image 
   */
  double cy, 
  /** Radius of vessel (mm) */
  double r,
  /** FWHM (mm) */
  double FWHM,
  /** Array of Background activities; size must be the same as img frame nr */
  double *cbkg,
  /** Array of Vessel activities; can be higher or lower than background;
   *  size must be the same as img frame nr */
  double *cblo
) {
  double s, d;
  int xi, yi, fi, ret;

  if(img->status!=IMG_STATUS_OCCUPIED) return(1);
  if(zi<0 || zi>=img->dimz) return(2);
  if(img->sizey<=0.0 || img->sizex<=0.0) return(3);
  if(cbkg==NULL || cblo==NULL) return(4);

  s=FWHM/2.354820; // 2*sqrt(2*ln(2))

  if(simmet==0) { // Germano
    double v, w1, w2, w3, w4, dy, dx;
    for(yi=0; yi<img->dimy; yi++) {
      dy=(0.5+(double)yi)*img->sizey - cy;
      for(xi=0; xi<img->dimx; xi++) {
        dx=(0.5+(double)xi)*img->sizex - cx;
        d=hypot(dx-0.25*img->sizex, dy-0.25*img->sizey);
        w1=0.5*( erf((d+r)/(M_SQRT2*s)) - erf((d-r)/(M_SQRT2*s)) );
        d=hypot(dx-0.25*img->sizex, dy+0.25*img->sizey);
        w2=0.5*( erf((d+r)/(M_SQRT2*s)) - erf((d-r)/(M_SQRT2*s)) );
        d=hypot(dx+0.25*img->sizex, dy-0.25*img->sizey);
        w3=0.5*( erf((d+r)/(M_SQRT2*s)) - erf((d-r)/(M_SQRT2*s)) );
        d=hypot(dx+0.25*img->sizex, dy+0.25*img->sizey);
        w4=0.5*( erf((d+r)/(M_SQRT2*s)) - erf((d-r)/(M_SQRT2*s)) );
        for(fi=0; fi<img->dimt; fi++) {
          v=cblo[fi]-cbkg[fi];
          img->m[zi][yi][xi][fi]= 0.25*v*(w1+w2+w3+w4) + cbkg[fi];
        }
      }
    }
  } else { // 2D Gaussian 
    double dx1, dy1, dx2, dy2, d1, d2, d3, d4, r2;
    int n;
    /* Simulate the circle */
    r2=r*r;
    for(yi=0; yi<img->dimy; yi++) {
      dy1=dy2=(0.5+(double)yi)*img->sizey - cy;
      dy1-=0.25*img->sizey; dy2+=0.25*img->sizey;
      for(xi=0; xi<img->dimx; xi++) {
        dx1=dx2=(0.5+(double)xi)*img->sizex - cx;
        dx1-=0.25*img->sizex; dx2+=0.25*img->sizex;
        d1=dx1*dx1+dy1*dy1; d2=dx1*dx1+dy2*dy2;
        d3=dx2*dx2+dy1*dy1; d4=dx2*dx2+dy2*dy2;
        n=0; if(d1<=r2) n++; if(d2<=r2) n++; if(d3<=r2) n++; if(d4<=r2) n++;
        for(fi=0; fi<img->dimt; fi++)
          img->m[zi][yi][xi][fi]=
            0.25*((double)n*cblo[fi]+(double)(4-n)*cbkg[fi]);
      }
    }
    /* Convert filter SD to pixels */
    d=0.5*(img->sizex+img->sizey);
    s/=d; // printf("filter_sd_in_pixels := %g\n", s);
    /* Apply Gaussian filter */
    ret=imgGaussianFilter(img, zi, -1, s, 0, 1, 0, NULL);
    if(ret!=0) return(100+ret);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
