/******************************************************************************

  Copyright (c) 2010,2012 Turku PET Centre

  Program:     dft2csv
  Description: converting DFT files to CSV format

  This program is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation; either version 2 of the License, or (at your option) any later
  version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 Temple
  Place, Suite 330, Boston, MA 02111-1307 USA.

  Turku PET Centre hereby disclaims all copyright interest in the program.
  Juhani Knuuti
  Director, Professor
  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/

  Modification history:
  0.1.0 2010-06-23 Vesa Oikonen
        First created.
  0.1.1 2010-06-30 VO
        Bug fix in writing of title line.
  0.2.0 2010-11-04 VO
        Added option to save data with tab delimiters.
        Added option to save data without any titles.
  0.2.1 2010-12-28 VO
        Data lines with NaN in time or all sample values is not written.
        NaN is written with . in tab delimited files, and otherwise nothing
        is written between limiting characters.
  0.2.2 2012-02-28 VO
        Updated print_build().
  0.3.0 2012-03-05 VO
        Supports also IFT files.
  0.3.1 2012-03-16 VO
        Added option -mid to save only frame mid times.
        User info now suggests using tabs for importing data into Origin. 
       


******************************************************************************/

/*****************************************************************************/
#define PROGRAM "dft2csv 0.3.1  (c) 2010,2012 by Turku PET Centre"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
/*****************************************************************************/
#include "curveio.h"
#include "libtpcmisc.h"
/*****************************************************************************/
int TEST=0;
int SILENT=0;
int VERBOSE=0;
/*****************************************************************************/
/* Local functions */
void print_usage();
void print_build();
/*****************************************************************************/

/*****************************************************************************/
/** Writes IFT struct data in CSV format.
\return Returns 0 when successful, otherwise <>0.
 */
int iftWriteAsCSV(
  /** Data that will be written; IFT contents are edited here! */
  IFT *ift,
  /** Filename for CSV file */
  char *fname,
  /** CSV separator: 0=comma, 1=semi-colon, 2=tab */
  int csv_separator,
  /** Decimal separator; 0=point (.), 1=comma (,) */  
  int decimal_separator,
  /** Pointer to a string (allocated for at least 64 chars) where error message
      or other execution status will be written; enter NULL, if not needed */     
  char *status
) {
  int li, n;
  char *cptr, *cptr2, tmp[256];

  /* Check the input */
  if(status!=NULL) sprintf(status, "program error");
  if(ift==NULL || strlen(fname)==0) return -1;
  if(ift->keyNr<1) {
    if(status!=NULL) sprintf(status, "no data to write");
    return 1;
  }
  if(csv_separator==0 && decimal_separator==1) {
    if(status!=NULL)
      sprintf(status, "invalid combination of decimal and column separators");
    return 1;
  }

  /* Set the separator */
  if(csv_separator==0) ift->type=6;      // ,
  else if(csv_separator==1) ift->type=7; // ;
  else if(csv_separator==2) ift->type=5; // tab

  /* Check IFT for decimal separators */
  for(li=0; li<ift->keyNr; li++) {
    /* Is item key a valid number? */
    cptr=strPtrToNextValue(ift->item[li].key, &cptr2);
    if(cptr!=NULL) {
      n=strlen(cptr); if(cptr2!=NULL) n-=strlen(cptr2);
      strncpy(tmp, cptr, n); tmp[n]=(char)0;
      /* if contains separator and its the wrong one, then change it */
      n=dec_separator(tmp); if(n!=0) {
        if((--n)!=decimal_separator)
          dec_separator_change(cptr, decimal_separator);
      }
    }
    /* Search the value string for numerical representations */
    cptr=strPtrToNextValue(ift->item[li].value, &cptr2);
    while(cptr!=NULL) {
      n=strlen(cptr); if(cptr2!=NULL) n-=strlen(cptr2);
      strncpy(tmp, cptr, n); tmp[n]=(char)0;
      n=dec_separator(tmp); if(n!=0) {
        if((--n)!=decimal_separator)
          dec_separator_change(cptr, decimal_separator);
      }
      if(cptr2==NULL || !*cptr2) break;
      if(*cptr2==' ' || *cptr2=='\t' || *cptr2==';' || *cptr2==',') {
        /* Replace the separator */
        if(csv_separator==0) *cptr2=',';
        else if(csv_separator==1) *cptr2=';';
        else if(csv_separator==2) *cptr2='\t';
      }
      cptr=cptr2; cptr=strPtrToNextValue(cptr, &cptr2);
    }
  }

  /* Write file */
  if(iftWrite(ift, fname)!=0) {
    if(status!=NULL) strcpy(status, "cannot write file");
    return(2);
  }

  if(status!=NULL) sprintf(status, "ok");
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Writes DFT struct data in CSV format.
\return Returns 0 when successful, otherwise <>0.
 */
int dftWriteAsCSV(
  /** Data that will be written */
  DFT *dft,
  /** Filename for CSV file */
  char *fname,
  /** CSV separator: 0=comma, 1=semi-colon, 2=tab */
  int csv_separator,
  /** Decimal separator; 0=point (.), 1=comma (,) */  
  int decimal_separator,
  /** Include header information; 0=no, 1=yes */
  int include_header, 
  /** Pointer to a string (allocated for at least 64 chars) where error message
      or other execution status will be written; enter NULL, if not needed */     
  char *status
) {
  int ri, fi, n, is_stdout=0;
  char *cptr, ss[2], tmp[FILENAME_MAX];
  FILE *fp;

  /* Check the input */
  if(status!=NULL) sprintf(status, "program error");
  if(dft==NULL || strlen(fname)==0) return -1;
  if(dft->frameNr<1 || dft->voiNr<1) {
    if(status!=NULL) sprintf(status, "no data to write");
    return 1;
  }
  if(csv_separator==0 && decimal_separator==1) {
    if(status!=NULL)
      sprintf(status, "invalid combination of decimal and column separators");
    return 1;
  }
  /* Check if writing to stdout */
  if(!strcasecmp(fname, "stdout")) is_stdout=1;
  /* Set the separator */
  if(csv_separator==0) strcpy(ss, ",");
  else if(csv_separator==1) strcpy(ss, ";");
  else strcpy(ss, "\t");

  /* Check if file exists; backup, if necessary */
  if(!is_stdout && access(fname, 0) != -1) {
    strcpy(tmp, fname); strcat(tmp, BACKUP_EXTENSION); rename(fname, tmp);}

  /* Open output file */
  if(is_stdout) fp=(FILE*)stdout;
  else if((fp=fopen(fname, "w"))==NULL) {
    if(status!=NULL) sprintf(status, "cannot open file");
    return(2);
  }

  /* Write the title lines, if there are titles */
  if(include_header>0 && dft->_type>0) {
    if(dft->timetype==0)
      fprintf(fp, "Time (%s)", petTunit(dft->timeunit));
    else if(dft->timetype==1)
      fprintf(fp, "Start time (%s)", petTunit(dft->timeunit));
    else if(dft->timetype==2)
      fprintf(fp, "End time (%s)", petTunit(dft->timeunit));
    else
      fprintf(fp, "Start time (%s)%sEnd time (%s)",
              petTunit(dft->timeunit), ss, petTunit(dft->timeunit));
    /* Region names */
    for(ri=0; ri<dft->voiNr; ri++) {
      fprintf(fp, "%s%s", ss, dft->voi[ri].voiname);
      if(strlen(dft->voi[ri].hemisphere)>0 && strcmp(dft->voi[ri].hemisphere, ".")!=0)
        fprintf(fp, " %s", dft->voi[ri].hemisphere);
      if(strlen(dft->voi[ri].place)>0 && strcmp(dft->voi[ri].place, ".")!=0)
        fprintf(fp, " %s", dft->voi[ri].place);
    }
    fprintf(fp, "\n");
  }

  /* Write data lines */
  for(fi=0; fi<dft->frameNr; fi++) {
    /* Write time */
    if(dft->timetype==0 || (dft->timetype==3 && include_header==0)) {
      if(isnan(dft->x[fi])) continue;
      sprintf(tmp, "%g", dft->x[fi]);
    } else if(dft->timetype==1) {
      if(isnan(dft->x1[fi])) continue;
      sprintf(tmp, "%g", dft->x1[fi]);
    } else if(dft->timetype==2) {
      if(isnan(dft->x2[fi])) continue;
      sprintf(tmp, "%g", dft->x2[fi]);
    } else {
      if(isnan(dft->x1[fi]) || isnan(dft->x2[fi])) continue;
      sprintf(tmp, "%g%s%g", dft->x1[fi], ss, dft->x2[fi]);
    }
    /* Check if all sample values are NaN */
    for(ri=n=0; ri<dft->voiNr; ri++) if(!isnan(dft->voi[ri].y[fi])) n++;
    if(n==0) continue; // do not write this line at all
    /* Convert all decimal points to commas, if necessary */
    if(decimal_separator!=0)  
      while((cptr=strchr(tmp, '.'))!=NULL) *cptr=',';
    fprintf(fp, "%s", tmp);
    /* Write values */
    for(ri=0; ri<dft->voiNr; ri++) {
      if(!isnan(dft->voi[ri].y[fi])) {
        // Not NaN
        sprintf(tmp, "%s%g", ss, dft->voi[ri].y[fi]);
        if(decimal_separator!=0)  /* Convert all decimal dots to commas */
          while((cptr=strchr(tmp, '.'))!=NULL) *cptr=',';
      } else if(csv_separator==2) {
        // with tab separator, write '.' to indicate missing value
        sprintf(tmp, "%s.", ss);
      } else {
        // with other separators separator, write nothing but the separator
        sprintf(tmp, "%s", ss);
      }
      fprintf(fp, "%s", tmp);
    }
    fprintf(fp, "\n");
  }

  /* Close file */
  if(!is_stdout) fclose(fp);
  if(status!=NULL) sprintf(status, "ok");
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/*
 *  main()
 */
int main(int argc, char *argv[])
{
  int ai;
  int csv_separator=0; // 0=comma, 1=semi-colon, 2=tab
  int decimal_separator=0; // 0=point, 1=comma
  int include_header=1; // 0=no, 1=yes
  int timetype=-1; // DFT timetype; <0: not changed
  char *cptr, dftfile[FILENAME_MAX], outfile[FILENAME_MAX], tmp[FILENAME_MAX];
  DFT data;
  IFT ift;

//test_strPtrToNextValue(); return(0);

  /*
   *  Get and check arguments
   */
  if(argc==1) {print_usage(); return(1);}
  /* Get command line arguments and options */
  dftfile[0]=outfile[0]=(char)0;
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* Options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(strncasecmp(cptr, "TEST", 4)==0) {
      cptr+=4; if(*cptr=='=') cptr++; TEST=atoi(cptr); if(TEST<1) TEST=1;
      continue;
    } else if(strcasecmp(cptr, "V")==0 || strcasecmp(cptr, "VERSION")==0 ||
              strcasecmp(cptr, "BUILD")==0) {
      print_build();
      return(0);
    } else if(strcasecmp(cptr, "H")==0 || strcasecmp(cptr, "HELP")==0) {
      print_usage(); return(0);
    } else if(strcasecmp(cptr, "SILENT")==0) {
      SILENT=1; VERBOSE=0; continue;
    } else if(strcasecmp(cptr, "VERBOSE")==0) {
      VERBOSE=1; SILENT=0; continue;
    } else if(strncasecmp(cptr, "SEPARATOR=", 10)==0) {
      cptr+=10;
      if(strncasecmp(cptr, "COMMA", 1)==0) {
        csv_separator=0; decimal_separator=0; continue;
      } else if(strncasecmp(cptr, "SEMI-COLON", 1)==0) {
        csv_separator=1; decimal_separator=1; continue;
      } else if(strncasecmp(cptr, "TABULATOR", 1)==0) {
        csv_separator=2; decimal_separator=0; continue;
      }
    } else if(strncasecmp(cptr, "HEADER=", 7)==0) {
      cptr+=7;
      if(strncasecmp(cptr, "YES", 1)==0) {
        include_header=1; continue;
      } else if(strncasecmp(cptr, "NO", 1)==0) {
        include_header=0; continue;
      }
    } else if(strncasecmp(cptr, "MID", 3)==0) {
      timetype=DFT_TIME_MIDDLE; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else {
    if(!dftfile[0]) {
      strcpy(dftfile, argv[ai]); continue;
    } else if(!outfile[0]) {
      strcpy(outfile, argv[ai]); continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }
  /* Check parameters */
  if(!dftfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; try %s --help\n", argv[0]);
    return(1);
  }
  /* Make output filename if necessary */
  if(!outfile[0]) {
    strcpy(outfile, dftfile); cptr=strrchr(outfile, '.');
    if(cptr!=NULL && (
        strcasecmp(cptr, ".dft")==0 || strcasecmp(cptr, ".kbq")==0 ||
        strcasecmp(cptr, ".nci")==0 || strcasecmp(cptr, ".dat")==0 ||
        strcasecmp(cptr, ".tac")==0 || strcasecmp(cptr, ".bld")==0 ||
        strcasecmp(cptr, ".ift")==0))
      *cptr=(char)0;
    strcat(outfile, ".csv");
  }
  if(TEST) {
    printf("TEST := %d\n", TEST);
    printf("SILENT := %d\n", SILENT);
    printf("VERBOSE := %d\n", VERBOSE);
    printf("csv_separator := %d\n", csv_separator);
    printf("decimal_separator := %d\n", decimal_separator);
    printf("include_header := %d\n", include_header);
    printf("timetype := %d\n", timetype);
    printf("dftfile := %s\n", dftfile);
    printf("outfile := %s\n", outfile);
  }
  if(TEST>0) VERBOSE=1;

  /* Read data */
  if(VERBOSE) printf("reading file %s\n", dftfile);
  dftInit(&data); iftInit(&ift);
  /* Try to read DFT first */
  if(dftRead(dftfile, &data)) {
    dftEmpty(&data);
    /* Not successful, then try to read as IFT */
    if(iftRead(&ift, dftfile, 1)) {
      /* if not successful either, then give error msg from DFT */
      fprintf(stderr, "Error in reading '%s': %s\n", dftfile, dfterrmsg);
      return 2;
    }
  } else {
    /* DFT could be read, use only frame mid times, if requested */
    if(timetype>=0) data.timetype=timetype;
  }
  if(TEST>19) dftPrint(&data);

  /* Write data */
  if(data.voiNr>0) { // write DFT as CSV
    if(dftWriteAsCSV(&data, outfile, csv_separator, decimal_separator,
                     include_header, tmp))
    {
      fprintf(stderr, "Error in writing '%s': %s\n", outfile, tmp);
      dftEmpty(&data); iftEmpty(&ift); return(11);
    }
  } else { // write IFT as CSV
    if(iftWriteAsCSV(&ift, outfile, csv_separator, decimal_separator, tmp))
    {
      fprintf(stderr, "Error in writing '%s': %s\n", outfile, tmp);
      dftEmpty(&data); iftEmpty(&ift); return(11);
    }
  }
  if(SILENT==0) printf("CSV data written in %s.\n", outfile);

  /* Free memory */
  dftEmpty(&data); iftEmpty(&ift);

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/*
 *  void print_usage()
 */
void print_usage()
{
  int i=0;
  static char *info[] = {
  " ",
  PROGRAM,
  " ",
  "Converts PET tracer time-activity curve data in DFT format (1), or",
  "IFT data, to CSV (comma-separated values) data, which can be easily imported",
  "to other software, including Excel.",
  " ",
  "Usage:  dft2csv [Options] <DFT file> [CSV file]",
  " ",
  "Options:",
  " -separator=<<comma>|<semi-colon>|<tab>>",
  "     By default comma is used as the separator, but semicolon or tabulator",
  "     may be needed to import data to Excel or other software, depending",
  "     on the localization settings. Use tabulator to import data to Origin.",
  "     This setting affects also decimal points/commas.",
  " -header=<Yes|no>",
  "     Descriptive title lines (if header information is available) can be",
  "     included (default), or not included.",
  " -mid",
  "     Frame mid time is written in CSV file instead of frame start and end.",
  " -h or --help",
  "     Print this message and exit.",
  " --version or --build",
  "     Print software build information and exit.",
  " --silent",
  "     Program works silently, printing only error and warning messages.",
  " --verbose",
  "     Program prints more information about what it is doing.",
  " ",
  "Examples:",
  "1. Convert one DFT file and enter the name for CSV file",
  "     dft2svg ae426.dft ae426.csv",
  "2. Convert all DFT files in current working directory in MS Windows",
  "   command shell, using decimal commas and semi-colon as separator",
  "     for %g in (*.dft) do dft2svg -separator=semi-colon %g",
  " ",
  "References:",
  "1. http://www.turkupetcentre.net/analysis/doc/format_dft.html",
  "2. http://en.wikipedia.org/wiki/Comma-separated_values",
  " ",
  "See also: csv2dft, dft2res, dft2html, dft2pmod, dftunit",
  " ",
  "Keywords: DFT, IFT, CSV, Excel, file format conversion, tools",
  " ",
  "This program comes with ABSOLUTELY NO WARRANTY. This is free software, and",
  "you are welcome to redistribute it under GNU General Public License.",
  " ",
  0};
  while(info[i]!=0) printf("  %s\n", info[i++]);
}
/*****************************************************************************/

/*****************************************************************************/
/**
 * Prints build information to stdout.
 */
void print_build() {

  printf("\n %s\n", PROGRAM);
  printf("\n Build %s %s\n\n",__DATE__,__TIME__);
#if defined(__FILE__) && defined(__TIMESTAMP__)
  printf(" File %s last edited on %s\n", __FILE__, __TIMESTAMP__);
#endif
  printf(" _______________________________________________\n");

  libtpccurveio_print_build(stdout);
  libtpcmisc_print_build(stdout);
  printf(" _______________________________________________\n");

#if defined(__STDC_VERSION__)
  printf(" Version of C: %ld\n", __STDC_VERSION__);
#endif
#if defined(__STDC_ISO_10646__)
  printf(" Compiler ISO/IEC 10646 specification: %ld\n", __STDC_ISO_10646__);
#endif
#if defined(__GNUC__) && defined(__VERSION__)
  printf(" GNU C version: %s\n", __VERSION__);
#endif
#ifdef _OPENMP
  printf(" OpenMP version: %d\n", _OPENMP);
#endif
#if defined(__x86_64__) || defined(__LP64__) || defined(__ppc64__)
  printf(" Architecture: 64-bit\n");
#else
  printf(" Architecture: 32-bit\n");
#endif
#if defined(__linux__)
  printf(" OS: linux\n");
#elif defined(__unix__)
  printf(" OS: unix\n");
#elif defined(__WIN32__) || defined(__WINNT__)
  printf(" OS: Windows\n");
#endif
#if defined(__sun__) || defined(__sparc__)
  printf(" Platform: sun\n");
#elif defined(__i486__) || defined(__i386__) || defined(_X86_)
  printf(" Platform: x86\n");
#endif

  printf(" _______________________________________________\n\n");
}
/*****************************************************************************/

/*****************************************************************************/

