/******************************************************************************
  Copyright (c) 2013 by Turku PET Centre

  File:      par.h

  Version:
  2013-07-12 Vesa Oikonen
  2013-09-02 VO


******************************************************************************/
#ifndef _PAR_H
#define _PAR_H
/*****************************************************************************/
/* Max nr of parameters */
#ifndef MAX_PETPARAMS
#define MAX_PETPARAMS 100
#endif
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "curveio.h"
/*****************************************************************************/
#include "tacid.h" // for now
#include "libtpcparv.h"
/*****************************************************************************/
/* Backup file extension */
#ifndef BACKUP_EXTENSION
#define BACKUP_EXTENSION ".bak"
#endif 
/* Max length of parameter names and units */
#ifndef MAX_PETPARNAME_LEN
#define MAX_PETPARNAME_LEN 15
#endif
/** Definitions for weighting */
#define WEIGHTING_UNKNOWN 0
#define WEIGHTING_ON      1
#define WEIGHTING_OFF     2
/** Definitions for parameter file format */
#define PETPAR_UNKNOWN 0
#define PETPAR_CSV_INT 1
#define PETPAR_CSV_UK  2
#define PETPAR_RES     3
#define PETPAR_FIT     4
/*****************************************************************************/

/*****************************************************************************/
/** Error codes */
enum {PAR_OK, PAR_ERROR, PAR_CANNOTOPEN, PAR_CANNOTWRITE, PAR_INVALIDFORMAT,
    PAR_TOOBIG, PAR_OUTOFMEMORY, PAR_NOTABLE
    };
/*****************************************************************************/

/*****************************************************************************/
/*
 *  Data structure definitions
 */
/** Struct for one (regional) set of parameters */
typedef struct {
  /** TAC ID */
  TACID     id;
  /** Name of the parameter set (region name) */
  char      name[MAX_REGIONNAME_LEN+1];
  /** Number (type) of mathematical function */
  int       function;
  /** Number (type) of model */
  int       model;
  /** Fit start time (unit specified in top struct) */
  double    start;
  /** Fit end time (unit specified in top struct) */
  double    end;
  /** The number of fitted parameters (excluding fixed parameters) */
  int       parNr;
  /** Number of fitted data points */
  int       dataNr;
  /** Array of parameters */
  double    p[MAX_FITPARAMS];
  /** Parameter SD's   */
  double    sd[MAX_RESPARAMS];
  /** Lower 95% confidence intervals */
  double    cl1[MAX_RESPARAMS];
  /** Upper 95% confidence intervals */
  double    cl2[MAX_RESPARAMS];
  /** (weighted) sum-of-squares */
  double    wss;
  /** Temporary switches for outside procedures */
  char      sw, sw2, sw3;
} PETPARVOI;

/** Main PET model parameters struct */
typedef struct {
  /** File format: PETPAR_CSV_INT, PETPAR_CSV_UK, ..., PETPAR_UNKNOWN */
  int       fileformat;
  /** Number of regions */
  int       voiNr;
  /** Name of study (study number) */
  char      studynr[MAX_STUDYNR_LEN+1];

  /** Parameter estimation date and time */
  time_t    time;
  /** Analysis program name */
  char      program[1024];

  /* Names of original datafiles */
  /** Name of original tissue datafile */
  char      datafile[FILENAME_MAX];
  /** Name of original ref datafile */
  char      reffile[FILENAME_MAX];
  /** Name of original plasmafile */
  char      plasmafile[FILENAME_MAX];
  /** Name of second original plasmafile */
  char      plasmafile2[FILENAME_MAX];
  /** Name of original bloodfile */
  char      bloodfile[FILENAME_MAX];
  /** Name of reference region */
  char      refname[MAX_REGIONNAME_LEN+1];
  /** Free text field describing fit method */
  char      fitmethod[FILENAME_MAX];
  /** Unit of original data */
  int       dataunit;
  /** Time unit of original data and fit start and end times */
  int       timeunit;

  /** Were data weighted in parameter estimation?
      WEIGHTING_UNKNOWN, WEIGHTING_ON, WEIGHTING_OFF */
  int       weighting;
  /** Tissue density (g/ml) */
  double    density;
  /** Lumped Constant (unitless) */
  double    lc;
  /** Beta */
  double    beta;
  /** Plasma concentration of native substrate, e.g. glucose of FFA */
  double    concentration;
  /** Vb percentage */
  double    Vb;
  /** fA percentage (arterial volume of Vb) */
  double    fA;
  /** Extraction fraction */
  double    E;
  
  /** Nr of parameter names and units; set also regional parNr */
  int       parNr;
  /** List of parameter names */
  char      parname[MAX_RESPARAMS][MAX_RESPARNAME_LEN+1];
  /** List of parameter units */
  int       parunit[MAX_RESPARAMS];

  /** Pointers to regional curves */
  PETPARVOI *voi;
  
  /** Internal variables: Number of allocated curves (VOIs) */
  int      _voidataNr;
} PETPAR;
/*****************************************************************************/

/*****************************************************************************/
/* par.c */
extern void parEmpty(PETPAR *par);
extern void parInit(PETPAR *par);
extern int parAllocate(PETPAR *par, int voiNr);
extern void parPrint(PETPAR *par);
extern int doubleMatch(double g1, double g2, double e);
extern int parMatchParameters(
  PETPAR *p1, PETPAR *p2, int par_index, int test_sd,
  double limit_abs, double limit_rel, int verbose
);
extern int parMatchExtraParameters(PETPAR *p1, PETPAR *p2);
extern int parMatchParameternames(PETPAR *p1, PETPAR *p2);
extern int parMatchRegionnames(PETPAR *p1, PETPAR *p2);
extern int parMatchHeader(PETPAR *p1, PETPAR *p2);
extern int parMatch(PETPAR *par1, PETPAR *par2, int verbose);
extern int parIsWSS(PETPAR *par);
extern int parIsFitrange(PETPAR *par);
extern int parIsFunction(PETPAR *par);
extern int parIsModel(PETPAR *par);
extern int parHighestParnr(PETPAR *par);
extern int parHighestDatanr(PETPAR *par);
extern int parSDWithPar(PETPAR *par, int pi);
extern int parCLWithPar(PETPAR *par, int pi);
extern int parNrofUnits(PETPAR *par);
/*****************************************************************************/

/*****************************************************************************/
/* pario.c */
extern void csvPrint(CSV *csv);
extern int csvWriteText(FILE *fp, const char *txt, int tointl);
extern int parWrite(PETPAR *par, char *filename, int verbose);
extern int parRead(PETPAR *par, char *filename, int verbose);
extern int parWriteCSV(PETPAR *par, char *filename, int verbose);
extern int parReadCSV(PETPAR *par, char *filename, int verbose);
/*****************************************************************************/

/*****************************************************************************/
/* parconv.c */
extern int parFit2Par(FIT *fit, PETPAR *par, int verbose);
extern int parPar2Fit(PETPAR *par, FIT *fit, int verbose);
extern int parPar2Res(PETPAR *par, RES *res, int verbose);
extern int parRes2Par(RES *res, PETPAR *par, int verbose);
/*****************************************************************************/

/*****************************************************************************/
#endif /* _PAR_H */
