/*******************************************************************************
  Copyright (c) 2013 by Turku PET Centre
   
  File: libtpcpar.c
   
  Purpose: methods for testing functions in the library libtpcpar,
  and for printing out library information, such as Readme,
  History and build information. 

  Version history:
  2013-07-12 Vesa Oikonen
       First created.

*******************************************************************************/

/******************************************************************************/
/** Library name */
#define LIB_NAME "libtpcpar"
/******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
/******************************************************************************/
#include "curveio.h"
#include "libtpcmisc.h"
/******************************************************************************/
#include "include/par.h"
#include "include/libtpcparv.h"
/******************************************************************************/
/** Serial numbers for function names. */
#define ALLTESTS 0
#define PAR1     1
#define PAR2     2
#define PARIO1  11
#define PARIO2  12
#define PARIO3  13
/******************************************************************************/

/******************************************************************************/
// Test function declarations
int test_par1();
int test_par2();
int test_pario1();
int test_pario2();
int test_pario3();
// Helping function declarations
int create_par(PETPAR *par);
int create_par_for_fit(PETPAR *par);
int create_par_for_res(PETPAR *par);
/******************************************************************************/

/******************************************************************************/
/** 
    Function for printing the usage information of the test program.
*/
void print_usage();

/******************************************************************************/
// Verbose mode switch:
int VERBOSE = 0;
/******************************************************************************/

/******************************************************************************/
int main (int argc, char *argv[ ])
{
  int c;
  int errflg=0, functionflag=-1, exit_code=0;
  extern char *optarg;
  extern int optind, optopt;
    
  while ((c = getopt(argc, argv, "vVbBrRhHf:F:")) != -1) {
    switch (c) {

    case 'V':
    case 'v':
      // Set verbose flag on:
      VERBOSE = 1;
      break;
    case 'B':
    case 'b':
      // Print out the build information:
      libtpcpar_print_build(stdout);
      return(0);
    case 'r':
      // Print out the Readme message:
      libtpcpar_print_readme(stdout);
      return(0);
    case 'R':
      // Print out the Readme message with Doxygen tags:
      libtpcpar_print_dreadme(stdout);
      return(0);
    case 'h':
      // Print out the History message:
      libtpcpar_print_history(stdout);
      return(0);
    case 'H':
      // Print out the History message with Doxygen tags:
      libtpcpar_print_dhistory(stdout);
      return(0);
    case 'f':
    case 'F':
      // Set function serial number to be tested:
      functionflag = atoi(optarg);
      break;
    case ':': // -f or -z without argument
      fprintf(stderr, "Option -%c requires an argument\n", optopt);
      errflg++;
      break;
    case '?':
      fprintf(stderr, "Unrecognised option: -%c\n", optopt);
      errflg++;
    }
  }// End parsing command line options...

  if(errflg) {print_usage(); return(2);}

  if(functionflag<0) {print_usage(); return(0);}

  libtpcpar_print_build(stdout);

  // Choose function(s) to test:
  switch(functionflag){
    case ALLTESTS:
      exit_code = test_par1();
      if(exit_code) exit(exit_code);
      exit_code = test_par2();
      if(exit_code) exit(exit_code);
      exit_code = test_pario1();
      if(exit_code) exit(exit_code);
      exit_code = test_pario2();
      if(exit_code) exit(exit_code);
      exit_code = test_pario3();
      if(exit_code) exit(exit_code);
      break;
    case PAR1:
      exit_code = test_par1();
      break;
    case PAR2:
      exit_code = test_par2();
      break;
    case PARIO1:
      exit_code = test_pario1();
      break;
    case PARIO2:
      exit_code = test_pario2();
      break;
    case PARIO3:
      exit_code = test_pario3();
      break;
    default:
      break;
  }
        
  exit(exit_code);
}
/******************************************************************************/

/******************************************************************************/
int test_par1()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret;

  /* Initiate PETPAR struct */
  PETPAR p;
  parInit(&p);
  /* Allocate memory for it */
  ret=parAllocate(&p, 1); if(ret!=PAR_OK) error_code=1;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  /* Free the memory */
  parEmpty(&p);
  /* Allocate memory again for it */
  ret=parAllocate(&p, 6); if(ret!=PAR_OK) error_code=2;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  /* Set VoiNr, too */
  p.voiNr=6;
  /* Print it */
  parPrint(&p);
  /* Free the memory */
  parEmpty(&p);
  /* Try to print it */
  parPrint(&p);

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_par2()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret;

  /* Initiate PETPAR struct */
  PETPAR p;
  parInit(&p);
  /* Fill it with test contents */
  ret=create_par(&p); if(ret!=PAR_OK) error_code=1;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  /* Print it */
  parPrint(&p);
  /* Free the memory */
  parEmpty(&p);

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_pario1()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret;
  char filename[FILENAME_MAX];
  char testdir[]="test";


  /* Create subdirectory for test data */
  if(access(testdir, 0) == -1) {
    if(VERBOSE) fprintf(stdout, "Creating subdirectory '%s'\n", testdir);
#ifdef WIN32
    ret=mkdir(testdir);
#else
    ret=mkdir(testdir, 00775);
#endif
    if(ret!=0) {
      fprintf(stderr, "  Error: cannot create test subdirectory.\n");
      fflush(stderr); error_code=-1;
      printf("\n %s FAILED with error code: %i\n", __func__, error_code);
      return(error_code);
    }
  }

  /* Initiate PETPAR struct */
  PETPAR p, p2;
  parInit(&p);
  //VERBOSE=30;

  /* Fill it with test contents */
  ret=create_par(&p); if(ret!=PAR_OK) error_code=1;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Save it in CSV UK format */
  p.fileformat=PETPAR_CSV_UK;
  sprintf(filename, "%s\\testdata1.csv", testdir);
  ret=parWrite(&p, filename, VERBOSE); if(ret!=PAR_OK) error_code=11;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Read the file */
  parInit(&p2);
  ret=parRead(&p2, filename, VERBOSE); if(ret!=PAR_OK) error_code=12;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Check that data has not changed */
  ret=parMatch(&p, &p2, VERBOSE); if(ret!=0) error_code=13;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Free the memory */
  parEmpty(&p2);

  /* Save data in CSV INTL format */
  p.fileformat=PETPAR_CSV_INT;
  sprintf(filename, "%s\\testdata2.csv", testdir);
  ret=parWrite(&p, filename, VERBOSE); if(ret!=PAR_OK) error_code=21;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Read the file */
  ret=parRead(&p2, filename, VERBOSE); if(ret!=PAR_OK) error_code=22;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Check that data has not changed */
  ret=parMatch(&p, &p2, VERBOSE); if(ret!=0) error_code=23;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Free the memory */
  parEmpty(&p); parEmpty(&p2);

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_pario2()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret;
  char filename[FILENAME_MAX];
  char testdir[]="test";


  /* Create subdirectory for test data */
  if(access(testdir, 0) == -1) {
    if(VERBOSE) fprintf(stdout, "Creating subdirectory '%s'\n", testdir);
#ifdef WIN32
    ret=mkdir(testdir);
#else
    ret=mkdir(testdir, 00775);
#endif
    if(ret!=0) {
      fprintf(stderr, "  Error: cannot create test subdirectory.\n");
      fflush(stderr); error_code=-1;
      printf("\n %s FAILED with error code: %i\n", __func__, error_code);
      return(error_code);
    }
  }

  /* Initiate PETPAR struct */
  PETPAR p, p2;
  parInit(&p);
  //VERBOSE=30;

  /* Fill it with test contents */
  ret=create_par_for_fit(&p); if(ret!=PAR_OK) error_code=1;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Save it in FIT format */
  p.fileformat=PETPAR_FIT;
  sprintf(filename, "%s\\testdata2.fit", testdir);
  ret=parWrite(&p, filename, VERBOSE); if(ret!=PAR_OK) error_code=11;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Read the file */
  parInit(&p2);
  ret=parRead(&p2, filename, VERBOSE); if(ret!=PAR_OK) error_code=12;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Check that data has not changed */
  ret=parMatch(&p, &p2, VERBOSE); if(ret!=0) error_code=13;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Free the memory */
  parEmpty(&p); parEmpty(&p2);

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_pario3()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret;
  char filename[FILENAME_MAX];
  char testdir[]="test";


  /* Create subdirectory for test data */
  if(access(testdir, 0) == -1) {
    if(VERBOSE) fprintf(stdout, "Creating subdirectory '%s'\n", testdir);
#ifdef WIN32
    ret=mkdir(testdir);
#else
    ret=mkdir(testdir, 00775);
#endif
    if(ret!=0) {
      fprintf(stderr, "  Error: cannot create test subdirectory.\n");
      fflush(stderr); error_code=-1;
      printf("\n %s FAILED with error code: %i\n", __func__, error_code);
      return(error_code);
    }
  }

  /* Initiate PETPAR struct */
  PETPAR p, p2;
  parInit(&p);
  VERBOSE=30;

  /* Fill it with test contents */
  ret=create_par_for_res(&p); if(ret!=PAR_OK) error_code=1;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Save it in RES format */
  p.fileformat=PETPAR_RES;
  sprintf(filename, "%s\\testdata3.res", testdir);
  ret=parWrite(&p, filename, VERBOSE); if(ret!=PAR_OK) error_code=11;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Read the file */
  parInit(&p2);
  ret=parRead(&p2, filename, VERBOSE); if(ret!=PAR_OK) error_code=12;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Check that data has not changed */
  ret=parMatch(&p, &p2, VERBOSE); if(ret!=0) error_code=13;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Save it in RES HTML format */
  p.fileformat=PETPAR_RES;
  sprintf(filename, "%s\\testdata3.htm", testdir);
  ret=parWrite(&p, filename, VERBOSE); if(ret!=PAR_OK) error_code=11;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  /* Its ok if we did not get any errors */


  /* Free the memory */
  parEmpty(&p); parEmpty(&p2);

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_pario4()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;

  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }


  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
/** Create test PETPAR data */
int create_par(
  PETPAR *par
) {
  int ret, pi, ri;
  
  /* Allocate memory */
  ret=parAllocate(par, 3); if(ret!=PAR_OK) return ret;
  par->voiNr=3;
  /* Set header fields */
  strcpy(par->studynr, "test0001");
  par->time=time(NULL);
  strcpy(par->program, "libtpcpar");
  strcpy(par->datafile, "C:\\My Documents\\PETData\\imaginary.dat");
  par->weighting=WEIGHTING_OFF;
  par->dataunit=CUNIT_KBQ_PER_ML;
  par->timeunit=TUNIT_MIN;
  /* Set parameter names */
  par->parNr=5;
  pi=0; strcpy(par->parname[pi], "K1"); par->parunit[pi]=CUNIT_ML_PER_ML_PER_MIN;
  pi++; strcpy(par->parname[pi], "K1/k2"); par->parunit[pi]=CUNIT_UNITLESS;
  pi++; strcpy(par->parname[pi], "k3"); par->parunit[pi]=CUNIT_PER_MIN;
  pi++; strcpy(par->parname[pi], "k4"); par->parunit[pi]=CUNIT_PER_MIN;
  pi++; strcpy(par->parname[pi], "Vb"); par->parunit[pi]=CUNIT_PERCENTAGE;
  //pi++; strcpy(par->parname[pi], "WSS"); par->parunit[pi]=CUNIT_UNITLESS;
  /* Set region names */
  for(ri=0; ri<par->voiNr; ri++) sprintf(par->voi[ri].name, "roi%d", 1+ri);
  /* Set regional fit options */
  for(ri=0; ri<par->voiNr; ri++) {
    par->voi[ri].parNr=4;
    par->voi[ri].dataNr=23;
    par->voi[ri].start=0.0;
    par->voi[ri].end=45.0;
    if(ri==0) par->voi[ri].wss=8.7654E+002;
    else par->voi[ri].wss=1.1*par->voi[ri-1].wss;
  }
  /* Set parameter values */
  ri=0; 
  pi=0; par->voi[ri].p[pi]=0.3;
  pi++; par->voi[ri].p[pi]=0.5;
  pi++; par->voi[ri].p[pi]=0.1;
  pi++; par->voi[ri].p[pi]=0.005;
  pi++; par->voi[ri].p[pi]=5.0;
  //pi++; par->voi[ri].p[pi]=9.8765E+002;
  for(ri=1; ri<par->voiNr; ri++)
    for(pi=0; pi<par->parNr; pi++)
      par->voi[ri].p[pi]=1.1*par->voi[ri-1].p[pi];
  /* Set SDs and CLs for certain parameters */
  for(pi=0; pi<par->parNr-1; pi++) {
    for(ri=0; ri<par->voiNr; ri++) {
      if(pi>0) par->voi[ri].sd[pi]=0.2*par->voi[ri].p[pi];
      if(pi==0) {
        par->voi[ri].cl1[pi]=0.9*par->voi[ri].p[pi];
        par->voi[ri].cl2[pi]=1.3*par->voi[ri].p[pi];
      }
      if(pi==2 || pi==3) {
        par->voi[ri].cl1[pi]=par->voi[ri].p[pi]-par->voi[ri].sd[pi];
        par->voi[ri].cl2[pi]=par->voi[ri].p[pi]+par->voi[ri].sd[pi];
      }
    }
  }
  
  return PAR_OK;
}
/******************************************************************************/

/******************************************************************************/
/** Create PETPAR data for testing FIT struct */
int create_par_for_fit(
  PETPAR *par
) {
  int ret, pi, ri;
  
  /* Allocate memory */
  ret=parAllocate(par, 3); if(ret!=PAR_OK) return ret;
  par->voiNr=3;
  /* Set header fields */
  par->time=time(NULL);
  strcpy(par->program, "libtpcpar");
  strcpy(par->datafile, "C:\\PETData\\imaginary.dat");
  par->weighting=WEIGHTING_UNKNOWN;
  par->dataunit=CUNIT_KBQ_PER_ML;
  par->timeunit=TUNIT_MIN;
  /* Set parameter names */
  par->parNr=4;
  for(pi=0; pi<par->parNr; pi++) sprintf(par->parname[pi], "p%d", pi+1);
  /* Set region names */
  for(ri=0; ri<par->voiNr; ri++) sprintf(par->voi[ri].name, "roi%d", 1+ri);
  /* Set regional fit options */
  for(ri=0; ri<par->voiNr; ri++) {
    par->voi[ri].dataNr=23;
    par->voi[ri].start=0.0;
    par->voi[ri].end=45.0;
    if(ri==0) par->voi[ri].wss=8.7654E+002;
    else par->voi[ri].wss=1.1*par->voi[ri-1].wss;
    /* Function id must be set */
    par->voi[ri].function=302;
  }
  /* Set parameter values */
  ri=0; 
  pi=0; par->voi[ri].p[pi]=0.3;
  pi++; par->voi[ri].p[pi]=0.5;
  pi++; par->voi[ri].p[pi]=0.1;
  pi++; par->voi[ri].p[pi]=0.005;
  for(ri=1; ri<par->voiNr; ri++)
    for(pi=0; pi<par->parNr; pi++)
      par->voi[ri].p[pi]=1.1*par->voi[ri-1].p[pi];
  
  return PAR_OK;
}
/******************************************************************************/

/******************************************************************************/
/** Create PETPAR data for testing RES struct */
int create_par_for_res(
  PETPAR *par
) {
  int ret, pi, ri;
  
  /* Allocate memory */
  ret=parAllocate(par, 3); if(ret!=PAR_OK) return ret;
  par->voiNr=3;
  par->parNr=5;
  /* Set header fields */
  strcpy(par->program, "libtpcpar");
  par->time=time(NULL);
  strcpy(par->studynr, "test0003");
  strcpy(par->datafile, "C:\\PETData\\imaginary.dat");
  strcpy(par->plasmafile, "C:\\PETData\\imaginary_ap.dat");
  strcpy(par->bloodfile, "C:\\PETData\\imaginary_ab.dat");
  par->weighting=WEIGHTING_ON;
  par->lc=1.0;
  par->Vb=5.0;
  /* Set parameter names */
  par->parNr=5;
  pi=0; strcpy(par->parname[pi], "K1"); par->parunit[pi]=CUNIT_ML_PER_ML_PER_MIN;
  pi++; strcpy(par->parname[pi], "K1/k2"); par->parunit[pi]=CUNIT_UNITLESS;
  pi++; strcpy(par->parname[pi], "k3"); par->parunit[pi]=CUNIT_PER_MIN;
  pi++; strcpy(par->parname[pi], "k4"); par->parunit[pi]=CUNIT_PER_MIN;
  pi++; strcpy(par->parname[pi], "Vb"); par->parunit[pi]=CUNIT_PERCENTAGE;
  /* Set region names */
  for(ri=0; ri<par->voiNr; ri++) sprintf(par->voi[ri].name, "roi%d", 1+ri);
  /* Set regional fit options */
  for(ri=0; ri<par->voiNr; ri++) {
    par->voi[ri].parNr=4;
    par->voi[ri].dataNr=23;
    par->voi[ri].start=0.0;
    par->voi[ri].end=45.0;
    if(ri==0) par->voi[ri].wss=8.7654E+002;
    else par->voi[ri].wss=1.1*par->voi[ri-1].wss;
  }
  /* Set parameter values */
  ri=0; 
  pi=0; par->voi[ri].p[pi]=0.3;
  pi++; par->voi[ri].p[pi]=0.5;
  pi++; par->voi[ri].p[pi]=0.1;
  pi++; par->voi[ri].p[pi]=0.005;
  pi++; par->voi[ri].p[pi]=5.0;
  for(ri=1; ri<par->voiNr; ri++)
    for(pi=0; pi<par->parNr; pi++)
      par->voi[ri].p[pi]=1.1*par->voi[ri-1].p[pi];
  /* Set SDs and CLs for certain parameters */
  for(pi=0; pi<par->parNr-1; pi++) {
    for(ri=0; ri<par->voiNr; ri++) {
      if(pi>0) par->voi[ri].sd[pi]=0.2*par->voi[ri].p[pi];
      if(pi==0) {
        par->voi[ri].cl1[pi]=0.9*par->voi[ri].p[pi];
        par->voi[ri].cl2[pi]=1.3*par->voi[ri].p[pi];
      }
      if(pi==2 || pi==3) {
        par->voi[ri].cl1[pi]=par->voi[ri].p[pi]-par->voi[ri].sd[pi];
        par->voi[ri].cl2[pi]=par->voi[ri].p[pi]+par->voi[ri].sd[pi];
      }
    }
  }
  
  return PAR_OK;
}
/******************************************************************************/

/******************************************************************************/
void print_usage(){

  libtpcpar_print_build(stdout);
  puts("\n"
  "  Methods for testing functions in the library "LIB_NAME"\n"
  "  and for printing out associated information, such as Readme,\n"
  "  History and build information. \n"
  "\n"
  "  Usage: "LIB_NAME" [-Options]\n"
  "\n"
  "  Options:\n"
  "  -h | H \n"
  "      Print out the "LIB_NAME" History message. Include Doxygen style tags\n"
  "      by using option '-H'.\n"
  "  -r | R \n"
  "      Print out the "LIB_NAME" Readme message. Include Doxygen style tags\n"
  "      by using option '-R'.\n"
  "  -b\n"
  "      Print out the "LIB_NAME" build information.\n"
  "  -f <function serial number>\n"
  "      Runs test procedures for the functions corresponding to given\n"
  "      'function serial number'. Serial numbers are defined in file\n"
  "      "LIB_NAME".c; enter 0 to execute them all.\n"
  "  -v\n"
  "      Run in verbose mode.\n"
  "\n"
  "  E.g.: "LIB_NAME" -h\n"
  "        "LIB_NAME" -v -f 1 \n"
  "\n"
  );
  fflush(stdout);
}
/******************************************************************************/

/******************************************************************************/
