/******************************************************************************
  Copyright (c) 2013 by Turku PET Centre

  File:        par.c
  Description: Functions for processing PET model parameters.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Version history:
  2013-07-12 Vesa Oikonen
       First created.
     


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "curveio.h"
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/par.h"
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for PETPAR. All data is cleared. */
void parEmpty(
  /** Pointer to PETPAR struct data */
  PETPAR *par
) {
  int i;
  if(par==NULL) return;

  par->studynr[0]=(char)0;
  par->program[0]=(char)0;

  par->datafile[0]=(char)0;
  par->reffile[0]=(char)0;
  par->plasmafile[0]=(char)0;
  par->plasmafile2[0]=(char)0;
  par->bloodfile[0]=(char)0;
  par->refname[0]=(char)0;
  par->fitmethod[0]=(char)0;
  par->dataunit=CUNIT_UNKNOWN;
  par->timeunit=TUNIT_UNKNOWN;
  
  par->weighting=WEIGHTING_UNKNOWN;
  par->density= par->lc= par->concentration= par->beta= nan("");
  par->Vb= par->fA= par->E= nan("");

  par->parNr=0;
  for(i=0; i<MAX_PETPARAMS; i++) {
    strcpy(par->parname[i], "");
    par->parunit[i]=CUNIT_UNKNOWN;
  }
  
  if(par->_voidataNr>0) free((char*)(par->voi));
  par->_voidataNr=0;
  par->fileformat=PETPAR_UNKNOWN;
  par->voiNr=0;

  return;
}
/*****************************************************************************/

/*****************************************************************************/
/** Initiate PARPET structure. This should be called once before first use. */
void parInit(
  /** Pointer to allocated PETPAR struct */
  PETPAR *par
) {
  if(par==NULL) return;
  par->_voidataNr=0; par->voiNr=0;
  parEmpty(par);
}
/*****************************************************************************/

/*****************************************************************************/
/** Allocate memory for parameters.
\return Returns 0 (PAR_OK) if successful, otherwise error code <>0.
*/
int parAllocate(
  /** Pointer to initiated and possibly used parameter struct;
   *  old contents are deleted;
   *  note that par->voiNr is still 0 after allocation. */
  PETPAR *par,
  /** Nr of parameter sets (regions) */
  int voiNr
) {
  int ri, pi;

  /* Check that there is something to do */
  if(par==NULL || voiNr<1) return(PAR_ERROR);

  /* Clear previous data, but only if necessary */
  if(par->_voidataNr>0) parEmpty(par);

  /* Allocate memory for parameter sets (regions) */
  par->voi=(PETPARVOI*)calloc(voiNr, sizeof(PETPARVOI));
  if(par->voi==NULL) return(PAR_OUTOFMEMORY);
  par->_voidataNr=voiNr;

  /* Set parameters, SDs and CLs, etc to NaN */
  for(ri=0; ri<par->_voidataNr; ri++) {
    strcpy(par->voi[ri].name, "");
    par->voi[ri].function=0;
    par->voi[ri].model=0;
    par->voi[ri].start=par->voi[ri].end=nan("");
    par->voi[ri].parNr=0;
    par->voi[ri].dataNr=0;
    for(pi=0; pi<MAX_PETPARAMS; pi++)
      par->voi[ri].sd[pi]=par->voi[ri].cl1[pi]=par->voi[ri].cl2[pi]=nan("");
    par->voi[ri].wss=nan("");
    par->voi[ri].sw=par->voi[ri].sw2=par->voi[ri].sw3=0;
  }

  return(PAR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Print to stdout the contents of PETPAR data structure in human readable
 *  format; mainly for testing purposes. */
void parPrint(
  /** Pointer to parameter data */
  PETPAR *par
) {
  int pi, ri;
  struct tm *st;
  char tmp[256];

  printf("\n");
  if(par==NULL) printf("par := NULL\n");
  
  printf("fileformat := %d\n", par->fileformat);
  printf("voiNr := %d\n", par->voiNr);
  printf("_voidataNr := %d\n", par->_voidataNr);
  if(strlen(par->studynr)>0) printf("studynr := %s\n", par->studynr);
  st=localtime(&par->time); if(st!=NULL) {
    strftime(tmp, 256, "%Y-%m-%d %H:%M:%S", st);
    printf("Date := %s\n", tmp);
  }
  if(strlen(par->program)>0) printf("program := %s\n", par->program);
  if(strlen(par->datafile)>0) printf("datafile := %s\n", par->datafile);
  if(strlen(par->reffile)>0) printf("reffile := %s\n", par->reffile);
  if(strlen(par->plasmafile)>0) printf("plasmafile := %s\n", par->plasmafile);
  if(strlen(par->plasmafile2)>0) printf("plasmafile2 := %s\n", par->plasmafile2);
  if(strlen(par->bloodfile)>0) printf("bloodfile := %s\n", par->bloodfile);
  if(strlen(par->refname)>0) printf("refname := %s\n", par->refname);
  if(strlen(par->fitmethod)>0) printf("fitmethod := %s\n", par->fitmethod);
  printf("dataunit := %s\n", petCunit(par->dataunit));
  printf("timeunit := %s\n", petTunit(par->timeunit));
  printf("weighting := %d\n", par->weighting);

  if(!isnan(par->density)) printf("density := %g\n", par->density);
  if(!isnan(par->lc)) printf("lc := %g\n", par->lc);
  if(!isnan(par->beta)) printf("beta := %g\n", par->beta);
  if(!isnan(par->concentration))
    printf("concentration := %g\n", par->concentration);
  if(!isnan(par->Vb)) printf("Vb := %g\n", par->Vb);
  if(!isnan(par->fA)) printf("fA := %g\n", par->fA);
  if(!isnan(par->E)) printf("E := %g\n", par->E);
  
  printf("parNr := %d\n", par->parNr);
  for(ri=0; ri<par->voiNr; ri++) {
    printf("\nParameter-set[%d]\n", 1+ri);  
    printf("name := %s\n", par->voi[ri].name);
    if(par->voi[ri].function>0)
      printf("function := %d\n", par->voi[ri].function);
    if(!isnan(par->voi[ri].start))
      printf("start := %g\n", par->voi[ri].start);
    if(!isnan(par->voi[ri].end))
      printf("end := %g\n", par->voi[ri].end);
    printf("parNr := %d\n", par->voi[ri].parNr);
    printf("dataNr := %d\n", par->voi[ri].dataNr);
    if(!isnan(par->voi[ri].wss)) printf("wss := %g\n", par->voi[ri].wss);
    for(pi=0; pi<par->parNr; pi++) {
      if(pi<par->parNr) printf("%s := %g %s\n",
        par->parname[pi], par->voi[ri].p[pi], petCunit(par->parunit[pi]));
      else printf("parameter[%d] := %g\n", 1+pi, par->voi[ri].p[pi]);
      if(!isnan(par->voi[ri].sd[pi]))
        printf("SD[%d] := %g\n", 1+pi, par->voi[ri].sd[pi]);
      if(!isnan(par->voi[ri].cl1[pi]))
        printf("CL1[%d] := %g\n", 1+pi, par->voi[ri].cl1[pi]);
      if(!isnan(par->voi[ri].cl2[pi]))
        printf("CL2[%d] := %g\n", 1+pi, par->voi[ri].cl2[pi]);
    }
  }
  printf("\n");
}
/*****************************************************************************/

/*****************************************************************************/
/** Verify that two double values are similar inside given limit.
 *  Doubles may be NaN: if both are NaN, then they are considered matching.
\return Returns 0 in case of match, and <>0 if not matching.
 */
int doubleMatch(
  /** Double 1 */
  double g1,
  /** Double 2 */
  double g2,
  /** Acceptable absolute difference */
  double e
) {
  double d;
  if(isnan(g1) && isnan(g2)) return(0);
  d=g1-g2; if(isnan(d)) return(-1);
  if(fabs(d)>e) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether PETPAR header field values are the same.
    Fields that are not checked: program, time, fileformat, voiNr, parNr.
\return Returns 0 in case of match, and <>0 if not matching.
 */
int parMatchHeader(
  /** Pointer to the parameter data that is tested */
  PETPAR *p1,
  /** Pointer to the parameter data that is tested */
  PETPAR *p2
) {
  if(p1==NULL || p2==NULL) return(1);
  if(strcasecmp(p1->studynr,     p2->studynr)!=0) return(11);
  if(strcasecmp(p1->datafile,    p2->datafile)!=0) return(12);
  if(strcasecmp(p1->reffile,     p2->reffile)!=0) return(13);
  if(strcasecmp(p1->plasmafile,  p2->plasmafile)!=0) return(14);
  if(strcasecmp(p1->plasmafile2, p2->plasmafile2)!=0) return(15);
  if(strcasecmp(p1->bloodfile,   p2->bloodfile)!=0) return(16);
  if(strcasecmp(p1->refname,     p2->refname)!=0) return(17);
  if(strcasecmp(p1->fitmethod,   p2->fitmethod)!=0) return(18);
  if(p1->dataunit  != p2->dataunit) return(21);
  if(p1->timeunit  != p2->timeunit) return(22);
  if(p1->weighting != p2->weighting) return(23);
  if(doubleMatch(p1->density,       p2->density, 1.0E-10)!=0) return(31);
  if(doubleMatch(p1->lc,            p2->lc, 1.0E-10)!=0) return(32);
  if(doubleMatch(p1->beta,          p2->beta, 1.0E-10)!=0) return(33);
  if(doubleMatch(p1->concentration, p2->concentration, 1.0E-10)!=0) return(34);
  if(doubleMatch(p1->Vb,            p2->Vb, 1.0E-10)!=0) return(35);
  if(doubleMatch(p1->fA,            p2->fA, 1.0E-10)!=0) return(36);
  if(doubleMatch(p1->E,             p2->E, 1.0E-10)!=0) return(37);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether parameter set (region) names are the same.
\return Returns 0 in case of match, and 1 if not matching.
 */
int parMatchRegionnames(
  /** Pointer to the parameter data that is tested */
  PETPAR *p1,
  /** Pointer to the parameter data that is tested */
  PETPAR *p2
) {
  int ri;

  if(p1==NULL || p2==NULL || p1->voiNr!=p2->voiNr) return(1);
  for(ri=0; ri<p1->voiNr; ri++)
    if(strcasecmp(p1->voi[ri].name, p2->voi[ri].name)!=0) return(2);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether parameter names are the same.
\return Returns 0 in case of match, and 1 if not matching.
 */
int parMatchParameternames(
  /** Pointer to the parameter data that is tested */
  PETPAR *p1,
  /** Pointer to the parameter data that is tested */
  PETPAR *p2
) {
  int i, n1, n2;
  if(p1==NULL || p2==NULL || p1->parNr!=p2->parNr) return(1);
  for(i=0; i<p1->parNr; i++)
    if(strcasecmp(p1->parname[i], p2->parname[i])!=0) return(2);
  n1=parNrofUnits(p1);
  n2=parNrofUnits(p2);
  if(n1==0 && n2==0) return(0);
  if(n1!=n2) return(11);
  for(i=0; i<p1->parNr; i++)
    if(p1->parunit[i]!=p2->parunit[i]) return(12);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether extra parameters (fit range, wss, etc) are the same.
\return Returns 0 in case of match, and 1 if not matching.
 */
int parMatchExtraParameters(
  /** Pointer to the parameter data that is tested */
  PETPAR *p1,
  /** Pointer to the parameter data that is tested */
  PETPAR *p2
) {
  int i;
  
  if(p1==NULL || p2==NULL) return(1);
  if(p1->voiNr!=p2->voiNr || p1->parNr!=p2->parNr) return(2);
  if(p1->voiNr<1) return(0);
  for(i=0; i<p1->voiNr; i++) {
    if(p1->voi[i].function!=p2->voi[i].function) return(11);
    if(p1->voi[i].model!=p2->voi[i].model) return(12);
    if(doubleMatch(p1->voi[i].start, p2->voi[i].start, 1.0E-10)!=0) return(13);
    if(doubleMatch(p1->voi[i].end, p2->voi[i].end, 1.0E-10)!=0) return(14);
    if(p1->voi[i].parNr!=p2->voi[i].parNr) return(21);
    if(p1->voi[i].dataNr!=p2->voi[i].dataNr) return(22);
    if(doubleMatch(p1->voi[i].wss, p2->voi[i].wss, 1.0E+000)!=0) return(31);
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether parameter values are the same.
 *  Values are considered to match, if either absolute or relative test
 *  is passed.
\return Returns 0 in case of match, and <>0 if not matching.
 */
int parMatchParameters(
  /** Pointer to the parameter data that is tested */
  PETPAR *p1,
  /** Pointer to the parameter data that is tested */
  PETPAR *p2,
  /** Parameter index (0..parNr-1) that is verified; <0, if all */
  int par_index,
  /** Test (1) or do not test (0) SD and Condifence limits */
  int test_sd,
  /** Maximum absolute difference that is allowed */
  double limit_abs,
  /** Maximum relative difference that is allowed */
  double limit_rel,
  /** Verbose level; if zero, nothing is printed into stdout */
  int verbose
) {
  int ri, pi;
  double s, d, r, v1, v2;

  if(verbose>0) printf("parMatchParameters(par1, par2, %d, %d, %g, %g, ...)\n",
                       par_index, test_sd, limit_abs, limit_rel);
  if(p1==NULL || p2==NULL) return(1);
  if(p1->parNr!=p2->parNr) {
    if(par_index<0) return(2);
    if(par_index+1>p1->parNr || par_index+1>p2->parNr) return(2);
  } else {
    if(par_index+1>p1->parNr) return(2);
  }
  if(p1->voiNr!=p2->voiNr) return(3);
  if(p1->voiNr<1) return(0);

  if(limit_abs<0.0) limit_abs=0.0;
  if(limit_rel<0.0) limit_rel=0.0;
  
  /* Test parameter values first */
  for(pi=0; pi<p1->parNr; pi++) {
    if(par_index>=0 && par_index!=pi) continue;
    for(ri=0; ri<p1->voiNr; ri++) {
      v1=p1->voi[ri].p[pi]; v2=p2->voi[ri].p[pi];
      if(isnan(v1) && isnan(v2)) continue; // ok if both are NaN
      if(verbose>4) {
        printf("  par1.roi[%d].param[%d] := %g\n", ri, pi, v1);
        printf("  par2.roi[%d].param[%d] := %g\n", ri, pi, v2);
      }
      s=fabs(v1+v2); d=fabs(v1-v2);
      if(isnan(s)) {  // Either one is NaN
        if(verbose>0) printf("  failed: %g vs %g\n", v1, v2);
        return(11);
      }
      /* Absolute difference */
      if(d<=limit_abs) continue;
      /* Relative difference, if possible */
      if(s<1.0E-20) continue;
      r=2.0*d/s;
      if(r<=limit_rel) continue;
      if(verbose>0) printf("  failed: %g vs %g\n", v1, v2);
      if(verbose>1) {
        printf("  par1.roi[%d].param[%d] := %g\n", ri, pi, v1);
        printf("  par2.roi[%d].param[%d] := %g\n", ri, pi, v2);
        printf("  absolute difference %g vs limit %g\n", d, limit_abs);
        printf("  relative difference %g vs limit %g\n", r, limit_rel);
      }
      return(12);
    } // next region
  } // next parameter

  /* Test parameter SD and CL only if required */
  if(test_sd==0) return(0);
  
  /* SD */
  for(pi=0; pi<p1->parNr; pi++) {
    if(par_index>=0 && par_index!=pi) continue;
    for(ri=0; ri<p1->voiNr; ri++) {
      v1=p1->voi[ri].sd[pi]; v2=p2->voi[ri].sd[pi];
      if(isnan(v1) && isnan(v2)) continue; // ok if both are NaN
      if(verbose>4) {
        printf("  par1.roi[%d].sd[%d] := %g\n", ri, pi, v1);
        printf("  par2.roi[%d].sd[%d] := %g\n", ri, pi, v2);
      }
      s=fabs(v1+v2); d=fabs(v1-v2);
      if(isnan(s)) {  // Either one is NaN
        if(verbose>0) printf("  failed: %g vs %g\n", v1, v2);
        return(21);
      }
      /* Absolute difference */
      if(d<=limit_abs) continue;
      /* Relative difference, if possible */
      if(s<1.0E-20) continue;
      r=2.0*d/s;
      if(r<=limit_rel) continue;
      if(verbose>0) printf("  failed: %g vs %g\n", v1, v2);
      if(verbose>1) {
        printf("  par1.roi[%d].sd[%d] := %g\n", ri, pi, v1);
        printf("  par2.roi[%d].sd[%d] := %g\n", ri, pi, v2);
        printf("  absolute difference %g vs limit %g\n", d, limit_abs);
        printf("  relative difference %g vs limit %g\n", r, limit_rel);
      }
      return(22);
    } // next region
  } // next parameter
  
  /* CL1 */
  for(pi=0; pi<p1->parNr; pi++) {
    if(par_index>=0 && par_index!=pi) continue;
    for(ri=0; ri<p1->voiNr; ri++) {
      v1=p1->voi[ri].cl1[pi]; v2=p2->voi[ri].cl1[pi];
      if(isnan(v1) && isnan(v2)) continue; // ok if both are NaN
      if(verbose>4) {
        printf("  par1.roi[%d].cl1[%d] := %g\n", ri, pi, v1);
        printf("  par2.roi[%d].cl1[%d] := %g\n", ri, pi, v2);
      }
      s=fabs(v1+v2); d=fabs(v1-v2);
      if(isnan(s)) {  // Either one is NaN
        if(verbose>0) printf("  failed: %g vs %g\n", v1, v2);
        return(31);
      }
      /* Absolute difference */
      if(d<=limit_abs) continue;
      /* Relative difference, if possible */
      if(s<1.0E-20) continue;
      r=2.0*d/s;
      if(r<=limit_rel) continue;
      if(verbose>0) printf("  failed: %g vs %g\n", v1, v2);
      if(verbose>1) {
        printf("  par1.roi[%d].cl1[%d] := %g\n", ri, pi, v1);
        printf("  par2.roi[%d].cl1[%d] := %g\n", ri, pi, v2);
        printf("  absolute difference %g vs limit %g\n", d, limit_abs);
        printf("  relative difference %g vs limit %g\n", r, limit_rel);
      }
      return(32);
    } // next region
  } // next parameter
  
  /* CL2 */
  for(pi=0; pi<p1->parNr; pi++) {
    if(par_index>=0 && par_index!=pi) continue;
    for(ri=0; ri<p1->voiNr; ri++) {
      v1=p1->voi[ri].cl2[pi]; v2=p2->voi[ri].cl2[pi];
      if(isnan(v1) && isnan(v2)) continue; // ok if both are NaN
      if(verbose>4) {
        printf("  par1.roi[%d].cl2[%d] := %g\n", ri, pi, v1);
        printf("  par2.roi[%d].cl2[%d] := %g\n", ri, pi, v2);
      }
      s=fabs(v1+v2); d=fabs(v1-v2);
      if(isnan(s)) {  // Either one is NaN
        if(verbose>0) printf("  failed: %g vs %g\n", v1, v2);
        return(41);
      }
      /* Absolute difference */
      if(d<=limit_abs) continue;
      /* Relative difference, if possible */
      if(s<1.0E-20) continue;
      r=2.0*d/s;
      if(r<=limit_rel) continue;
      if(verbose>0) printf("  failed: %g vs %g\n", v1, v2);
      if(verbose>1) {
        printf("  par1.roi[%d].cl2[%d] := %g\n", ri, pi, v1);
        printf("  par2.roi[%d].cl2[%d] := %g\n", ri, pi, v2);
        printf("  absolute difference %g vs limit %g\n", d, limit_abs);
        printf("  relative difference %g vs limit %g\n", r, limit_rel);
      }
      return(42);
    } // next region
  } // next parameter

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether PETPAR struct contents are the same.
\return Returns 0 in case of match, and <>0 if not matching.
 */
int parMatch(
  /** Pointer to the parameter data that is tested */
  PETPAR *par1,
  /** Pointer to the parameter data that is tested */
  PETPAR *par2,
  /** Verbose level; if zero, then only warnings are printed into stderr */
  int verbose
) {
  int ret;
  
  if(verbose>0) printf("parMatch(par1, par2, ...)\n");
  if(par1==NULL || par2==NULL) {
    if(verbose>0) printf("invalid pointers\n");
    return(1);
  }
  if(verbose>10) {
    printf("\nParameter data #1:\n"); parPrint(par1);
    printf("\nParameter data #2:\n"); parPrint(par2);
    printf("\n");
  }

  /* Check header fields */
  ret=parMatchHeader(par1, par2); if(ret!=0) {
    if(verbose>0) printf("no match in headers (%d)\n", ret);
    return(100+ret);
  }
  /* Check region names */
  ret=parMatchRegionnames(par1, par2); if(ret!=0) {
    if(verbose>0) printf("no match in region names (%d)\n", ret);
    return(200+ret);
  }
  /* Check parameter names */
  ret=parMatchParameternames(par1, par2); if(ret!=0) {
    if(verbose>0) printf("no match in parameter names (%d)\n", ret);
    return(300+ret);
  }
  /* Check extra parameters */
  ret=parMatchExtraParameters(par1, par2); if(ret!=0) {
    if(verbose>0) printf("no match in extra parameters (%d)\n", ret);
    return(400+ret);
  }
  /* Check parameters, including SD and CLs */
  ret=parMatchParameters(par1, par2, -1, 1, 1.0E-03, 1.0E-03, verbose-1);
  if(ret!=0) {
    if(verbose>0) printf("no match in parameters (%d)\n", ret);
    return(500+ret);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if any of parameter sets has WSS.
\return Returns 1 if WSS exists, and 0 if not (or in case of an error).
 */ 
int parIsWSS(
  PETPAR *par
) {
  int ri;
  if(par==NULL || par->voiNr<1) return(0);
  for(ri=0; ri<par->voiNr; ri++) if(!isnan(par->voi[ri].wss)) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if any of parameter sets has fit range specified.
\return Returns 1 if it exists, and 0 if not (or in case of an error).
 */ 
int parIsFitrange(
  PETPAR *par
) {
  int ri;
  if(par==NULL || par->voiNr<1) return(0);
  for(ri=0; ri<par->voiNr; ri++)
    if(!isnan(par->voi[ri].start) && !isnan(par->voi[ri].end)) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if any of parameter sets has function id specified.
\return Returns 0 if not, 1 if the same in each set, and 2 if variable.
 */ 
int parIsFunction(
  PETPAR *par
) {
  int ri, ret=0;
  if(par==NULL || par->voiNr<1) return(0);
  for(ri=0; ri<par->voiNr; ri++) {
    if(ret==0 && par->voi[ri].function>0) ret=1;
    if(ri>0 && par->voi[ri].function!=par->voi[ri-1].function) ret=2;
  }
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if any of parameter sets has model id specified.
\return Returns 0 if not, 1 if the same in each set, and 2 if variable.
 */ 
int parIsModel(
  PETPAR *par
) {
  int ri, ret=0;
  if(par==NULL || par->voiNr<1) return(0);
  for(ri=0; ri<par->voiNr; ri++) {
    if(ret==0 && par->voi[ri].model>0) ret=1;
    if(ri>0 && par->voi[ri].model!=par->voi[ri-1].model) ret=2;
  }
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if any of parameter sets has parNr specified.
\return Returns the highest parNr, or 0 if not specified or in case of an error.
 */ 
int parHighestParnr(
  PETPAR *par
) {
  int ri, n=0;
  if(par==NULL || par->voiNr<1) return(0);
  for(ri=0; ri<par->voiNr; ri++) if(par->voi[ri].parNr>n) n=par->voi[ri].parNr;
  return(n);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if any of parameter sets has dataNr specified.
\return Returns the highest dataNr, or 0 if not specified or in case of an error.
 */ 
int parHighestDatanr(
  PETPAR *par
) {
  int ri, n=0;
  if(par==NULL || par->voiNr<1) return(0);
  for(ri=0; ri<par->voiNr; ri++) if(par->voi[ri].dataNr>n) n=par->voi[ri].dataNr;
  return(n);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if specified parameter has SD in at least one of the regions.
\return Returns 1 if SD exists, and 0 if not (or in case of an error).
 */ 
int parSDWithPar(
  PETPAR *par,
  int pi
) {
  int ri;
  if(par==NULL || par->voiNr<1 || pi<0 || pi>=MAX_PETPARAMS) return(0);
  for(ri=0; ri<par->voiNr; ri++) if(!isnan(par->voi[ri].sd[pi])) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if specified parameter has confidence limits in at least one
 *  of the regions.
\return Returns 1 if CLs exist, and 0 if not (or in case of an error).
 */ 
int parCLWithPar(
  PETPAR *par,
  int pi
) {
  int ri;
  if(par==NULL || par->voiNr<1 || pi<0 || pi>=MAX_PETPARAMS) return(0);
  for(ri=0; ri<par->voiNr; ri++)
    if(!isnan(par->voi[ri].cl1[pi]) && !isnan(par->voi[ri].cl2[pi])) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if specified parameter struct contains parameter units.
\return Returns the nr of specified parameter units.
 */ 
int parNrofUnits(
  PETPAR *par
) {
  int pi, n=0;
  if(par==NULL || par->parNr<1) return(0);
  for(pi=0; pi<par->parNr; pi++) if(par->parunit[pi]!=CUNIT_UNKNOWN) n++;
  return(n);
}
/*****************************************************************************/

/*****************************************************************************/
