/******************************************************************************

  Copyright (c) 2003-2013 by Turku PET Centre

  Program:     reslist
  Description: Program for listing part of result file contents.

  This program is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation; either version 3 of the License, or (at your option) any later
  version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 Temple
  Place, Suite 330, Boston, MA 02111-1307 USA.

  Turku PET Centre hereby disclaims all copyright interest in the program.
  Juhani Knuuti
  Director, Professor
  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/

  Modification history:
  1.0   2003-01-19 Vesa Oikonen
        First created.
        2003-12-02 VO
        Removed unnecessary includes.
  1.1   2004-01-21 VO
        Works with only filename as an argument.
  1.1.1 2004-11-28 VO
        Accepts options -V and -H.
  1.2.1 2006-07-12 VO
        Compiled with new library versions.
        Uses library functions instead of local procedures.
        Changes in info text.
  1.2.2 2007-09-13 VO
        Compiled with new library versions.
        Changes in info text.
        Uses brand new resSelectRegions() instead of resSelect().
  1.2.3 2012-04-20 VO
        Updated build info.
        Added option --verbose.
        Defined NA replaced by NaN.
  1.3.0 2013-07-08 VO
        Started implementing undocumented feature to read RES and FIT files
        in nonstandard CSV format.



******************************************************************************/
#define PROGRAM "reslist 1.3.0 (c) 2003-2013 by Turku PET Centre"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "curveio.h"
#include "libtpcmisc.h"
/*****************************************************************************/
/* Local functions */
void print_usage();
void print_build();
/*****************************************************************************/
int TEST=0;
int SILENT=0;
int VERBOSE=0;
/*****************************************************************************/

/*****************************************************************************/
/** Print contents of CSV struct in stdout */
void csvPrint(
  /** Pointer to struct containing CSV data */
  CSV *csv
) {
  int i, row;
  if(csv==NULL) {printf("csv := NULL\n"); return;}
  if(csv->nr<1) {printf("csv := empty\n"); return;}
  printf("csv_nr := %d\n", csv->nr);
  printf("csv_row_nr := %d\n", csv->row_nr);
  printf("csv_col_nr := %d\n", csv->col_nr);
  printf("csv_separator := %c\n", csv->separator);
  row=0;
  for(i=0; i<csv->nr; i++) {
    if(csv->c[i].row>row) {row=csv->c[i].row; printf("\n");}
    else if(csv->c[i].col>1) printf("\t");
    if(csv->c[i].content!=NULL) printf("'%s'", csv->c[i].content);
    else printf("''");
  }
  printf("\n");
  return;
}
/*****************************************************************************/

/*****************************************************************************/
/** Read simple result table in CSV format into RES struct.
 *
\return Returns <>0 in case of an error. 
 */ 
int resReadCSV(
  /** Pointer to string containing filename */
  char *fname,
  /** Poiner to initiated RES struct */
  RES *res,
  /** Verbose level; if zero, then only warnings are printed into stderr */
  int verbose,
  /** Pointer to a string (allocated for at least 256 chars) where error message
      or other execution status will be written; enter NULL, if not needed */     
  char *status
) {
  int i, j, k, ret;
  CSV csv;
  int title_index, units_index, data_index, colNr, rowNr, voiNr;
  double f;
  char *cptr1, *cptr2;

  if(verbose>0) printf("resReadCSV(%s, *res, ...)\n", fname);
  if(status!=NULL) strcpy(status, "invalid function parameter");
  if(fname==NULL || strlen(fname)<1 || res==NULL) return(CSV_ERROR);

  /* Try to read file as CSV */
  csvInit(&csv);
  ret=csvRead(&csv, fname);
  if(ret!=0) {
    if(verbose>0) printf("csvRead(%s) := %d\n", fname, ret);
    if(status!=NULL) strcpy(status, "cannot read file");
    csvEmpty(&csv); return(ret);
  }
  if(verbose>10) csvPrint(&csv);

  /* Find the title line, starting with 'Region' */
  title_index=-1;
  for(i=0; i<csv.nr; i++) {
    if(verbose>25)
      printf("col=%d row=%d content='%s'\n",
             csv.c[i].col, csv.c[i].row, csv.c[i].content);
    if(csv.c[i].col!=1) continue;
    if(strcasecmp(csv.c[i].content, "REGION")==0) {title_index=i;}
  }
  if(title_index<0) {
    if(verbose>0) printf("title line not found in %s\n", fname);
    if(status!=NULL) strcpy(status, "file format not supported");
    csvEmpty(&csv); return(3);
  }
  /* Determine the nr of columns from title line */
  for(i=title_index+1, colNr=1; i<csv.nr; i++)
    if(csv.c[i].row==csv.c[i-1].row) colNr++; else break;
  if(colNr<2) {
    if(verbose>0) printf("invalid title line in %s\n", fname);
    if(status!=NULL) strcpy(status, "invalid title line");
    csvEmpty(&csv); return(CSV_INVALIDFORMAT);
  }
  if(verbose>1) printf("colNr := %d\n", colNr);
  /* Preliminary index for data start is right after the title line */
  data_index=title_index+colNr;
  /* Check if the line below titles contains the units */
  units_index=-1;
  if(strncasecmp(csv.c[i].content, "UNITS", 4)==0) {
    units_index=i;
    /* then also data is starting one row later */
    data_index+=colNr;
  }
  if(verbose>1) {
    if(units_index<0) printf("units not specified\n");
    else printf("units specified on line %d\n", csv.c[units_index].row);
  }
  /* Determine the nr of rows containing equal nr of columns as title line */
  for(i=title_index+colNr+1, rowNr=0, j=1; i<csv.nr; i++) {
    //if(verbose>26) printf("  i=%d j=%d\n", i, j);
    if(csv.c[i].row==csv.c[i-1].row) {j++; continue;}
    /*if(verbose>20)
      printf("new data row at content[%d]='%s'\n", i, csv.c[i].content);*/
    if(j!=colNr) break;
    j=1; rowNr++;
  }
  if(i==csv.nr && j==colNr) rowNr++;
  if(units_index>=0) rowNr--; // exclude units line from rowNr
  if(verbose>1) printf("rowNr := %d\n", rowNr);
  /* Check if table contains SD or CL rows */
  for(i=data_index, j=0; i<csv.nr; i+=colNr) {
    cptr1=csv.c[i].content;
    if(strcasecmp(cptr1, "SD")==0) {j++; continue;}
    if(strncasecmp(cptr1, "CL ", 3)==0) {j++; continue;}
  }
  if(verbose>6) printf("%d rows with SD or CL\n", j);
  voiNr=rowNr-j;
  if(verbose>1) printf("voiNr := %d\n", voiNr);
  if(voiNr<1) {
    if(verbose>0) printf("invalid data table in %s\n", fname);
    if(status!=NULL) strcpy(status, "invalid table contents");
    csvEmpty(&csv); return(CSV_NOTABLE);
  }
  /* Allocate RES, delete any previous contents */
  if(verbose>3) printf("allocating memory for RES\n");
  if((colNr-1)>MAX_RESPARAMS) {
    if(verbose>0) printf("too many columns in data table in %s\n", fname);
    if(status!=NULL) strcpy(status, "table is too large");
    csvEmpty(&csv); return(CSV_TOOBIG);
  }
  resEmpty(res);
  ret=resSetmem(res, voiNr);
  if(ret!=0) {
    if(verbose>0) printf("resSetmem(*res, %d) := %d\n", voiNr, ret);
    if(status!=NULL) strcpy(status, "cannot allocate memory");
    csvEmpty(&csv); return(CSV_OUTOFMEMORY);
  }
  /* Fill RES contents from CSV */
  if(verbose>3) printf("filling RES\n");
  res->parNr=colNr-1;
  res->voiNr=voiNr;
  //data_index=title_index+colNr; if(units_index>=0) data_index+=colNr;
  /* Parameter names */
  for(i=title_index+1, j=0; j<colNr-1; i++, j++)
    strncpy(res->parname[j], csv.c[i].content, MAX_RESPARNAME_LEN);
  /* Units */
  if(units_index>=0) for(i=units_index+1, j=0; j<colNr-1; i++, j++)
    strncpy(res->parunit[j], csv.c[i].content, MAX_RESPARNAME_LEN);
  /* Region names */
  for(j=0, k=0; j<rowNr && k<voiNr; j++) {
    cptr1=csv.c[data_index+j*colNr].content;
    if(strcasecmp(cptr1, "SD")==0) continue;
    if(strncasecmp(cptr1, "CL ", 3)==0) continue;
    strncpy(res->voi[k].name, cptr1, MAX_REGIONNAME_LEN);
    rnameSplit(res->voi[k].name, res->voi[k].voiname, res->voi[k].hemisphere,
               res->voi[k].place, MAX_REGIONNAME_LEN);
    k++;
  }
  /* Parameter values */
  for(j=0, k=0; j<rowNr; j++) {
    cptr1=csv.c[data_index+j*colNr].content;

    if(strcasecmp(cptr1, "SD")==0) {
      if(k<1) continue;
      for(i=1; i<colNr; i++) {
        cptr2=csv.c[data_index+j*colNr+i].content;
        ret=atof_with_check(cptr2, &f);
        if(ret!=0) res->voi[k-1].sd[i-1]=nan("");
        else res->voi[k-1].sd[i-1]=f;
      }
      continue;
    }

    if(strcasecmp(cptr1, "CL 95% Lower")==0) {
      if(k<1) continue;
      for(i=1; i<colNr; i++) {
        cptr2=csv.c[data_index+j*colNr+i].content;
        ret=atof_with_check(cptr2, &f);
        if(ret!=0) res->voi[k-1].cl1[i-1]=nan("");
        else res->voi[k-1].cl1[i-1]=f;
      }
      continue;
    }

    if(strcasecmp(cptr1, "CL 95% Upper")==0) {
      if(k<1) continue;
      for(i=1; i<colNr; i++) {
        cptr2=csv.c[data_index+j*colNr+i].content;
        ret=atof_with_check(cptr2, &f);
        if(ret!=0) res->voi[k-1].cl2[i-1]=nan("");
        else res->voi[k-1].cl2[i-1]=f;
      }
      continue;
    }

    for(i=1; i<colNr; i++) {
      cptr2=csv.c[data_index+j*colNr+i].content;
      ret=atof_with_check(cptr2, &f);
      if(ret!=0) res->voi[k].parameter[i-1]=nan("");
      else res->voi[k].parameter[i-1]=f;
    }
    k++;
  }
  /* Try to find any extra information from lines before parameter table */
  for(i=0; i<title_index; i++) {
    /*printf("  content := '%s' col=%d row=%d\n", csv.c[i].content,
            csv.c[i].col, csv.c[i].row);*/
    /* Check that this is a first column */
    if(csv.c[i].col!=1) continue;
    if(csv.c[i].content==NULL) continue;
    /* Check that a second column with value exists */
    if(csv.c[i+1].row!=csv.c[i].row) continue;
    if(csv.c[i+1].content==NULL) {i++; continue;}
    if(verbose>5) printf("  info name := '%s' value := '%s'\n",
                         csv.c[i].content, csv.c[i+1].content);
    cptr1=csv.c[i].content; cptr2=csv.c[i+1].content;
    if(strcasecmp(cptr1, "STUDYNR")==0 || strcasecmp(cptr1, "STUDY")==0) {
      strncpy(res->studynr, cptr2, MAX_STUDYNR_LEN);
      i++; continue;
    }
    if(strcasecmp(cptr1, "DATE")==0 || strcasecmp(cptr1, "STUDY_DATE")==0) {
      struct tm st;
      if(get_datetime(cptr2, &st)==0) res->time=mktime(&st);
      else if(get_date(cptr2, &st)==0) res->time=mktime(&st);
      else res->time=(time_t)0;
      i++; continue;
    }
    if(strcasecmp(cptr1, "WEIGHTING")==0) {
      if(strncasecmp(cptr2, "YES", 1)==0 || strcasecmp(cptr2, "ON")==0)
        res->isweight=1;
      else if(strncasecmp(cptr2, "NO", 1)==0 || strcasecmp(cptr2, "OFF")==0)
        res->isweight=0;
      else
        res->isweight=-1;
      i++; continue;
    }
    
    if(verbose>1) printf("  info title '%s' not identified\n", cptr1);
  } // next info line

  csvEmpty(&csv);
  if(status!=NULL) strcpy(status, "ok");
  return CSV_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/*
 *   main()
 */
int main(int argc, char *argv[])
{
  int ai, pi, ri, ret, n, listStat=1;
  RES res;
  char resfile[FILENAME_MAX], voiname[128], *cptr;
  char tmp[512];


  /*
   *  Check arguments
   */
  if(argc==1) {print_usage(); return(1);}
  resfile[0]=voiname[0]=(char)0;
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(strncasecmp(cptr, "TEST", 4)==0) {
      cptr+=4; if(*cptr=='=') TEST=atoi(cptr+1); else TEST=1;
      continue;
    } else if(strcasecmp(cptr, "V")==0 || strcasecmp(cptr, "VERSION")==0 ||
              strcasecmp(cptr, "BUILD")==0) {
      print_build();
      return(0);
    } else if(strcasecmp(cptr, "H")==0 || strcasecmp(cptr, "HELP")==0) {
      print_usage(); return(0);
    } else if(strcasecmp(cptr, "SILENT")==0) {
      SILENT=1; VERBOSE=0; continue;
    } else if(strcasecmp(cptr, "VERBOSE")==0) {
      VERBOSE=1; SILENT=0; continue;
    } else if(*cptr=='N' || *cptr=='n') {
      listStat=0; continue;
    }
    fprintf(stderr, "Error: invalid option %s\n", argv[ai]);
    return(1);
  } else {
    if(!resfile[0]) {strcpy(resfile, argv[ai]); continue;}
    if(!voiname[0]) {strcpy(voiname, argv[ai]); continue;}
    print_usage(); return(1);
  }
  /* Did we get all the information that we need? */
  if(!resfile[0]) {print_usage(); return(1);}
  /* Print arguments for testing */
  if(TEST) {
    printf("TEST := %d\n", TEST);
    printf("SILENT := %d\n", SILENT);
    printf("VERBOSE := %d\n", VERBOSE);
    printf("resfile := %s\n", resfile);
    printf("voiname := %s\n", voiname);
    printf("listStat := %d\n", listStat);
  }
  if(TEST>0) VERBOSE=1;
  if(TEST>1) RESULT_TEST=TEST-1;


  /*
   *  Read result file
   */
  if(VERBOSE) printf("reading %s\n", resfile);
  resInit(&res);
  ret=resRead(resfile, &res);
  if(ret) {
    /* Not successful, try to read as nonstandard CSV format */
    ret=resReadCSV(resfile, &res, TEST-1, tmp);
    if(ret==CSV_OK) {
      if(SILENT==0) fprintf(stderr,
                     "Warning: '%s' is not in standard RES format.\n", resfile);
    }
    if(ret!=0) {
      fprintf(stderr, "Error in reading '%s': %s\n", resfile, reserrmsg);
      resEmpty(&res);
      return(3);
    }
  }
  if(TEST>10) resPrint(&res);

  /*
   *  Select the specified regions
   */
  if(voiname[0]) {
    if(VERBOSE) printf("searching region(s) '%s'\n", voiname);
    for(ri=0; ri<res.voiNr; ri++) res.voi[ri].sw=0;
    n=resSelectRegions(&res, voiname, 1);
    if(n<=0) {        /* no vois are matching */
      fprintf(stderr, "Error: Cannot find voi '%s'.\n", voiname);
      resEmpty(&res); return(4);
    }
  } else {
    for(ri=0; ri<res.voiNr; ri++) res.voi[ri].sw=1;
  }


  /*
   *  Delete those regions that were not selected
   */
  ri=res.voiNr-1;
  while(ri>=0) {
    if(res.voi[ri].sw!=1) ret=resDelete(&res, ri);
    ri--;
  }


  /*
   *  Set SDs and CVs to NaN, if they are not to be printed
   */
  if(listStat==0) for(ri=0; ri<res.voiNr; ri++) for(pi=0; pi<res.parNr; pi++) {
    res.voi[ri].sd[pi]=nan("");
    res.voi[ri].cl1[pi]=nan("");
    res.voi[ri].cl2[pi]=nan("");
  }


  /*
   *  Print the ones that are left
   */
  resPrint(&res); fprintf(stdout, "\n");

  resEmpty(&res);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/*
 *   print_usage()
 */
void print_usage()
{
  int i=0;
  static char *info[] = {
  " ",
  PROGRAM,
  " ",
  "Lists the user-specified regional results on screen.",
  "Note: result file must not be in HTML format.",
  " ",
  "Usage: reslist [Options] <Results file> [VOI name]",
  " ",
  "Example: reslist ut6789ki.res put",
  " ",
  "Options:",
  " -n  CL or SD values are not shown",
  " -h or --help",
  "     Print this message and exit",
  " --version or --build",
  "     Print software build information and exit.",
  " ",
  "See also: resdel, rescoll, resmatch, res2html, resdiff, fit2res, resrmdpl",
  " ",
  "Keywords: RES, FIT, results, tools",
  " ",
  "This program comes with ABSOLUTELY NO WARRANTY. This is free software, and",
  "you are welcome to redistribute it under GNU General Public License.",
  " ",
  0};

  while(info[i]!=0) printf("  %s\n", info[i++]);
}
/*****************************************************************************/

/*****************************************************************************/
/*!
 * Prints build information to stdout.
 */
void print_build() {

  printf("\n %s\n", PROGRAM);
  printf("\n Build %s %s\n\n",__DATE__,__TIME__);
#if defined(__FILE__) && defined(__TIMESTAMP__)
  printf(" File %s last edited on %s\n", __FILE__, __TIMESTAMP__);
#endif
  printf(" _______________________________________________\n");

  libtpccurveio_print_build(stdout);
  libtpcmisc_print_build(stdout);
  printf(" _______________________________________________\n\n");

#if defined(__STDC_VERSION__)
  printf(" Version of C: %ld\n", __STDC_VERSION__);
#endif
#if defined(__STDC_ISO_10646__)
  printf(" Compiler ISO/IEC 10646 specification: %ld\n", __STDC_ISO_10646__);
#endif
#if defined(__GNUC__) && defined(__VERSION__)
  printf(" GNU C version: %s\n", __VERSION__);
#endif
#ifdef _OPENMP
  printf(" OpenMP version: %d\n", _OPENMP);
#endif
#if defined(__x86_64__) || defined(__LP64__) || defined(__ppc64__)
  printf(" Architecture: 64-bit\n");
#else
  printf(" Architecture: 32-bit\n");
#endif

  printf(" _______________________________________________\n\n");
}
/*****************************************************************************/

/*****************************************************************************/

