/***********************************************************
*
*  Filename:      ellipse.c  (c) 2003-2004 Turku PET Centre
*
*  First version: 10.07.2003
*
*  Programmer:    Jarkko Johansson
*
*  Description:   ELLIPSE is a structure modelling an ellipse
*                 on two dimensional plane.
*
*  Updated:       20.4.2004 version 1.0
*
**********************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>

/*****************************************************************************/

#include "include/ellipse.h"

/*****************************************************************************/

/* Initialization and memory handling for ellipse data.*/

/**
   Initializes ELLIPSE datatype for use. 
   To be used before any use of ELLIPSE type variables.
   @precondition .
   @postcondition ellipse is initialized.
 @param ELLIPSE *ell pointer to ellipse to be initialized. 
 */
void ellipseInit(ELLIPSE *ell){
  
  if(ELLIPSE_VERBOSE) printf("ELLIPSE: ellipseInit(). \n");

  /*Init the information on ellipse.*/
  //buffer ell to contain zero
  memset(ell, 0, sizeof(ELLIPSE)); 
  ell->status=ELLIPSE_STATUS_INITIALIZED;
  ell->semiaxis[0]=ell->semiaxis[1]=0.;
  ell->center[0]=ell->center[1]=0.;
  ell->inclination=0.;
  ell->imageDim=0;
  ell->value=0;

  /*Init the ellipse array*/
  ell->ellipseptr=(int**)NULL;

}//END OF INIT

/**
   Frees the memory allocated for ellipse.
   All data is cleared.
   @precondition .
   @postcondition ellipse is emptied.
   @param ELLIPSE ell pointer to ellipse to be emptied.
*/
void ellipseEmpty(ELLIPSE *ell){

 if(ELLIPSE_VERBOSE) printf("ELLIPSE: ellipseEmpty(). \n");
 
 //if ell is already empty
 if(ell->status<ELLIPSE_STATUS_OCCUPIED) return;

 //free the memory occupied by ellipse array
 free(ell->ellipseptr);

 //Init the information again
  ell->semiaxis[0]=ell->semiaxis[1]=0.;
  ell->center[0]=ell->center[1]=0.;
  ell->inclination=0.;
  ell->imageDim=0;
  ell->value=0;
  /*Init the ellipse array*/
  ell->ellipseptr=(int**)NULL;

  ell->status=ELLIPSE_STATUS_INITIALIZED;
}

/**
   Prints the information of the ellipse to the screen.
 @param ELLIPSE *ell pointer to ellipse to be printed.
 */
void ellipseInfo(ELLIPSE *ell){
  printf("ellipseInfo()\n");
  if(ell->status==ELLIPSE_STATUS_UNINITIALIZED){
    printf("Ellipse data not initialized.\n"); return;
  }
  printf("Semiaxis: (%.5f,%.5f)\n",ell->semiaxis[0], ell->semiaxis[1]);
  printf("Center: (%.2f,%.2f)\n",ell->center[0],ell->center[1]);
  printf("Inclination (degrees): %f\n",ell->inclination);
  printf("Image dimension: %i\n",ell->imageDim);
  printf("Value: %f\n",ell->value);
}

/**
   Allocates memory for ELLIPSE data. Normally used only in SET-functions.   
   @precondition *ell is initialized && imgDim is positive 
   @postcondition memory is allocated for ELLIPSE structure.
   @param ELLIPSE *ell pointer to ellipse for which the allocation is done.
   @param int imgDim size of the image plane on which the ellipse is to be done.
   @return 0 if ok.
 */
int ellipseAllocate(ELLIPSE *ell, int imgDim){

  int i;

  if(ELLIPSE_VERBOSE) printf("ELLIPSE:ellipseAllocate(*ell,%i) \n",imgDim);
  
  /*check the arguments*/
  if(ell->status==ELLIPSE_STATUS_UNINITIALIZED) return(1);
  if(imgDim<1 || imgDim>4096) return(2);
  
  //allocate memory
  ell->ellipseptr=(int**)calloc(imgDim,sizeof(int*));
  for(i=0;i<imgDim;i++){
    ell->ellipseptr[i]=(int*)calloc(imgDim,sizeof(int));
  }
  return(0);
}

/* GET and SET procedures for ELLIPSE datatype. USE ONLY THESE FOR SETTING AND GETTING. */

 /**
   Sets the ellipse according to given coordinates and image dimension.
   Notice that the origin is in the middle.
   @precondition *ell is initialized.
   @precondition imgDim is positive.
   @precondition semis[i] are positive.
   @precondition -imgDim/2<=cent[i]<=imgDim/2.
   @postcondition ellipse is set.
   @param ELLIPSE *ell pointer to ellipse for which the setting is to be done.
   @param int imgDim size of the image plane.
   @param float *semis major and minor semiaxis of the ellipse.
   @param float *cent center of the ellipse.
   @param float incli inclination of the ellipse.
   @param float value value inside the ellipse.
   @return 0 if ok.
*/
int ellipseSetFromParams(ELLIPSE *ell, int imgDim, float *semis, float *cent, float incli, float val){

  float    d, x, y, inclination;
  int      row, col;
 
  /*check the arguments*/
  if(ell->status==ELLIPSE_STATUS_UNINITIALIZED) return(1);
  if(imgDim<1 || imgDim>4096) return(2);
  if(0>semis[0] || 0>semis[1]) return(3);
  if(-(float)imgDim/2.>cent[0] || (float)imgDim/2.<cent[0]) return(4);
  if(-(float)imgDim/2.>cent[1] || (float)imgDim/2.<cent[1]) return(5);

  //allocate memory for ellipse
  ellipseAllocate(ell,imgDim);

  //set the information on ellipse
  ell->semiaxis[0]=semis[0];
  ell->semiaxis[1]=semis[1];
  ell->center[0]=cent[0];
  ell->center[1]=cent[1];
  ell->inclination=incli;
  ell->imageDim=imgDim;
  ell->value=val;

  //set inclination ratio for setting inclination approximately
  inclination= -(float)(incli/90.);

  //set ellipse array according to given arguments
  for(row=0; row<imgDim; row++) {
   for(col=0; col<imgDim; col++) {
     //the y-coordinate of row in coordinates where origin is set in the middle
     y=(float)imgDim*0.5 - (float)row - cent[1]; 
     //the x-coordinate of column in coordinates where origin is set in the middle
     //and inclination is added
     x=(float)col - (float)imgDim*0.5 - (cent[0] + inclination*y); 
     d=(x*x)/(semis[0]*semis[0]) + (y*y)/(semis[1]*semis[1]); 
       //if this pixel is inside the ellipse
       if(d<1.0) { 
         //put 1 in place (row,col)
	 ell->ellipseptr[row][col]=1;
       }
   }//END OF COL-LOOP 
  }//END OF ROW-LOOP
  
  return(0);
}

/**
   Reads one ellipse from the given file to the given ELLIPSE structure.
   A coordinate file contains the parameters of the ellipses in one line
   in the following order:
   Coordinate 1:  v    the additive intensity value of the ellipse
   Coordinate 2:  a    the length of the horizontal semi-axis of the ellipse
   Coordinate 3:  b    the length of the vertical semi-axis of the ellipse
   Coordinate 4:  x    the x-coordinate of the center of the ellipse
   Coordinate 5:  y    the y-coordinate of the center of the ellipse
   Coordinate 6:  p    the angle (in degrees) between the horizontal semi-axis
                       of the ellipse and the x-axis of the image        
   Coordinate 7:  d    the image dimension
   @precondition *fp is a pointer to open file containing ellipse(s) in correct format.
   @precondition *ell is initialized 
   @postcondition an ellipse is read from the file fname. 
   @param FILE *fp a pointer to open file containing ellipse(s).
   @param ELLIPSE *ell pointer to ELLIPSE structure where the read ellipse is to be set.
   @return 0 if ok 1 if there were no more ellipses.
*/
int ellipseReadEllipse(FILE *fp, ELLIPSE *ell){

  float v, incli, semiaxis[2], center[2], imgDim;
  int ret=0;

  if(ELLIPSE_VERBOSE) printf("ellipseReadEllipse() \n");

  fread(&v,sizeof(float),1,fp);
  fread(&semiaxis[0],sizeof(float),1,fp);
  fread(&semiaxis[1],sizeof(float),1,fp);
  fread(&center[0],sizeof(float),1,fp);
  fread(&center[1],sizeof(float),1,fp);
  fread(&incli,sizeof(float),1,fp);
  fread(&imgDim,sizeof(float),1,fp);

  if(feof(fp)) return 1;

  //create the ellipse
  ret=ellipseSetFromParams(ell, imgDim, semiaxis, center, incli,v);
  if(ret){ ellipseEmpty(ell); return ret;}
  return 0;
}

/**
   Adds the given ellipse to the file.
   @precondition *ell is initialized.
   @postcondition ellipse is saved in the file called fname. 
   @param ELLIPSE *ell pointer to the ellipse to be saved.
   @param FILE *fp open file for saving.
   @return 0 if OK.
*/
int ellipseSaveEllipse(ELLIPSE *ell, FILE *fp){

  if(ELLIPSE_VERBOSE) printf("ellipseSaveEllipse()\n");

  // Put ellipse in the file.
  fwrite(&(ell->value),sizeof(float),1,fp);
  fwrite(&(ell->semiaxis[0]),sizeof(float),1,fp);
  fwrite(&(ell->semiaxis[1]),sizeof(float),1,fp);
  fwrite(&(ell->center[0]),sizeof(float),1,fp);
  fwrite(&(ell->center[1]),sizeof(float),1,fp);
  fwrite(&(ell->inclination),sizeof(float),1,fp);
  fwrite(&(ell->imageDim),sizeof(float),1,fp);

  return(0);
}

/**
   Returns the major semiaxe of the ellipse.
   @precondition ell is initialized. 
   @postcondition .
   @param ELLIPSE *ell pointer to ellipse.
   @return the major semiaxe of the given ellipse, some negative value if ERROR.
 */
float ellipseGetMajor(ELLIPSE *ell){

  if(ELLIPSE_VERBOSE) printf("ELLIPSE:ellipseGetMajor(). \n");
  
  if(ell->status==ELLIPSE_STATUS_UNINITIALIZED) return(-1);

  return(ell->semiaxis[0]);
}

/**
   Returns the minor semiaxe of the ellipse.
   @precondition ell is initialized. 
   @postcondition .
   @param ELLIPSE *ell pointer to ellipse.
   @return the minor semiaxe of the given ellipse, some negative value if ERROR.
 */
float ellipseGetMinor(ELLIPSE *ell){

  if(ELLIPSE_VERBOSE) printf("ELLIPSE:ellipseGetMinor(). \n");
  
  if(ell->status==ELLIPSE_STATUS_UNINITIALIZED) return(-1);

  return(ell->semiaxis[1]);
}

/**
   Returns the center x-coordinate of the ellipse.
   @precondition ell is initialized. 
   @postcondition .
   @param ELLIPSE *ell pointer to ellipse.
   @return the center x-coordinate of the given ellipse, some negative value if ERROR.
 */
float ellipseGetCenterX(ELLIPSE *ell){

  if(ELLIPSE_VERBOSE) printf("ELLIPSE:ellipseGetCenterX(). \n");
  
  if(ell->status==ELLIPSE_STATUS_UNINITIALIZED) return(-1);

  return(ell->center[0]);
}

/**
   Returns the center y-coordinate of the ellipse.
   @precondition ell is initialized. 
   @postcondition .
   @param ELLIPSE *ell pointer to ellipse.
   @return the center y-coordinate of the given ellipse, some negative value if ERROR.
 */
float ellipseGetCenterY(ELLIPSE *ell){

  if(ELLIPSE_VERBOSE) printf("ELLIPSE:ellipseGetCenterY(). \n");
  
  if(ell->status==ELLIPSE_STATUS_UNINITIALIZED) return(-1);

  return(ell->center[1]);
}

/**
   Returns the inclination of the ellipse.
   @precondition ell is initialized. 
   @postcondition .
   @param ELLIPSE *ell pointer to ellipse.
   @return the inclination of the given ellipse, some negative value if ERROR.
 */
float ellipseGetInclination(ELLIPSE *ell){

  if(ELLIPSE_VERBOSE) printf("ELLIPSE:ellipseGetInclination(). \n");
  
  if(ell->status==ELLIPSE_STATUS_UNINITIALIZED) return(-1);

  return(ell->inclination);
}

/**
   Returns the size of the image plane on which the ellipse is drawn.
   @precondition ell is initialized. 
   @postcondition .
   @param ELLIPSE *ell pointer to ellipse.
   @return the size of the image plane of the given ellipse, some negative value if ERROR.
 */
int ellipseGetImgSize(ELLIPSE *ell){

  if(ELLIPSE_VERBOSE) printf("ELLIPSE:ellipseGetImgSize(). \n");
  
  if(ell->status==ELLIPSE_STATUS_UNINITIALIZED) return(-1);

  return(ell->imageDim);
}

/**
   Returns the value of the pixels inside the ellipse.
   @param ELLIPSE *ell pointer to ellipse.
   @return the value of the pixels inside the ellipse.
 */
int ellipseGetValue(ELLIPSE *ell){

if(ELLIPSE_VERBOSE) printf("ELLIPSE:ellipseGetValue(). \n");
  
  if(ell->status==ELLIPSE_STATUS_UNINITIALIZED) return(-1);

  return(ell->value);
}

/**
   Returns the ellipse array of the given ellipse. Ellipse array contains n x n items, 
   single item is one if it is inside the ellipse and zero otherwise. 
   Coordinates on a two dimensional plane are numbered from upper left corner.
   @precondition ell is initialized.
   @postcondition .
   @param ELLIPSE *ell pointer to ellipse.
   @return the ellipse array of the given ellipse.
*/
int** ellipseGetArray(ELLIPSE *ell){
  
  if(ELLIPSE_VERBOSE) printf("ELLIPSE:ellipseGetArray(). \n");
  
  if(ell->status==ELLIPSE_STATUS_UNINITIALIZED) return((int**)-1);

  return(ell->ellipseptr);
}
 
/*Testing procedures for ellipse.*/

/**
   Tests whether the given pixel is inside the given ellipse or not.
   @precondition ell is initialized && 0<=row<=imgDim-1 && 0<=col<=imgDim-1
   @postcondition .
   @param ELLIPSE *ell ellipse on which the testing is to be done.
   @param int row row coordinate of a pixel.
   @param int col column coordinate of a pixel.
   @return one if the given pixel is inside the given ellipse zero otherwise, some negative value if ERROR.
*/
int ellipseIsInside(ELLIPSE *ell, int row, int col){

  //no checking on parameters to accelerate use inside a loop
  return(ell->ellipseptr[row][col]);
}
