/**@name ELLIPSE data structure
*
*  Filename:      ellipse.h  (c) 2003 Turku PET Centre
*
*  First version: 10.07.2003
*
*  Description:   Header file for ellipse.c. Defines an ellipse 
*                 on an n x n image grid.
*
*  Updated:       20.4.2004 version 1.0 (Jarkko Johansson).
*                 
@version 1.0 
@author Jarkko Johansson
*/
//@{
//@}
#ifndef _ELLIPSE_H
#define _ELLIPSE_H
/**********************************************************/
/*Definitions for ellipse status*/
///Status uninitialized, value 0.
#define ELLIPSE_STATUS_UNINITIALIZED 0
///Status initialized, value 1.
#define ELLIPSE_STATUS_INITIALIZED 1
///Status occupied, value 2.
#define ELLIPSE_STATUS_OCCUPIED 2
///Status error, value 3.
#define ELLIPSE_STATUS_ERROR 3
/**********************************************************/ 
///Drive in test mode if not 0.
int ELLIPSE_TEST;
///Drive in verbose mode if not 0.
int ELLIPSE_VERBOSE;
/**********************************************************/ 

/**
   Ellipse on two dimensional plane. Ellipse is defined on 
   an n x n image grid by the center coordinate and the length 
   of the semiaxis.
 */
typedef struct{

  ///ellipse status.
  char status;

  /*Ellipse*/

  /**Semiaxis of the ellipse.
    semiaxis[0]=major semiaxe, semiaxis[1]=minor semiaxe.
  */
  float semiaxis[2]; 
  
  /**Center of the ellipse.
     center[0]=x-coordinate, center[1]=y-coordinate in cartesian coordinate system.
  */
  float center[2];

  ///Inclination (degrees).
  float inclination;

  ///Size of the image plane on which the ellipse is defined.
  int imageDim;
  
  /**Ellipse array containing n x n entries. Single entry is one
     if it is inside the ellipse and zero otherwise. Coordinates on a
     two dimensional plane are numbered from left to right and from 
     top to bottom (not cartesian). 
  */
  int **ellipseptr;

  ///Value inside the ellipse.
  float value;

} ELLIPSE;

/*********************************************************
 *
 * Function definitions.
 *
 *********************************************************/

/* Initialization and memory handling of ellipse data.*/

/**
   Initializes ELLIPSE datatype for use. 
   To be used before any use of ELLIPSE type variables.
 @postcondition ellipse is initialized.
 @param ELLIPSE *ell pointer to ellipse to be initialized. 
 */
void ellipseInit(ELLIPSE *ell);

/**
   Frees the memory allocated for ellipse.
   All data is cleared.
   @postcondition ellipse is emptied.
   @param ELLIPSE ell pointer to ellipse to be emptied.
 */
void ellipseEmpty(ELLIPSE *ell);

/**
   Prints the information of the ellipse to the screen.
   @param ELLIPSE *ell pointer to ellipse to be printed.
 */
void ellipseInfo(ELLIPSE *ell);

/**
   Allocates memory for ELLIPSE data. Normally used only in SET-functions.   
   @precondition *ell is initialized and imgDim is positive 
   @postcondition memory is allocated for ELLIPSE structure.
   @param ELLIPSE *ell pointer to ellipse for which the allocation is done.
   @param int imgDim size of the image plane on which the ellipse is to be defined.
   @return 0 if ok.
 */
int ellipseAllocate(ELLIPSE *ell, int imgDim);

/* GET and SET procedures for ELLIPSE datatype. USE ONLY THESE FOR SETTING AND GETTING. */

/**
   Sets the ellipse according to given coordinates and image dimension.
   Notice that the origin is in the middle.
   @precondition *ell is initialized.
   @precondition imgDim is positive.
   @precondition semis[i] are positive.
   @precondition -imgDim/2<=cent[i]<=imgDim/2.
   @postcondition ellipse is set.
   @param ELLIPSE *ell pointer to ellipse for which the setting is to be done.
   @param int imgDim size of the image plane.
   @param float *semis major and minor semiaxis of the ellipse.
   @param float *cent center of the ellipse.
   @param float incli inclination of the ellipse.
   @param float value value inside the ellipse.
   @return 0 if ok.
*/
int ellipseSetFromParams(ELLIPSE *ell, int imgDim, float *semis, float *cent, float incli, float val);

/**
   Reads one ellipse from the given file to the given ELLIPSE structure.
   A coordinate file contains the parameters of the ellipses in one line
   in the following order:
   Coordinate 1:  v    the additive intensity value of the ellipse
   Coordinate 2:  a    the length of the horizontal semi-axis of the ellipse
   Coordinate 3:  b    the length of the vertical semi-axis of the ellipse
   Coordinate 4:  x    the x-coordinate of the center of the ellipse
   Coordinate 5:  y    the y-coordinate of the center of the ellipse
   Coordinate 6:  p    the angle (in degrees) between the horizontal semi-axis
                       of the ellipse and the x-axis of the image        
   Coordinate 7:  d    the image dimension
   @precondition *fp is a pointer to open file containing ellipse(s) in correct format.
   @precondition *ell is initialized 
   @postcondition an ellipse is read from the file fname. 
   @param FILE *fp a pointer to open file containing ellipse(s).
   @param ELLIPSE *ell pointer to ELLIPSE structure where the read ellipse is to be set.
   @return 0 if ok 1 if there were no more ellipses.
*/
int ellipseReadEllipse(FILE *fp, ELLIPSE *ell);

/**
   Adds the given ellipse to the file called fname.
   @precondition *ell is initialized.
   @postcondition ellipse is saved in the file called fname. 
   @param ELLIPSE *ell pointer to the ellipse to be saved.
   @param FILE *fp open file for saving.
   @return 0 if OK.
*/
int ellipseSaveEllipse(ELLIPSE *ell, FILE *fp);

/**
   Returns the major semiaxe of the ellipse.
   @param ELLIPSE *ell pointer to ellipse.
   @return the major semiaxe of the given ellipse, some negative value if ERROR.
 */
float ellipseGetMajor(ELLIPSE *ell);

/**
   Returns the minor semiaxe of the ellipse.
   @param ELLIPSE *ell pointer to ellipse.
   @return the minor semiaxe of the given ellipse, some negative value if ERROR.
 */
float ellipseGetMinor(ELLIPSE *ell);

/**
   Returns the center x-coordinate of the ellipse.
   @param ELLIPSE *ell pointer to ellipse.
   @return the center x-coordinate of the given ellipse, some negative value if ERROR.
 */
float ellipseGetCenterX(ELLIPSE *ell);

/**
   Returns the center y-coordinate of the ellipse.
   @param ELLIPSE *ell pointer to ellipse.
   @return the center y-coordinate of the given ellipse, some negative value if ERROR.
 */
float ellipseGetCenterY(ELLIPSE *ell);

/**
   Returns the inclination of the ellipse.
   @param ELLIPSE *ell pointer to ellipse.
   @return the inclination of the given ellipse, some negative value if ERROR.
*/
float ellipseGetInclination(ELLIPSE *ell);

/**
   Returns the size of the image plane on which the ellipse is drawn.
   @param ELLIPSE *ell pointer to ellipse.
   @return the size of the image plane of the given ellipse, some negative value if ERROR.
 */
int ellipseGetImgSize(ELLIPSE *ell);

/**
   Returns the value of the pixels inside the ellipse.
   @param ELLIPSE *ell pointer to ellipse.
   @return the value of the pixels inside the ellipse.
 */
int ellipseGetValue(ELLIPSE *ell);

/**
   Returns the ellipse array of the given ellipse. Ellipse array contains n x n entries, 
   single entry is one if it is inside the ellipse and zero otherwise. 
   Coordinates on a two dimensional plane are numbered from upper left corner.
   @param ELLIPSE *ell pointer to ellipse.
   @return the ellipse array of the given ellipse.
*/
int** ellipseGetArray(ELLIPSE *ell);

 
/*Testing procedures for ellipse.*/

/**
   Checks whether the given pixel is inside the given ellipse or not.
   @precondition ell is initialized and 0<=row,col<=imgDim-1.
   @param ELLIPSE *ell ellipse on which the testing is to be done.
   @param int row row coordinate of a pixel in an image grid (not cartesian).
   @param int col column coordinate of a pixel in an image grid (not cartesian).
   @return one if the given pixel is inside the given ellipse zero otherwise, some negative value if ERROR.
*/
int ellipseIsInside(ELLIPSE *ell, int row, int col);

#endif
