/**@name PRMAT data structure.

   Filename:      prmat.h  (c) 2003-2004 Turku PET Centre
 
   First version: 04.06.2003
 
   Description:   PRMAT is the structure for storing and processing
                  the projection (system) matrix utilized in PET
		  image reconstruction. 
	       
   Updated:       20.4.2004 version 1.0 (Jarkko Johansson).

@see ellipse.h radon.h                 
@version 1.0 
@author Jarkko Johansson
*/
//@{ 
//@}
#ifndef _PRMAT_H
#define _PRMAT_H
/**********************************************************/
#include "ellipse.h"
/**********************************************************/
/*Definitions for prmat status*/
///Structure status uninitialized. Value 0.
#define PRMAT_STATUS_UNINITIALIZED 0
///Stucture status initialized. Value 1.
#define PRMAT_STATUS_INITIALIZED 1
///Base set of projection matrix status initialized and set. Value 2.
#define PRMAT_STATUS_BS_OCCUPIED 2
///Look up table status initialized and set. Value 3.
#define PRMAT_STATUS_LU_OCCUPIED 3
///The whole projection matrix status initialized and set. Value 4.
#define PRMAT_STATUS_PR_OCCUPIED 4
///Stucture status error. Value 5.
#define PRMAT_STATUS_ERROR 5
/*Definitions for prmat type*/
///Projection matrix is based on scanner dimensions 192*256 (ECAT). Value 0.
#define PRMAT_TYPE_ECAT931 0 
///Projection matrix is based on scanner dimensions 281*336 (GE). Value 1.
#define PRMAT_TYPE_GE 1 
///Projection matrix is based on unknown scanner geometrics. Value 2.
#define PRMAT_TYPE_NA 2 
/*Definitions for discretization model*/
///Discretization model in discrete Radon transform is "0/1". Value 0.
#define PRMAT_DMODE_01 0
///Discretization model in discrete Radon transform is "length of intersection". Value 1.
#define PRMAT_DMODE_LOI 1
///Discretization model in discrete Radon transform is "exact area". Value 2.
#define PRMAT_DMODE_EA 2 
///Discretization model in discrete Radon transform is "linear interpolation". Value 3.
#define PRMAT_DMODE_LI 3 
///Discretization model in discrete Radon transform is "nearest neighbour". Value 4.
#define PRMAT_DMODE_NN 4 
/**********************************************************/ 
///If not 0 drive in test mode.
int PRMAT_TEST;
///If not 0 drive in verbose mode.
int PRMAT_VERBOSE;
/**********************************************************/
typedef struct{

  ///Prmat status.
  char status;

  ///Scanner information on the prmat. 0=ECAT931 1=GE Advance.
  char type;

  ///Scanner geometrics, field viewNr.
  int viewNr;

  ///Scanner geometrics, field binNr.
  int binNr;

  ///Scanner geometrics, field imgDim.
  int imgDim;

  ///Discretisation model utilised.
  int mode; 

  /**FOV of the projection matrix.
    prmatfov[0]=major semiaxe, prmatfov[1]=minor semiaxe.
  */
  int *prmatfov; 
  
  ///Minimal factor value in the projection matrix.
  float min;

  ///Maximal factor value in the projection matrix.
  float max;
  
  ///The sum of all factors in the projection matrix
  float factor_sum;

  ///Scaling factor for factors (notice that factors are stored in integers).
  float scaling_factor;

  /*PRMAT LOOK-UP TABLE: for restoring lines of response intersecting a pixel.*/

  ///Number of pixels inside the fov. i.e. number of rows in the look-up table.
  int nrp;

  /**Coordinates of a pixel, number of lines of response intersecting the pixel 
     and the coordinates of coincidence lines (nrp x (lines + 2) array).
     lines[row][0] = pixel coordinates , lines[row][1] = lines hitting a pixel.
  */
  unsigned int **lines;

  ///Hidden pointer for the actual data.
  unsigned int **_linesdata;

  /*PRMAT DATA: for restoring lines of response in the projection matrix. */
  
  ///Dimension of rows (lines of response) in the projection matrix.
  int dimr;

  ///Square sums of factors in each row in the projection matrix.
  float *factor_sqr_sum;

  ///Number of pixels hit by a line for every line.
  int *dime;

  /**Coordinates and factors of pixels hit by a line of response.
    fact[row][pix][0]=x-coordinate && fact[row][pix][1]=y-coordinate && fact[row][pix][2]=factor.
   */
  unsigned short int ***fact; 
  
  ///Hidden pointer for the actual data. 
  unsigned short int ***_factdata;

  
}PRMAT;

/*********************************************************
 *
 * Function definitions.
 *
 *********************************************************/

/* Initialization and memory handling for prmat data      */

/**
   Initializes PRMAT datatype for use.
   To be utilised before any use of PRMAT type variables.
   @param PRMAT *mat pointer to projection matrix to be initialized. 
*/
void prmatInit(PRMAT *mat);

/**
   Frees the memory allocated for mat.
   All data is cleared.
   @param PRMAT *mat pointer to projection matrix to be emptied.
*/
void prmatEmpty(PRMAT *mat);

/**
   Allocates memory for PRMAT data. Normally used only in SET-functions.   
   Allocates memory for look-up table if set = 0 and for projection matrix 
   otherwise.
   @precondition *mat is initialized.
   @precondition rows is positive.
   @precondition *coords list contains as many positive integers as there are rows.
   @postcondition memory is allocated for PRMAT structure mat.
   @param PRMAT *mat pointer to projection matrix for which the allocation is to be done.
   @param int set tells for which part of the structure allocation is to be done.
   @param int rows number of rows in a matrix.
   @param int *coords list of number of entries in each row.
   @return 0 if ok.
 */
int prmatAllocate(PRMAT *mat, int set,int rows, int *coords);

/* GET procedures for PRMAT datatype. USE ONLY THESE FOR GETTING. */

/* GET FUNCTIONS FOR THE PARAMETERS */

/**
   Returns the number of views (angles) in the geometrics.
   @precondition mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return the number of views in the geometrics.
*/
int prmatGetNV(PRMAT *mat);

/**
   Returns the number of bins (distances) in the geometrics.
   @precondition mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return the number of bins in the geometrics.
*/
int prmatGetNB(PRMAT *mat);

/**
   Returns the image dimension in the geometrics.
   @precondition mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return the image dimension in the geometrics.
*/
int prmatGetID(PRMAT *mat);

/* GET FUNCTIONS FOR ACCESSING LOOK-UP TABLE */

/**
   Returns number of rows (pixels) in the look-up table.
   @precondition mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return number of pixels in look-up table, some negative value if ERROR.
*/
int prmatGetPIX(PRMAT *mat);

/**
   Returns coordinates of a pixel in place 'row' in the look-up table.
   @precondition 0<=row<=prmatGetPIX(*mat). 
   @precondition mat is initialized.
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @param int row index of the pixel whose coordinates are to be returned.
   @return coordinates of a pixel in the look-up table, some negative value if ERROR.
*/
int prmatGetPixCoord(PRMAT *mat,int row);

/**
   Returns the number of lines of response (rays) intersecting the pixel in place 'row'.
   @precondition 0<=row<=prmatGetPIX(*mat). 
   @precondition mat is initialized.
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @param int row index of the pixel for which the number of lors is returned.
   @return number of lines of response intersecting the pixel, some negative value if ERROR.
*/
int prmatGetRays(PRMAT *mat,int row);

/**
   Returns coordinates of a line of response in place (row,ind) in the look-up table.
   @precondition 0<=row<=prmatGetPIX(*mat). 
   @precondition 0<=ind<=prmatGetRays(*mat,row).
   @precondition mat is initialized.
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @param int row the row index.
   @param int ind the (non-zero) lor index.
   @return int coordinates (view*binNr + bin) of the referred line of response, some negative value if ERROR.
 */
int prmatGetBinView(PRMAT *mat, int row, int ind);

/* GET FUNCTIONS FOR ACCESSING PROJECTION MATRIX BY A LINE OF RESPONSE.*/

/**
   Returns number of rows (lines of response) in the given projection matrix.
   @precondition mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to the projection matrix.
   @return number of rows in the given projection matrix, some negative value if ERROR.
*/
int prmatGetRows(PRMAT *mat);

/**
   Returns number of pixels intersected by the given line of response.
   @precondition 0<=row<=prmatGetRows(*mat).
   @precondition and mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to the projection matrix.
   @param int row index of the line of response for which the number of pixels is to be returned.
   @return number of pixels intersected by given line of response, some negative value if ERROR.
*/
int prmatGetPixels(PRMAT *mat, int row);

/**
   Returns coordinates of a pixel in place (row,pix) in the given projection matrix.
   @precondition 0<=row<=prmatGetRows(*mat).
   @precondition 0<=pix<=prmatGetPixels(row,*mat).
   @precondition mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @param int row the row index.
   @param int pix the (non-zero) pixel index.
   @return coordinates of the referred pixel, some negative value if ERROR.
*/
int prmatGetCoord(PRMAT *mat, int row, int pix);

/**
   Returns the x-coordinate of a pixel in place (row,pix) in the given projection matrix.
   @precondition 0<=row<=prmatGetRows(*mat).
   @precondition 0<=pix<=prmatGetPixels(row,*mat).
   @precondition mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @param int row the row index.
   @param int pix the (non-zero) pixel index.
   @return coordinate of the referred pixel, some negative value if ERROR.
*/
int prmatGetXCoord(PRMAT *mat, int row, int pix);

/**
   Returns the y-coordinate of a pixel in place (row,pix) in the given projection matrix.
   @precondition 0<=row<=prmatGetRows(*mat).
   @precondition 0<=pix<=prmatGetPixels(row,*mat).
   @precondition mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @param int row the row index.
   @param int pix the (non-zero) pixel index.
   @return coordinate of the referred pixel, some negative value if ERROR.
*/
int prmatGetYCoord(PRMAT *mat, int row, int pix);

/**
   Returns factor (weighting coefficient) for a pixel in place (row,pix) in given projection matrix.
   @precondition 0<=row<=prmatGetRows(*mat).
   @precondition 0<=pix<=prmatGetPixels(row,*mat).
   @precondition mat is initialized.
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @param int row the row index.
   @param int pix the (non-zero) pixel index.
   @return factor for the referred pixel, some negative value if ERROR.
*/
float prmatGetFactor(PRMAT *mat, int row, int pix);

/**
   Returns major semiaxe of the FOV in given projection matrix.
   @precondition mat is initialized.
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return major semiaxe of the FOV in given projection matrix
*/
float prmatGetMajor(PRMAT *mat);

/**
   Returns minor semiaxe of the FOV in given projection matrix.
   @precondition mat is initialized.
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return minor semiaxe of the FOV in given projection matrix
*/
float prmatGetMinor(PRMAT *mat);

/**
   Returns minimal value in given projection matrix.
   @precondition mat is initialized.
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return minimal value in given projection matrix.
*/
float prmatGetMin(PRMAT *mat);

/**
   Returns greatest value in given projection matrix.
   @precondition mat is initialized.
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return maximal value in given projection matrix.
*/
float prmatGetMax(PRMAT *mat);

/**
   Returns sum of all factors in given projection matrix.
   For initializing image matrix.
   @precondition mat is initialized.
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return sum of all factors in given projection matrix.
*/
float prmatGetFactorSum(PRMAT *mat);

/**
   Returns square sum of all factors in given projection matrix in given row.
   @precondition mat is initialized. 
   @precondition 0<=row<=prmatGetRows(*mat).
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @param int row row index for accessing factors.
   @return square sum of all factors in given projection matrix in given row.
*/
float prmatGetFactorSqrSum(PRMAT *mat, int row);

/*ARCHIVING METHODS*/

/**
   Reads one projection matrix from the given file to the given PRMAT structure mat.
   @precondition mat is initialized.
   @postcondition a projection matrix is read from the file fname. 
   @param char *fname file name to read.
   @param PRMAT *mat pointer to the projection matrix datatype.
   @return 0 if OK.
*/
int prmatReadMatrix(char *fname, PRMAT *mat);

/**
   Adds the given projection matrix to the file called prTDXXYY.prm. 
   Where T is the type of the scanner 
   (E=ECAT931 and G=GE Advance), D is the type of discretisation
   (0='0/1', 1='length of intersection' and 2='exact area'),
   XX is the major semiaxe of the FOV, and YY the minor semiaxe.
   If file already exists it is NOT created or rewritten.
   @precondition mat is initialized (contains a projection matrix). 
   @postcondition mat is saved in the file  prTDXXYY.prm. 
   @param PRMAT *mat pointer to the projection matrix to be saved.
   @return 0 if OK.
*/
int prmatSaveMatrix(PRMAT *mat);

#endif

