/**@name RADON data structure.
*
*  Filename: radon.h  (c) 2003-2004 Turku PET Centre
*
*  First version: 11.09.2003
*
*  Description: Header file of radon.c.  
*  Radon data structure contains the parameters defining a Radon transform. 
*  Functions for transforming to and from the Radon domain are introduced in this file.
*  See documentation for more detailed information.
*  
*  Updated: 20.4.2004 version 1.0 (Jarkko Johansson).
*
*
@see ellipse prmat
@author Jarkko Johansson
@version 1.0
*/
//@{
//@}

#ifndef RADON_H
#define RADON_H
/**********************************************************/
/**********************************************************/
#include "ellipse.h"
#include "prmat.h"
/**********************************************************/
/**********************************************************/
/*Definitions for radon transform status*/
///Status uninitialized.
#define RADON_STATUS_UNINITIALIZED 0
///Status initialized.
#define RADON_STATUS_INITIALIZED 1
///Status occupied.
#define RADON_STATUS_OCCUPIED 2
///Status error.
#define RADON_STATUS_ERROR 3
/**********************************************************/ 
/// Drive in test mode if not 0.
int RADON_TEST;
/// Drive in verbose mode if not 0.
int RADON_VERBOSE;
/**********************************************************/ 

typedef struct{

  ///Transform status. See definitions.
  char status;

  /*Properties of this Radon transform*/

  ///Discretization model.
  char mode;

  ///Dimension of the image plane.
  int imgDim;
  
  ///Number of views (angles).
  int viewNr;

  ///Number of bins (distances).
  int binNr;

  ///Half of the bins.
  int half;

  ///Bin going through the origin.
  int centerBin;

  ///Sample distance. i.e. distance between two lines of response.
  float sampleDist;

  ///Sine table. i.e. the tabulated values of the sine function for this transform.
  float *sines;

}RADON;

/*********************************************************
 *
 * Function definitions.
 *
 *********************************************************/
/* Initialization and memory handling for radon transform data*/

/**
   Frees the memory allocated for radon transform.
   All data is cleared.
   @precondition .
   @postcondition radon transform is emptied.
   @param RADON *radtra pointer to transform data to be emptied
*/
void radonEmpty(RADON *radtra);

/**
   Sets the data for the 2-D Radon transform.  
   @precondition imgDim, viewNr and binNr > 0.
   @postcondition Parameters for the radon transform are set. 
   @param RADON *radtra radon transform for which the paramaters are to be set.
   @param int mode discretisation mode.
   @param int imgDim image dimension.
   @param int viewNr Number of views (angles).
   @param int binNr Number of bins (distances).
   @return 0 if ok
*/
int radonSet(RADON *radtra,int mode,int imgDim,int viewNr,int binNr);

/*Get functions for Radon data*/

/** 
    Returns the discretization model of this radon transform.
    @param RADON *radtra radon transform for which the mode is to be returned.
    @return int mode.
*/
int radonGetMO(RADON *radtra);

/** 
    Returns the image dimension in this radon transform.
    @param RADON *radtra radon transform for which the image dimension is to be returned.
    @return int image dimension.
*/
int radonGetID(RADON *radtra);

/**
   Returns the number of views in this radon transform.
   @param RADON *radtra radon transform for which the number of views is to be returned.
   @return int number of views.
*/
int radonGetNV(RADON *radtra);

/**
   Returns the number of bins in this radon transform.
   @param RADON *radtra radon transform for which the number of bins is to be returned.
   @return int number of bins.
*/
int radonGetNB(RADON *radtra);

/**
   Returns the sample distance in this radon transform.
   @param RADON *radtra radon transform for which the sample distance is to be returned.
   @return float sample distance.
*/
float radonGetSD(RADON *radtra);

/**
   Returns the half index of the bins in this radon transform.
   @param RADON *radtra radon transform for which the half index is to be returned.
   @return int half index of the bins.
*/
int radonGetHI(RADON *radtra);

/**
   Returns the center bin in this radon transform.
   @param RADON *radtra radon transform for which the center bin is to be returned.
   @return int index of the center bin.
*/
int radonGetCB(RADON *radtra);

/**
   Returns the sine for given angle (index).
   @param RADON *radtra radon transform for which the sine is to be returned.
   @param int nr index of the angle to be returned (angle=nr*pi/radonGetNB(RADON)).
   @return float sine for given angle.
*/
float radonGetSin(RADON *radtra, int nr);


////////// THE ACTUAL TRANSFORM FUNCTIONS /////////////////////////////////////////

/**
   Transforms the given intensity image in spatial domain to Radon domain.
   Parameters of the discrete Radon transform are stored in the RADON object.
   Transform is calculated only in those angles belonging into given subset.
   Subset contains angles starting from the index 'set' with spacing 'setNr'.
   Discretisation model utilised in this function is '0/1' or 'length of intersection'
   according to the given RADON object.
   @precondition imgdata contains pixel values for every pixel in n x n image grid.
   @postcondition imgdata is mapped into the projection space.
   @param RADON *radtra contains an initialized radon transform object.
   @param int set tells which set is to be utilized; 0 < set < setNr-1.
   @param int setNr number of subsets (spacing between indices). 
   @param float *imgdata contains pixel values for every pixel in n x n image grid.
   @param float *scndata v x b vector for storing the projection.
   @return int 0 if ok
*/
int radonFwdTransform(RADON *radtra, int set, int setNr, float *imgdata, float *scndata);

/** 
    Transforms the given intensity image in spatial domain to Radon domain.
    Same as 'radonFwdTransform()' but discretisation model in this function is 
    'exact area'.
    @see radonFwdTransform().
*/
int radonFwdTransformEA(RADON *radtra, int set, int setNr, float *imgdata, float *scndata);

/** 
    Transforms the given intensity image in spatial domain to Radon domain.
    Same as 'radonFwdTransform()' but discretisation model in this function is 
    'linear interpolation' or 'nearest neighbour interpolation' according to the
    given Radon transform object.
    NOTE: first written by Sakari Alenius 1998.
    @see radonFwdTransform().
*/
int radonFwdTransformSA(RADON *radtra, int set, int setNr, float *imgdata, float *scndata);

/** 
   Transforms the given intensity image in spatial domain to Radon domain.
   Transform is calculated by multiplying with the given projection matrix from the left.
   Transform is calculated only in those angles belonging into given subset.
   Subset contains angles starting from the index 'set' with spacing 'setNr'.
   Discretisation model utilised in this function is '0/1', 'length of intersection' or
   'exact area' according to the given projection matrix.
   @precondition imgdata contains pixel values for every pixel in n x n image grid.
   @postcondition imgdata is mapped into the projection space.
   @param PRMAT *mat contains an initialised projection matrix.
   @param int set tells which set is to be utilized; 0 < set < setNr-1.
   @param int setNr number of subsets (spacing between indices).
   @param float *imgdata contains pixel values for every pixel in dim*dim image grid.
   @param float *scndata viewNr*binNr vector for storing the projection.
   @return int 0 if ok
*/
int radonFwdTransformPRM(PRMAT *mat, int set, int setNr, float *imgdata, float *scndata);

/** 
   Transforms the given sinogram in Radon domain to spatial domain.
   Parameters of the discrete Radon transform are stored in the RADON object.
   Transform is calculated only in those angles belonging into given subset.
   Subset contains angles starting from the index 'set' with spacing 'setNr'.
   Discretisation model utilised in this function is '0/1' or 'length of intersection'
   according to the given RADON object.
   @precondition scndata contains the projections in v x b lines of response.
   @postcondition scndata is mapped into the cartesian space.
   @param RADON *radtra contains an initialized radon transform object.
   @param int   set tells which set is to be utilized; 0 < set < setNr-1.
   @param int   setNr number of subsets .
   @param float *scndata v x b vector contains the projections.
   @param float *imgdata n x n vector for storing the back-projection.
   @return int 0 if ok
*/
int radonBackTransform(RADON *radtra, int set, int setNr, float *scndata, float *imgdata);

/**
   Transforms the given sinogram in Radon domain to spatial domain.
   Same as 'radonBackTransform()' but discretisation model in this function is 
   'exact area'.
   @see radonBackTransform().
*/
int radonBackTransformEA(RADON *radtra, int set, int setNr, float *scndata, float *imgdata);

/** 
    Transforms the given sinogram in Radon domain to spatial domain.
    Same as 'radonBackTransform()' but discretisation model in this function is 
    'linear interpolation' or 'nearest neighbour interpolation' according to the
    given Radon transform object.
    NOTE: first written by Sakari Alenius 1998.
    @see radonBackTransform().
*/
int radonBackTransformSA(RADON *radtra, int set, int setNr, float *imgdata, float *scndata);

/**
   Transforms the given sinogram in Radon domain to spatial domain.
   Transform is calculated by multiplying with the transpose of the 
   given projection matrix from the right.
   Transform is calculated only in those angles belonging into given subset.
   Subset contains angles starting from the index 'set' with spacing 'setNr'.
   Discretisation model utilised in this function is '0/1', 'length of intersection' or
   'exact area' according to the given projection matrix. 
   @precondition scndata contains the projections in v x b lines of response.
   @postcondition scndata is mapped into the cartesian space.
   @param RADON *radtra contains an initialized radon transform object.
   @param int   set tells which set is to be utilized; 0 < set < setNr-1.
   @param int   setNr number of subsets .
   @param float *scndata v x b vector contains the projections.
   @param float *imgdata n x n vector for storing the back-projection.
   @return int 0 if ok
*/
int radonBackTransformPRM(PRMAT *mat, int set, int setNr, float *scndata, float *imgdata);

////////////// FUNCTIONS FOR SETTING THE FACTORS IN THE PRMAT DATA STUCTURE ////////////////////////

/**
   Sets the coordinates and factors for intersected pixels. Setting is done according to
   given special Radon transform operator for every coincidence line in the BASE set.
   Base set includes coincidence lines in range [0,pi/4].
   @precondition radtra is a radon transform operator
   @precondition elli defines a field of view 
   @precondition mat is initialized.
   @postcondition coordinates and factors, for pixels contributing to coincidence lines in the base set are set.
   @param RADON *radtra special Radon transform operator.
   @param ELLIPSE *elli field of view.
   @param PRMAT *mat pointer to the datastructure where coordinates and values are to be stored.
   @return 0 if ok.
*/
int radonSetBases(RADON *radtra,ELLIPSE *elli, PRMAT *mat);

/**
   Sets the coordinates and factors for intersected pixels.
   Same as 'radonSetBases()' but discretisation model in this function is 
   'exact area'. 
   @see radonSetBases().
*/
int radonSetBasesEA(RADON *radtra,ELLIPSE *elli, PRMAT *mat);

/**
   Sets look-up table filled by coordinates of lines of response intersecting pixels. 
   Setting is done according to given special Radon transform operator, for pixels inside the FOV.
   @precondition radtra is a radon transform operator.
   @precondition elli defines a field of view.
   @precondition mat is initialized.
   @postcondition coordinates of lines response intersecting a pixel are set in a table.
   @param RADON *radtra special Radon transform operator.
   @param ELLIPSE *elli field of view.
   @param PRMAT *mat pointer to the datastructure where coordinates and values are to be stored.
   @return 0 if ok.
*/
int radonSetLUT(RADON *radtra, ELLIPSE *elli, PRMAT *mat);

/**
   Sets the coordinates and factors for intersected pixels. Setting is done according to
   given special Radon transform operator for every line of response.
   @precondition radtra is a radon transform operator.
   @precondition elli defines a field of view.
   @precondition mat is initialized.
   @postcondition coordinates and factors for pixels contributing to all lines of response are set. 
   @param RADON *radtra special Radon transform operator.
   @param ELLIPSE *elli field of view.
   @param PRMAT *mat pointer to the datastructure where coordinates and values are stored for base lines.
   @return 0 if ok.
*/
int radonSetLORS(RADON *radtra,ELLIPSE *elli, PRMAT *mat);

#endif
