/***********************************************************
*
*
*  Filename:      prmat.c  (c) 2003 Turku PET Centre
*
*  First version: 04.06.2003
*
*  Programmer:    Jarkko Johansson
*
*  Description:   PRMAT is the structure for storing and processing
*                 the projection (system) matrix utilized in PET
*		  image reconstruction.
*                 See closer definition from documentation. 
*                 
*  Updated:       20.4.2004 version 1.0 Jarkko Johansson
*
**********************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
//#include <ctype.h>
#include <string.h>
/*****************************************************************************/
#include "include/prmat.h"
/*****************************************************************************/

/* Initialization and memory handling for prmat data      */

/**
   Initializes PRMAT datatype for use.
   To be utilised before any use of PRMAT type variables.
   @param PRMAT *mat pointer to projection matrix to be initialized. 
*/
void prmatInit(PRMAT *mat){

  if(PRMAT_VERBOSE) printf("PRMAT:prmatInit() \n"); 
  
  /*init the information of projection matrix*/
  //buffer mat to contain zero
  memset(mat, 0, sizeof(PRMAT)); 
  mat->type=PRMAT_TYPE_NA;
  mat->viewNr=0;
  mat->binNr=0;
  mat->mode=PRMAT_DMODE_01;
  mat->prmatfov= (int*)NULL;
  mat->max=mat->min=0.0;
  mat->scaling_factor=1.0;
  mat->factor_sum=0.0;
  mat->factor_sqr_sum=(float*)NULL;

  /*init look-up table*/
  mat->nrp=0;
  mat->lines=mat->_linesdata=(unsigned int**)NULL; //data array

  /*init projection data*/
  mat->dimr=0;
  mat->dime=(int*)NULL;
  mat->fact=mat->_factdata=(unsigned short int***)NULL; //data array
  
  mat->status=PRMAT_STATUS_INITIALIZED;
}//END OF INIT

/**
   Frees the memory allocated for mat.
   All data is cleared.
   @param PRMAT *mat pointer to projection matrix to be emptied.
*/
void prmatEmpty(PRMAT *mat){

  int ir, ip;

  if(PRMAT_VERBOSE) printf("PRMAT:prmatEmpty() \n");
  
  //if mat is allready empty
  if(mat->status==PRMAT_STATUS_INITIALIZED) return; 
  
  //if look-up table is occupied
  if(mat->status==PRMAT_STATUS_LU_OCCUPIED){ 
    /*free the occupied memory*/
    for(ir=0; ir<mat->nrp; ir++){ //every row
      free((unsigned int*)mat->_linesdata[ir]);
    }
    if(mat->nrp > 0) free(mat->_linesdata);
  }

  //if projection matrix is occupied
  if(mat->status==PRMAT_STATUS_PR_OCCUPIED || mat->status==PRMAT_STATUS_BS_OCCUPIED ){ 
    free((float*)mat->factor_sqr_sum);
    for(ir=0; ir<mat->dimr; ir++){ //every row
      for(ip=0; ip<mat->dime[ir]; ip++){ //every factor
	free((unsigned short int*)mat->_factdata[ir][ip]);
      }
    }

    free((int*)mat->dime);
  
    if(mat->dimr>0) free(mat->_factdata); 
    mat->dimr=0; 
  }

  return;
}//END OF EMPTY

/**
   Allocates memory for PRMAT data. Normally used only in SET-functions.   
   Allocates memory for look-up table if set = 0 and for projection matrix 
   otherwise.
   @precondition *mat is initialized && rows is positive && *coords list contains 
             as many positive integers as there are rows.
   @postcondition memory is allocated for PRMAT structure mat.
   @param PRMAT *mat pointer to projection matrix for which the allocation is done.
   @param int set tells for which part of the structure allocation is to be done.
   @param int rows number of rows in a matrix.
   @param int *coords list of number of entries in each row.
   @return 0 if ok.
 */
int prmatAllocate(PRMAT *mat, int set,int rows, int *coords){

  int *iptr, ir, ip;

  if(PRMAT_VERBOSE) printf("PRMAT:prmatAllocate(*mat,%d,*coords) \n",rows);
  
  /*check the arguments*/
  if(mat->status==PRMAT_STATUS_UNINITIALIZED) return(1);
  if(rows<1) return(2);
  
  //If set != 0 allocate for projection matrix.
  if(set){
    /*allocate memory*/
    mat->factor_sqr_sum=calloc(rows,sizeof(float));
    mat->dimr=rows; //for dimr
    mat->dime=calloc(rows,sizeof(int)); 
    mat->_factdata=(unsigned short int***)calloc(rows,sizeof(unsigned short int**)); 
    //if not enough memory
    if(mat->_factdata==NULL || mat->dime==NULL) return(4);
    //iterate through the rows
    for(ir=0,iptr=coords; ir<rows; ir++){
      //copy the coords vector in the same loop
      mat->dime[ir]=*iptr; 
      //*iptr is the number of (non-zero) factors in this row
      mat->_factdata[ir]=(unsigned short int**)calloc(*iptr,sizeof(unsigned short int*)); 
      //if not enough memory
      if(mat->_factdata[ir]==NULL) return(4); 
      //iterate through the factors in this row
      for(ip=0; ip<(*iptr); ip++){ 
	//NOTE that the leaves are coordinates and factor in this order
	mat->_factdata[ir][ip]=(unsigned short int*)calloc(3,sizeof(unsigned short int)); 
	if(mat->_factdata[ir][ip]==NULL) return(4);
      }
      iptr++;
    }
    /*set data pointers; these pointers are used to avoid data loss*/
    mat->fact=mat->_factdata;    
  }// END OF ALLOCATING PROJECTION MATRIX

  //If set was 0 allocate for look-up table.
  else{
    //allocate memory for look up table
    mat->nrp=rows; 
    mat->_linesdata=(unsigned int**)calloc(rows,sizeof(unsigned int*)); 
    //if not enough memory
    if(mat->_linesdata==NULL) return(4);
    //iterate through the rows
    for(ir=0,iptr=coords; ir<rows; ir++){
      //iptr is the number of lors belonging to this pixel
      mat->_linesdata[ir]=(unsigned int*)calloc(*iptr + 2,sizeof(unsigned int)); 
      //if not enough memory
      if(mat->_linesdata[ir]==NULL) return(4); 
      iptr++;
    }
    //set data pointers; these pointers are used to avoid data loss
    mat->lines=mat->_linesdata;    
  }

  return(0);

}//END OF ALLOCATE

/* GET procedures for PRMAT datatype. USE ONLY THESE FOR GETTING. */

/* GET FUNCTIONS FOR THE PARAMETERS */

/**
   Returns the number of views (angles) in the geometrics.
   @precondition mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return the number of views in the geometrics.
*/
int prmatGetNV(PRMAT *mat){
  return mat->viewNr;
}

/**
   Returns the number of bins (distances) in the geometrics.
   @precondition mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return the number of bins in the geometrics.
*/
int prmatGetNB(PRMAT *mat){
  return mat->binNr;
}

/**
   Returns the image dimension in the geometrics.
   @precondition mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return the image dimension in the geometrics.
*/
int prmatGetID(PRMAT *mat){
  return mat->imgDim;
}

/* GET FUNCTIONS FOR ACCESSING LOOK-UP TABLE*/

/**
   Returns number of rows (pixels) in the look-up table.
   @precondition mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return number of pixels in look-up table, some negative value if ERROR.
*/
int prmatGetPIX(PRMAT *mat){
  return mat->nrp;
}

/**
   Returns coordinates of a pixel in place 'row' in the look-up table.
   @precondition 0<=row<=prmatGetPIX(*mat) && mat is initialized.
   END .
   @param PRMAT *mat pointer to a projection matrix.
   @param int row index of the pixel whose coordinates are to be returned.
   @return coordinates of a pixel in the look-up table, some negative value if ERROR.
*/
int prmatGetPixCoord(PRMAT *mat,int row){
  return mat->lines[row][0];
}

/**
   Returns the number of lines of response (rays) intersecting the pixel in place 'row'.
   @precondition 0<=row<=prmatGetPIX(*mat) && mat is initialized.
   END .
   @param PRMAT *mat pointer to a projection matrix.
   @param int row index of the pixel for which the number of lors is returned.
   @return number of lines of response intersecting the pixel, some negative value if ERROR.
*/
int prmatGetRays(PRMAT *mat,int row){
  return mat->lines[row][1];
}

/**
   Returns coordinates of a line of response in place (row,ind) in the look-up table.
   @precondition 0<=row<=prmatGetPIX(*mat) && 0<=ind<=prmatGetRays(*mat,row) && mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @param int row the row index.
   @param int ind the (non-zero) lor index.
   @return int coordinates (view*binNr + bin) of the referred line of response, some negative value if ERROR.
 */
int prmatGetBinView(PRMAT *mat, int row, int ind){
  return mat->lines[row][ind+2];
}

/* GET FUNCTIONS FOR ACCESSING PROJECTION MATRIX BY A LINE OF RESPONSE.*/

/**
   Returns number of rows (lines of response) in the given projection matrix.
   @precondition mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to the projection matrix.
   @return number of rows in the given projection matrix, some negative value if ERROR.
*/
int prmatGetRows(PRMAT *mat){

  //NOTE no checking on parameters to accelerate iteration
  return(mat->dimr);
}

/**
   Returns number of pixels intersected by the given line of response.
   @precondition 0<=row<=prmatGetLORS(*mat) && mat is initialized.
   END .
   @param PRMAT *mat pointer to the projection matrix.
   @param int row index of the line of response for which the number of pixels is returned.
   @return number of pixels intersected by given line of response, some negative value if ERROR.
*/
int prmatGetPixels(PRMAT *mat, int row){
  return(mat->dime[row]);
}

/**
   Returns coordinate of a pixel in place (row,pix) in the given projection matrix.
   @precondition 0<=row<=prmatGetLORS(*mat) && 0<=pix<=prmatGetPixels(row,*mat) && mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @param int row the row index.
   @param int pix the (non-zero) pixel index.
   @return coordinate of the referred pixel, some negative value if ERROR.
*/
int prmatGetCoord(PRMAT *mat, int row, int pix){
  return (int)(mat->fact[row][pix][1])*mat->imgDim + (int)(mat->fact[row][pix][0]);
}

/**
   Returns the x-coordinate of a pixel in place (row,pix) in the given projection matrix.
   @precondition 0<=row<=prmatGetLORS(*mat) && 0<=pix<=prmatGetPixels(row,*mat) && mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @param int row the row index.
   @param int pix the (non-zero) pixel index.
   @return coordinate of the referred pixel, some negative value if ERROR.
*/
int prmatGetXCoord(PRMAT *mat, int row, int pix){
  return (int)(mat->fact[row][pix][0]);
}

/**
   Returns the y-coordinate of a pixel in place (row,pix) in the given projection matrix.
   @precondition 0<=row<=prmatGetLORS(*mat) && 0<=pix<=prmatGetPixels(row,*mat) && mat is initialized. 
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @param int row the row index.
   @param int pix the (non-zero) pixel index.
   @return coordinate of the referred pixel, some negative value if ERROR.
*/
int prmatGetYCoord(PRMAT *mat, int row, int pix){
  return (int)(mat->fact[row][pix][1]);
}

/**
   Returns factor (weighting coefficient) for a pixel in place (row,pix) in given projection matrix.
   @precondition 0<=row<=prmatGetLORS(*mat) && 0<=pix<=prmatGetPixels(row,*mat) && mat is initialized.
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @param int row the row index.
   @param int pix the (non-zero) pixel index.
   @return factor for the referred pixel, some negative value if ERROR.
*/
float prmatGetFactor(PRMAT *mat, int row, int pix){
  return((float)(mat->scaling_factor)*(mat->fact[row][pix][2]));
}

/**
   Returns major semiaxe of the FOV in given projection matrix.
   @precondition mat is initialized.
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return major semiaxe of the FOV in given projection matrix
*/
float prmatGetMajor(PRMAT *mat){
  return(mat->prmatfov[0]);
}

/**
   Returns minor semiaxe of the FOV in given projection matrix.
   @precondition mat is initialized.
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return minor semiaxe of the FOV in given projection matrix
*/
float prmatGetMinor(PRMAT *mat){
  return(mat->prmatfov[1]);
}

/**
   Returns minimal value in given projection matrix.
   @precondition mat is initialized.
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return minimal value in given projection matrix.
*/
float prmatGetMin(PRMAT *mat){
  return(mat->min);
}

/**
   Returns greatest value in given projection matrix.
   @precondition mat is initialized.
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return maximal value in given projection matrix.
*/
float prmatGetMax(PRMAT *mat){
  return(mat->max);
}

/**
   Returns sum of all factors in given projection matrix.
   For initializing image matrix.
   @precondition mat is initialized.
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @return sum of all factors in given projection matrix.
*/
float prmatGetFactorSum(PRMAT *mat){
  return(mat->factor_sum);
}

/**
   Returns square sum of all factors in given projection matrix in given row.
   @precondition mat is initialized && 0<=row<=prmatGetLORS(*mat).
   @postcondition .
   @param PRMAT *mat pointer to a projection matrix.
   @param int row row index for accessing factors.
   @return square sum of all factors in given projection matrix in given row.
*/
float prmatGetFactorSqrSum(PRMAT *mat, int row){
 return(mat->factor_sqr_sum[row]);
}

/*ARCHIVING METHODS*/
/**
   Adds the given projection matrix to the file called prTDXXYY.prm. 
   Where T is the type of the scanner 
   (E=ECAT931 and G=GE Advance), D is the type of discretisation
   (0='0/1', 1='length of intersection' and 2='exact area'),
   XX is the major semiaxe of the FOV, and YY the minor semiaxe.
   If file already exists it is NOT created or rewritten.
   @precondition mat is initialized (contains a projection matrix). 
   @postcondition mat is saved in the file  prTDXXYY.prm. 
   @param PRMAT *mat pointer to the projection matrix to be saved.
   @return 0 if OK.
*/
int prmatSaveMatrix(PRMAT *mat){

  char matrixfile[FILENAME_MAX], semi_x_c[2], semi_y_c[2], mod[2];
  int     semi_x=mat->prmatfov[0], semi_y=mat->prmatfov[1], mode=mat->mode, row, fac, itmp, ind;
  unsigned short int x_coordinate, y_coordinate, value, p_coord, lors;
  unsigned int lor;
  float ftmp;
  FILE *fp;
  
  //Compose the file name to write
  sprintf(semi_x_c,"%i",semi_x);
  sprintf(semi_y_c,"%i",semi_y);
  //decide the type
  if(mat->type==PRMAT_TYPE_ECAT931) strcpy(matrixfile,"prE");
  else 
    if(mat->type==PRMAT_TYPE_GE) strcpy(matrixfile,"prG");
    else strcpy(matrixfile,"prN");
  //decide the mode 
  sprintf(mod,"%i",mode);
  strcat(matrixfile,mod);
  strcat(matrixfile,semi_x_c);
  strcat(matrixfile,semi_y_c);
  strcat(matrixfile,".prm");
  
  if(access(matrixfile, 0) != -1) {
    if(PRMAT_VERBOSE) printf("PRMAT:File %s already exists.\n", matrixfile);
    return(0);
  }
  
  fp=fopen(matrixfile,"wb+");
  if(fp==NULL) return(1);

  // Save status.
  fwrite(&(mat->status),sizeof(int),1,fp);  

  // Save type of the scanner.
  fwrite(&(mat->type),sizeof(int),1,fp);  

  // Save scanner geometrics.
  fwrite(&(mat->viewNr),sizeof(int),1,fp);  
  fwrite(&(mat->binNr),sizeof(int),1,fp);  
  fwrite(&(mat->imgDim),sizeof(int),1,fp);  

  // Save mode.
  fwrite(&(mat->mode),sizeof(int),1,fp);  

  // Save FOV.
  fwrite(&(mat->prmatfov[0]),sizeof(int),1,fp);
  fwrite(&(mat->prmatfov[1]),sizeof(int),1,fp); 

  // Save min, max, sum and scaling factor.
  fwrite(&(mat->min),sizeof(float),1,fp);
  fwrite(&(mat->max),sizeof(float),1,fp);
  fwrite(&(mat->factor_sum),sizeof(float),1,fp);
  fwrite(&(mat->scaling_factor),sizeof(float),1,fp);  

  // If projection matrix is set.
  if(mat->status >= PRMAT_STATUS_BS_OCCUPIED){

    fwrite(&(mat->dimr),sizeof(int),1,fp);
    //square sums
    for(row=0;row<prmatGetRows(mat);row++){
      ftmp=prmatGetFactorSqrSum(mat,row);
      fwrite(&ftmp,sizeof(float),1,fp);
    }
    //dime
    for(row=0;row<prmatGetRows(mat);row++){
      itmp=prmatGetPixels(mat,row);
      fwrite(&itmp,sizeof(int),1,fp);
    }
    //the actual data _factdata
    for(row=0;row<prmatGetRows(mat);row++){
      //coordinates and values in every row
      for(fac=0;fac<prmatGetPixels(mat,row);fac++){
	x_coordinate=prmatGetXCoord(mat,row,fac);
	fwrite(&x_coordinate,sizeof(unsigned short int),1,fp);
	y_coordinate=prmatGetYCoord(mat,row,fac);
	fwrite(&y_coordinate,sizeof(unsigned short int),1,fp);
	//NOTE that we wan't to save the values in unsigned short ints
	value=mat->fact[row][fac][2];
	fwrite(&value,sizeof(unsigned short int),1,fp);
      }
    }
  }// END OF SAVING PROJECTION MATRIX.

  // If look-up table is set.
  if(mat->status == PRMAT_STATUS_LU_OCCUPIED){
    //save first the statical parts of the structure
    fwrite(&(mat->nrp),sizeof(int),1,fp);
    //save lines: first coordinates of pixels inside the fov,
    for(row=0;row<prmatGetPIX(mat);row++){
      p_coord=prmatGetPixCoord(mat,row);
      fwrite(&p_coord,sizeof(unsigned short int),1,fp);
    }
    // then number of lors hitting a pixel.
    for(row=0; row<prmatGetPIX(mat); row++){
      lors=prmatGetRays(mat,row);
      fwrite(&lors,sizeof(unsigned short int),1,fp);
    }

    //save lors
    for(row=0;row<prmatGetPIX(mat);row++){
      for(ind=0; ind<prmatGetRays(mat,row); ind++){
	lor = prmatGetBinView(mat,row,ind);
	fwrite(&lor,sizeof(unsigned int),1,fp);
      }
    }
  }// END OF SAVING LOOK-UP TABLE.

  
  fclose(fp);
  return(0);
}

/**
   Reads one projection matrix from the given file to the given PRMAT structure mat.
   @precondition mat is initialized.
   @postcondition a projection matrix is read from the file fname. 
   @param char *fname file name to read.
   @param PRMAT *mat pointer to the projection matrix datatype.
   @return 0 if OK.
*/
int prmatReadMatrix(char *fname, PRMAT *mat){

  int fov[2], *dimentry=NULL, *nrlor, *iptr, row, fac, dimtmp, ind, count=0;
  unsigned int lor=0;
  unsigned short int  *nrlor_c, *nrlor_l, *usiptr, *usiptr_c;
  unsigned short int x_coordinate=0, y_coordinate=0, value=0;
  float *sqrsum=NULL, *fptr, sqrtmp;
  FILE *fp;

  if(PRMAT_VERBOSE) printf("\n PRMAT:prmatReadMatrix(%s,mat) \n",fname);
  if(mat->status==PRMAT_STATUS_UNINITIALIZED) return(1);
  prmatEmpty(mat);

  fp=fopen(fname,"rb");
  if(fp==NULL) return(1);

  //read first the statical parts of the structure
  fread(&(mat->status),sizeof(int),1,fp);
  if(PRMAT_VERBOSE) printf("PRMAT: status: %i \n",mat->status);
  fread(&(mat->type),sizeof(int),1,fp);
  if(PRMAT_VERBOSE) printf("PRMAT: scanner type: %i \n",mat->type);
  fread(&(mat->viewNr),sizeof(int),1,fp);
  fread(&(mat->binNr),sizeof(int),1,fp);
  fread(&(mat->imgDim),sizeof(int),1,fp);
  if(PRMAT_VERBOSE) printf("PRMAT: scanner geometrics: (%i,%i,%i) \n",mat->viewNr,mat->binNr,mat->imgDim);
  fread(&(mat->mode),sizeof(int),1,fp);
  if(PRMAT_VERBOSE) printf("PRMAT: discretisation mode: %i \n",mat->mode);
  fread(fov,sizeof(int),1,fp);
  fread((fov+1),sizeof(int),1,fp);
  mat->prmatfov=calloc(2,sizeof(int)); 
  mat->prmatfov[0]=fov[0];
  mat->prmatfov[1]=fov[1];
  if(PRMAT_VERBOSE) printf("PRMAT: FOV: (%i,%i) \n",mat->prmatfov[0],mat->prmatfov[1]);
  // Read min, max, sum and scaling factor.
  fread(&(mat->min),sizeof(float),1,fp);
  if(PRMAT_VERBOSE) printf("PRMAT: min: %f \n",mat->min);
  fread(&(mat->max),sizeof(float),1,fp);
  if(PRMAT_VERBOSE) printf("PRMAT: max: %f \n",mat->max);
  fread(&(mat->factor_sum),sizeof(float),1,fp);
  if(PRMAT_VERBOSE) printf("PRMAT: sum of all factors: %f \n",mat->factor_sum);
  fread(&(mat->scaling_factor),sizeof(float),1,fp);
  if(PRMAT_VERBOSE) printf("PRMAT: scaling factor: %f \n",mat->scaling_factor);  
 
  // If projection matrix has been occupied.
  if(mat->status >= PRMAT_STATUS_BS_OCCUPIED){
    if(PRMAT_VERBOSE) printf("PRMAT: reading projection matrix \n");  
    fread(&(mat->dimr),sizeof(int),1,fp);
    if(PRMAT_VERBOSE) printf("PRMAT: number of rows in the projection matrix: %i \n",mat->dimr);  
    //read square sums
    sqrsum=calloc(mat->dimr,sizeof(float));  
    for(row=0, fptr=sqrsum;row<(mat->dimr);row++, fptr++){
      fread(&sqrtmp,sizeof(float),1,fp);
      *fptr=sqrtmp;
    }
    //read dime
    dimentry=(int*)calloc(mat->dimr,sizeof(int));  
    count = 0;
    for(row=0,iptr=dimentry;row<(mat->dimr);row++,iptr++){
      fread(&dimtmp,sizeof(int),1,fp);
      *iptr=dimtmp;
      count += dimtmp;
    }

    //ready to allocate; NOTE that dime is also set in function prmatAllocate()    
    prmatAllocate(mat,1,mat->dimr,dimentry);
    if(PRMAT_VERBOSE) printf("PRMAT:prmatAllocate(mat) done in prmatReadMatrix(%s,mat)\n",fname);

    //now we can set square sums
    for(row=0, fptr=sqrsum;row<(mat->dimr);row++, fptr++){
      mat->factor_sqr_sum[row]=*fptr;
    }
    //read the actual data _factdata
    for(row=0;row<(mat->dimr);row++){
      //read coordinates and values in every row
      for(fac=0;fac<prmatGetPixels(mat,row);fac++){
	fread(&x_coordinate,sizeof(unsigned short int),1,fp);
	mat->fact[row][fac][0]=x_coordinate;
	fread(&y_coordinate,sizeof(unsigned short int),1,fp);
	mat->fact[row][fac][1]=y_coordinate;
	fread(&value,sizeof(unsigned short int),1,fp);
	mat->fact[row][fac][2]=value;
      }
    }
  }// END OF READING PROJECTIONS

  // If look-up table has been occupied.
  if(mat->status == PRMAT_STATUS_LU_OCCUPIED){
    if(PRMAT_VERBOSE) printf("PRMAT: reading look-up table \n");  
    fread(&(mat->nrp),sizeof(int),1,fp);
    if(PRMAT_VERBOSE) printf("PRMAT: number of pixels inside the FOV: %i \n",mat->nrp);
    //read nrlor
    nrlor_c=(unsigned short int*)calloc(mat->nrp,sizeof(unsigned short int));  
    nrlor_l=(unsigned short int*)calloc(mat->nrp,sizeof(unsigned short int));  

    //first coordinates of pixels.
    usiptr = nrlor_c;
    for(row=0; row<mat->nrp; row++, usiptr++){
      // Read next coordinate from file.
      fread(usiptr,sizeof(unsigned short int),1,fp);
    }
    //then number of lors hitting a pixel
    usiptr = nrlor_l;
    for(row=0; row<mat->nrp; row++, usiptr++){
      // Read next lornr from file.
      fread(usiptr,sizeof(unsigned short int),1,fp);
    }

    if(PRMAT_VERBOSE){
      // Get number of lors in lut.
      usiptr = nrlor_l;
      count = 0;
      for(row=0; row<mat->nrp; row++, usiptr++){
	count += *usiptr;
      }
      printf("PRMAT: nr of lors in lut:  %i \n",count);
      //printf("  nr of lors in ave:  %i \n",count/mat->nrp);
    }

    // Copy list usi format to integer list.
    nrlor = (int*)calloc(mat->nrp,sizeof(int));
    usiptr = nrlor_l;
    iptr = nrlor;
    for(row=0; row<mat->nrp; row++, iptr++){
      *iptr = (int)*usiptr++;
    }    

    // Now we can allocate memory for look-up table.
    prmatAllocate(mat,0,mat->nrp,nrlor);
    if(PRMAT_VERBOSE) printf("PRMAT:prmatAllocate(mat) done in prmatReadMatrix(%s,mat)\n",fname);
    // And fill mat->lines.
    usiptr_c = nrlor_c;
    usiptr = nrlor_l;
    for(row=0;row<mat->nrp;row++ , usiptr++, usiptr_c++){
      mat->lines[row][0] = *usiptr_c;
      mat->lines[row][1] = *usiptr;
    }

    // Read the data _linesdata.
    for(row=0;row<prmatGetPIX(mat);row++){
      for(ind=0; ind<prmatGetRays(mat,row); ind++){
	fread(&lor,sizeof(unsigned int),1,fp);
	mat->lines[row][ind+2] = lor;
      }
    }

    free((unsigned short int*)nrlor_l);
    free((unsigned short int*)nrlor_c);
  }// END OF READING LOOK-UP TABLE


  if(PRMAT_VERBOSE) printf("PRMAT: prmatReadMatrix() done. \n");
  free((float*)sqrsum); free((int*)dimentry);


  fclose(fp);

  return(0);
}

