/**
   Copyright (c) 2004,2005,2013 by Turku PET Centre
   
   File: libtpcroi.c
   
   Purpose: methods for testing functions in the library libtpcroi,
   and for printing out libtpcroi information, such as Readme,
   History and build information. 

   Version:
   2005-04-25 1.0.0 Jarkko Johansson
   2005-04-26 1.1.0 Jarkko Johansson
   2005-05-13 1.2.0 Harri Merisaari
   2013-02-12 1.2.2 Vesa Oikonen
     Updated print_usage().
   
*/
/*****************************************************************************/
/** Library name */
#define LIB_NAME "libtpcroi"

/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
/*****************************************************************************/
#include <img.h>
#include "include/libtpcroiv.h"
/*****************************************************************************/
#define MAXVAL 100000;
/*****************************************************************************/

// Functions for initializing image data:

/**
   Function for drawing a cros on the center of the image region,
   of width 2 pixels and value 1.
 */
void drawCross(IMG *img);

/**
   Function for drawing an unsymmetric cross for testing image transformation
   functions.
 */
void drawUnSymmetricCross(IMG *img);

/** 
    Function for printing the usage information of the libtpcroi test program.
*/
void print_usage();

// Verbose mode switch:
int VERBOSE = 0;

int main (int argc, char *argv[ ])
{
    int c;
    int errflg=0, functionflag=0, exit_code=0;
    extern char *optarg;
    extern int optind, optopt;
    
    while ((c = getopt(argc, argv, "vVbBrRhHf:F:")) != -1) {
      switch (c) {

      case 'V':
      case 'v':
	// Set verbose flag on:
	VERBOSE = 1;
	break;

      case 'B':
      case 'b':
	// Print out the build information:
	libtpcroi_print_build(stdout);
	return(0);

      case 'r':
	// Print out the Readme message:
	libtpcroi_print_readme(stdout);
	return(0);
      case 'R':
	// Print out the Readme message with Doxygen tags:
	libtpcroi_print_dreadme(stdout);
	return(0);

      case 'h':
	// Print out the History message:
	libtpcroi_print_history(stdout);
	return(0);
      case 'H':
	// Print out the History message with Doxygen tags:
	libtpcroi_print_dhistory(stdout);
	return(0);

      case 'f':
      case 'F':
	// Set function serial number to be tested:
	functionflag = atoi(optarg);
	break;

      case ':': // -f or -z without argument
	fprintf(stderr,"Option -%c requires an argument\n", optopt);
	errflg++;
	break;
      case '?':
	fprintf(stderr,"Unrecognised option: -%c\n", optopt);
	errflg++;
      }
    }// End parsing command line options...

    if (errflg) {
      print_usage();
      return(2);
    }

    if(!functionflag){
      print_usage();
      return(0);
    }

    libtpcroi_print_build(stdout);

    fprintf(stdout,"\n    NO TEST FUNCTION IMPLEMENTATIONS!\n");
    
    exit(exit_code);
}

void drawCross(IMG *img){

  int plane, row, column, frame, r_limit, c_limit;

// Draw a symmetric cross on all planes and frames in the image:
  r_limit = (img->dimy - 2)/2;
  c_limit = (img->dimx - 2)/2;
  for(plane = 0; plane < img->dimz; plane++)
    for(row = 0; row < img->dimy; row++)
      for(column = 0; column < img->dimx; column++)
	for(frame = 0; frame < img->dimt; frame++){
	  if((row >= r_limit) && (row <= r_limit + 1))
	    img->m[plane][row][column][frame] = 1;
	  if((column >= c_limit) && (column <= c_limit + 1)\
	     && !((row >= r_limit) && (row <= r_limit + 1)))
	    img->m[plane][row][column][frame] = 1;
	}

}

void drawUnSymmetricCross(IMG *img){


  int plane, row, column, frame, r_limit, c_limit;

// Draw an unsymmetric cross on all planes and frames in the image:
  r_limit = (img->dimy - 2)/2;
  c_limit = (img->dimx - 2)/2;
  for(plane = 0; plane < img->dimz; plane++)
    for(row = 0; row < img->dimy; row++)
      for(column = 0; column < img->dimx; column++)
	for(frame = 0; frame < img->dimt; frame++){
	  if((row >= r_limit) && (row <= r_limit + 1)){
	    if(column < c_limit) // left 
	      img->m[plane][row][column][frame] = 1;
	    if(column > c_limit + 1) // right
	      img->m[plane][row][column][frame] = 3;
	    if((column >= c_limit) && (column <= c_limit + 1)) //center
	      img->m[plane][row][column][frame] = 5;
	  }
	  else{
	    if((column >= c_limit) && (column <= c_limit + 1)){	     
	      if(row < r_limit) // top
		img->m[plane][row][column][frame] = 4;
	      if(row > r_limit) // bottom
		img->m[plane][row][column][frame] = 2;

	    }
	    else{ //other
	      img->m[plane][row][column][frame] = 0;
	    }
	  }
	}

}
/******************************************************************************/

/******************************************************************************/
void print_usage() {

  libtpcroi_print_build(stdout);
  puts("\n"
  "   Methods for testing functions in the library "LIB_NAME"\n"
  "   and for printing out associated information, such as Readme,\n"
  "   History and build information. \n"
  "\n"
  "   Usage: "LIB_NAME" [-Options]\n"
  "\n"
  "   Options:\n"
  "   -h | H \n"
  "       Print out the "LIB_NAME" History message. Include Doxygen style tags\n"
  "       by using option '-H'.\n"
  "   -r | R \n"
  "       Print out the "LIB_NAME" Readme message. Include Doxygen style tags\n"
  "       by using option '-R'.\n"
  "   -b\n"
  "       Print out the "LIB_NAME" build information.\n"
  "   -f <function serial number>\n"
  "       Runs test procedures for the functions corresponding to given\n"
  "       'function serial number'. Serial numbers are defined in file\n"
  "       "LIB_NAME".c; enter 0 to execute them all.\n"
  "   -v\n"
  "       Run in verbose mode.\n"
  "\n"
  "   E.g.: "LIB_NAME" -h\n"
  "         "LIB_NAME" -v -f 1 \n"
  "\n"
  );
  fflush(stdout);
}
/******************************************************************************/

/******************************************************************************/
