/******************************************************************************
  Copyright (c) 2013 by Turku PET Centre

  File:      tacio.h

  Version:
  2013-09-19 Vesa Oikonen


******************************************************************************/
#ifndef _TACIO_H
#define _TACIO_H
/*****************************************************************************/
/* Max nr of parameters */
#ifndef MAX_PETPARAMS
#define MAX_PETPARAMS 100
#endif
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/
#include "libtpctaciov.h"
/*****************************************************************************/
/* Backup file extension */
#ifndef BACKUP_EXTENSION
#define BACKUP_EXTENSION ".bak"
#endif 
/* Max length of parameter names and units */
#ifndef MAX_PETPARNAME_LEN
#define MAX_PETPARNAME_LEN 15
#endif
/** Max length of TAC ID and PETPAR parameter names */
#ifndef MAX_TACNAME_LEN
#define MAX_TACNAME_LEN 20
#endif
/** Max nr of sub name fields; do not set to less than 3 */
#ifndef MAX_SUBTACID_NR
#define MAX_SUBTACID_NR 3
#endif
/* Max length of parameter names and units */
#ifndef MAX_PARNAME_LEN
#define MAX_PARNAME_LEN MAX_TACNAME_LEN
#endif
/** Definitions for weighting */
#define WEIGHTING_UNKNOWN 0
#define WEIGHTING_ON      1
#define WEIGHTING_OFF     2
/** Definitions for TAC origin or content */
#define TACID_UNKNOWN    0
#define TACID_AVG        1
#define TACID_DX         2
#define TACID_SIN        3
#define TACID_REF        4
#define TACID_PLASMA     5
#define TACID_BLOOD      6
#define TACID_PARENT     7
#define TACID_METABOLITE 8
/** Definitions for Decay correction */
#define DECAY_UNKNOWN      0
#define DECAY_CORRECTED    1
#define DECAY_NOTCORRECTED 2
/* Max nr of parameters */
#ifndef MAX_PETPAR_NR
#define MAX_PETPAR_NR 100
#endif
/*****************************************************************************/

/*****************************************************************************/
/** tacio function return codes; see status strings in tacio_status */
enum {TACIO_OK, TACIO_FAULT, TACIO_CANNOTOPEN, TACIO_CANNOTWRITE,
    TACIO_INVALIDFORMAT, TACIO_TOOBIG, TACIO_OUTOFMEMORY, TACIO_NOTABLE,
    TACIO_BADTABLE, TACIO_NAN, TACIO_NOTAC, TACIO_NOMATCH,
    TACIO_NOTUNIT, TACIO_NOCUNIT,
    TACIO_INVALIDISOTOPE
    };
/** TAC and Parameter file format codes; see tacio_format strings */
enum {TAC_FORMAT_UNKNOWN, TAC_FORMAT_SIMPLE, TAC_FORMAT_DFT, TAC_FORMAT_IFT,
    PAR_FORMAT_FIT, TAC_FORMAT_NCI, TAC_FORMAT_PMOD, TAC_FORMAT_CSV_INT,
    TAC_FORMAT_CSV_UK, TAC_FORMAT_TSV_INT, TAC_FORMAT_TSV_UK, TAC_FORMAT_CPT,
    TAC_FORMAT_IDWC, TAC_FORMAT_IF, TAC_FORMAT_XML, TAC_FORMAT_HTML,
    TAC_FORMAT_XELERIS, TAC_FORMAT_AMIDE, TAC_FORMAT_BINARY, PAR_FORMAT_RES,
    PAR_FORMAT_CSV_INT, PAR_FORMAT_CSV_UK, PAR_FORMAT_TSV_INT, PAR_FORMAT_TSV_UK
    };
/*****************************************************************************/
/** Nr of decimals for sample concentration (y) values */
extern int TAC_NR_OF_DECIMALS;
/*****************************************************************************/
/*
 *  Data structure definitions
 */
/** One CSV content field */
typedef struct {
  /** Position in CSV: row index */
  int row;
  /** Position in CSV: column index */
  int col;
  /** Pointer to field content */
  char *content;
} CSV_item;

/** Main CSV content data */
typedef struct {
  /** Pointer to one data field */
  CSV_item *c;
  /** Nr of data fields */
  int nr;
  /** Nr of rows */
  int row_nr;
  /** Max column number per row */
  int col_nr;
  /** CSV field separator (',', ';', or '\t') */
  char separator;
} CSV;
/*****************************************************************************/
/**
 *  CSV structure definition
 */
/*****************************************************************************/
/**
 *  TAC ID structure definition
 */
typedef struct {
  /** Full name string, using '_' to separate subparts when necessary */
  char      name[MAX_TACNAME_LEN+1];
  /** TAC ID string subfields, contained in full name; traditionally,
   *  the send field is used to store hemisphere, and 3rd field the image plane;
      only for temporary use */
  char      sub[MAX_SUBTACID_NR][MAX_TACNAME_LEN+1];
  /** ROI side: TACID_UNKNOWN, TACID_AVG, TACID_DX, TACID_SIN */ 
  int       side;
  /** ROI image plane */ 
  int       plane;
  /** Input type: TACID_UNKNOWN, TACID_REF, TACID_PLASMA, TACID_BLOOD,
   *  TACID_PARENT, TACID_METABOLITE */
  int       input;  
#if(1)
  /** Temporary switches for outside procedures */
  char      sw, sw2, sw3;
#endif
} TACID;
/*****************************************************************************/
/**
 *  Struct for one (regional) set of parameters
 */
typedef struct {
  /** TAC ID */
  TACID     id;
  /** Number (type) of mathematical function */
  int       function;
  /** Number (type) of model */
  int       model;
  /** Fit start time (unit specified in top struct) */
  double    start;
  /** Fit end time (unit specified in top struct) */
  double    end;
  /** Number of saved parameters (index of last listed parameter + 1);
   *  must be < MAX_PETPAR_NR  */
  int       parNr;
  /** The number of fitted parameters (excluding fixed parameters);
   *  must be < MAX_PETPAR_NR  */
  int       fitNr;
  /** Number of fitted data points */
  int       dataNr;
  /** Array of parameters */
  double    p[MAX_PETPAR_NR];
  /** Parameter SD's   */
  double    sd[MAX_PETPAR_NR];
  /** Lower 95% confidence intervals */
  double    cl1[MAX_PETPAR_NR];
  /** Upper 95% confidence intervals */
  double    cl2[MAX_PETPAR_NR];
  /** (weighted) sum-of-squares */
  double    wss;
#if(0)
  /** Temporary switches for outside procedures */
  char      sw, sw2, sw3;
#endif
} PETPARVOI;

/** Main PET model parameters struct */
typedef struct {
  /** File format: PAR_FORMAT_XXX */
  int       format;
  /** Number of regions */
  int       voiNr;
  /** Name of study (study number) */
  char      studynr[MAX_STUDYNR_LEN+1];

  /** Parameter estimation date and time */
  time_t    time;
  /** Analysis program name */
  char      program[1024];

  /* Names of original datafiles */
  /** Name of original tissue datafile */
  char      datafile[FILENAME_MAX];
  /** Name of original ref datafile */
  char      reffile[FILENAME_MAX];
  /** Name of original plasmafile */
  char      plasmafile[FILENAME_MAX];
  /** Name of second original plasmafile */
  char      plasmafile2[FILENAME_MAX];
  /** Name of original bloodfile */
  char      bloodfile[FILENAME_MAX];
  /** Name of reference region */
  char      refname[MAX_REGIONNAME_LEN+1];
  /** Free text field describing fit method */
  char      fitmethod[FILENAME_MAX];
  /** Unit of original data */
  int       cunit;
  /** Time unit of original data and fit start and end times */
  int       tunit;

  /** Were data weighted in parameter estimation?
      WEIGHTING_UNKNOWN, WEIGHTING_ON, WEIGHTING_OFF */
  int       weighting;
  /** Tissue density (g/ml) */
  double    density;
  /** Lumped Constant (unitless) */
  double    lc;
  /** Beta */
  double    beta;
  /** Plasma concentration of native substrate, e.g. glucose of FFA */
  double    concentration;
  /** Vb percentage */
  double    Vb;
  /** fA percentage (arterial volume of Vb) */
  double    fA;
  /** Extraction fraction */
  double    E;
  
  /** Nr of parameter names and units; set also regional parNr */
  int       parNr;

  /** List of parameter names */
  char      parname[MAX_PETPAR_NR][MAX_PARNAME_LEN+1];
  /** List of parameter units */
  int       parunit[MAX_PETPAR_NR];
  
  /** Internal variables: Number of allocated curves (VOIs) */
  int      _voidataNr;

  /** Pointers to regional curves */
  PETPARVOI *voi;

} PETPAR;
/*****************************************************************************/
/**
 *  Struct for TAC data (may contain several TACs which have the same sample
 *  times)
 */
/** Definition for information regarding one TAC */
typedef struct {
  /** TAC ID */
  TACID    id;
  /** Volume of region (mm x mm x mm by default) */
  double   size;
  /* Pointer to curve data and two modified curves */
  /** Pointer to original TAC */
  double   *y;
  /** Pointer to 1st modified TAC (for user) */
  double   *y2;
  /** Pointer to 2nd modified TAC (for user) */
  double   *y3;
#if(0)
  /** Temporary switches for outside procedures */
  char      sw, sw2, sw3;
#endif
  } TACVOI;

/* Definition for a set of TACs */
typedef struct {
  /* Number of data frames (samples), and number of planes (curves) */
  /** Nr of samples (frames) in each TAC */
  int           frameNr;
  /** Nr of regional TACs */
  int           voiNr;
  /** Name of study (study number) */
  char          studynr[MAX_STUDYNR_LEN+1];
  /* Units */
  /** Unit of y values */
  int           cunit;
  /** Unit of x values: TUNIT_UNKNOWN, TUNIT_SEC, TUNIT_MIN, ... */
  int           tunit;
  /* Study information */
  /** Name of radiopharmaceutical */
  char          radiopharmaceutical[32];
  /** Isotope code: TPCISOT_BR_75, ...*/
  int           isocode;
  /** Decay correction: DECAY_UNKNOWN, DECAY_CORRECTED,
   *                    DECAY_NOTCORRECTED */
  char          decayCorrected;
  /** Scan start date and time: YYYY-MM-DD hh:mm:ss */
  time_t        scantime;
  /** Tracer injection date and time: YYYY-MM-DD hh:mm:ss */
  time_t        injtime;
  /* Specified frame time */
  /** Sample time: are sample frame times (start and end) available (1), or
   *  just the frame middle time point or blood sampling time (0) */    
  int           isframe;
  /* Pointers to time values */
  /** Middle frame time (or blood sample time); units specified with tunit */
  double       *x;
  /** Frame start time; units specified with tunit  */
  double       *x1;
  /** Frame end time; units specified with tunit    */
  double       *x2;
  /** Pointers to curves */
  TACVOI       *voi;
  /** Variable indicating whether weights are present
      WEIGHTING_UNKNOWN, WEIGHTING_ON, WEIGHTING_OFF */
  int           weighting;
  /** Pointer to weight factors */
  double       *w;
  /* Internal variables for TAC procedures */
  /** Internal variable: Size of allocated memory (doubles) */
  int           _dataSize;
  /** TAC file format:  */
  int           format;
  /** Additional interfile type header fields saved in TAC file */
  IFT           ift;
  /** Internal variable: Pointer to memory */
  double       *_data;
  /** Internal variable: Number of allocated curves (VOIs)*/
  int           _voidataNr;
  /** Internal variable: Number of allocated frames (samples)*/
  int           _framedataNr;
  } TAC;

/*****************************************************************************/

/*****************************************************************************/
/* par.c */
extern void parEmpty(PETPAR *par);
extern void parInit(PETPAR *par);
extern int parAllocate(PETPAR *par, int voiNr);
extern void parPrint(PETPAR *par);
extern int doubleMatch(double g1, double g2, double e);
extern int parMatchParameters(
  PETPAR *p1, PETPAR *p2, int par_index, int test_sd,
  double limit_abs, double limit_rel, int verbose
);
extern int parMatchExtraParameters(PETPAR *p1, PETPAR *p2);
extern int parMatchParameternames(PETPAR *p1, PETPAR *p2);
extern int parMatchRegionnames(PETPAR *p1, PETPAR *p2);
extern int parMatchHeader(PETPAR *p1, PETPAR *p2);
extern int parMatch(PETPAR *par1, PETPAR *par2, int verbose);
extern int parIsWSS(PETPAR *par);
extern int parIsFitrange(PETPAR *par);
extern int parIsFunction(PETPAR *par);
extern int parIsModel(PETPAR *par);
extern int parHighestFitnr(PETPAR *par);
extern int parHighestParnr(PETPAR *par);
extern int parHighestDatanr(PETPAR *par);
extern int parSDWithPar(PETPAR *par, int pi);
extern int parCLWithPar(PETPAR *par, int pi);
extern int parNrofUnits(PETPAR *par);
extern void parEnsureNames(PETPAR *par);
/*****************************************************************************/

/*****************************************************************************/
/* pario.c */
extern int parWrite(PETPAR *par, char *filename, int format, int verbose);
extern int parRead(PETPAR *par, char *filename, int verbose);
extern int parPrintType(PETPAR *par, int parIndex);
/*****************************************************************************/
/* parcsv.c */
extern int parWriteCSV(PETPAR *par, FILE *fp, int verbose);
extern int parReadCSV(PETPAR *par, char *filename, int verbose);
/*****************************************************************************/
/* parres.c */
extern int parWriteRES(PETPAR *par, FILE *fp, int verbose);
extern int parReadRES(PETPAR *par, char *filename, int verbose);
extern int parProcessRESData(PETPAR *par, IFT *ift, int verbose);
/*****************************************************************************/
/* parfit.c */
extern int parWriteFIT(PETPAR *par, FILE *fp, int verbose);
extern int parReadFIT(PETPAR *par, char *filename, int verbose);
extern int parProcessFITData(PETPAR *par, IFT *ift, int verbose);
/*****************************************************************************/

/*****************************************************************************/
/* csv.c */
extern void csvInit(CSV *csv);
extern void csvEmpty(CSV *csv);
extern int csvRead(CSV *csv, char *fname, int verbose);
extern int csvWrite(CSV *csv, char *fname, int verbose);
extern void csvPrint(CSV *csv);
extern int csvWriteText(FILE *fp, const char *txt, int tointl);
void csvSeparator(FILE *fp, char *field_separator, char *decimal_separator,
  int verbose);
/*****************************************************************************/

/*****************************************************************************/
/* tac.c */
extern char *tacStatus(int status_index);
extern void tacInit(TAC *d);
extern void tacEmptyExceptIFT(TAC *d);
extern void tacEmpty(TAC *d);
extern int tacSetmem(TAC *tac, int frameNr, int voiNr);
extern int tacCopyTacvoihdr(TACVOI *d1, TACVOI *d2);
extern int tacCopyVoihdr(TAC *d1, int from, TAC *d2, int to);
extern int tacCopyTacvoidata(TACVOI *d1, TACVOI *d2, int frameNr);
extern int tacCopyTacvoi(TACVOI *d1, TACVOI *d2, int frameNr);
extern int tacAddmem(TAC *tac, int voiNr);
extern int tacCopyhdr(TAC *d1, TAC *d2, int ow, int ift);
extern int tacAdd(TAC *d1, TAC *d2, int index);
extern void tacEnsureNames(TAC *d);
extern int tacIsSize(TAC *d);
/*****************************************************************************/

/*****************************************************************************/
/* tacid.c */
extern int tacCopyTacid(TACID *d1, TACID *d2);
extern int tacNameSplit(TACID *d);
extern int tacNameCatenate(TACID *id);
extern int tacNameMatch(TACID *d, char *test_str, int verbose);
/*****************************************************************************/

/*****************************************************************************/
/* tacio.c */
extern char *tacFormattxt(int format_index);
extern int tacFormatDetermine(const char *fname, int verbose);
extern int tacRead(TAC *d, char *filename, int verbose);
extern int iftGetDataDimensions(IFT *ift, int *lineNr, int *colNr);
extern int tacWrite(TAC *d, char *filename, int format, int extra, int verbose);
/*****************************************************************************/
/* tacsimp.c */
extern int tacReadSimple(TAC *d, char *filename, int verbose);
extern int tacProcessSimpleData(TAC *d, int verbose);
extern int tacWriteSimple(TAC *d, FILE *fp, int extra, int verbose);
/*****************************************************************************/
/* tacdft.c */
extern int tacWriteDFT(TAC *d, FILE *fp, int extra, int verbose);
extern int tacProcessDFTData(TAC *d, int verbose);
extern int tacReadDFT(TAC *d, char *filename, int verbose);
/*****************************************************************************/
/* tacpmod.c */
extern int tacWritePmod(TAC *d, FILE *fp, int verbose);
extern int tacProcessPmodData(TAC *d, int verbose);
extern int tacReadPmod(TAC *d, char *filename, int verbose);
/*****************************************************************************/
/* taccsv.c */
extern int tacWriteCSV(TAC *d, FILE *fp, int verbose);
/*****************************************************************************/

/*****************************************************************************/
/* taccomp.c */
extern int tacMatchUnit(TAC *d1, TAC *d2, int verbose);
extern int tacMatchTacID(TAC *d1, TAC *d2, int whatname, int verbose);
extern int tacMatchConc(TAC *d1, TAC *d2, double test_abs, int verbose);
extern int tacMatchFrames(TAC *d1, TAC *d2, double test_abs, int verbose);
/*****************************************************************************/

/*****************************************************************************/
/* tacdecay.c */
extern int tacDecayCorrection(
  TAC *d, double hl, int mode, int y, int y2, int y3, int verbose);
/*****************************************************************************/

/*****************************************************************************/
/* tacnan.c */
extern int tacFrameNaNs(TAC *d);
extern int tacValueNaNs(TAC *d);
extern int tacNaNs(TAC *d);
/*****************************************************************************/

/*****************************************************************************/
/* tacsort.c */
extern int tacSortByFrame(TAC *d);
extern int tacSortByName(TAC *d);
/*****************************************************************************/

/*****************************************************************************/
/* tacsel.c */
extern int tacSelectVOIs(TAC *d, const char *region_name, int reset);
extern int tacSelectedVOIs(TAC *d);
extern int tacSelectBestReference(TAC *d);
/*****************************************************************************/

/*****************************************************************************/
#endif /* _TACIO_H */
