/*******************************************************************************
  Copyright (c) 2013 by Turku PET Centre
   
  File: libtpctacio.c
   
  Purpose: methods for testing functions in the library libtpctacio,
  and for printing out library information, such as Readme,
  History and build information. 

  Version history:
  2013-09-27 Vesa Oikonen
       First created.

*******************************************************************************/

/******************************************************************************/
/** Library name */
#define LIB_NAME "libtpctacio"
/******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h> // for mkdir()
/******************************************************************************/
#include "libtpcmisc.h"
/******************************************************************************/
#include "include/tacio.h"
#include "include/libtpctaciov.h"
/******************************************************************************/
/** Serial numbers for function names. */
#define ALLTESTS    0
#define PAR1        1
#define PAR2        2
#define PARIO1    101
#define PARIO2    102
#define PARIO3    103
#define TACIO1    201
#define TACIO2    210
#define TACIO3    220
#define TACIO4    230
#define TACID1    301
#define TACID2    302
#define TACNAN1   401
#define TACSORT1  501
#define TACSORT2  502
#define TACSEL1   601
#define TACSEL2   602
/******************************************************************************/

/******************************************************************************/
// Test function declarations
int test_par1();
int test_par2();
int test_pario1();
int test_pario2();
int test_pario3();
int test_tacio1();
int test_tacio2();
int test_tacio3();
int test_tacio4();
int test_tacid1();
int test_tacid2();
int test_tacnan1();
int test_tacsort1();
int test_tacsort2();
int test_tacsel1();
int test_tacsel2();
// Helping function declarations
int create_par(PETPAR *par);
int create_par_for_fit(PETPAR *par);
int create_par_for_res(PETPAR *par);
int create_tac(TAC *tac);
/******************************************************************************/

/******************************************************************************/
/** 
    Function for printing the usage information of the test program.
*/
void print_usage();

/******************************************************************************/
// Verbose mode switch:
int VERBOSE = 0;
/******************************************************************************/

/******************************************************************************/
int main(
  /** Nr of arguments */
  int argc,
  /** Pointer to arrays of argument string */
  char *argv[ ]
) {
  int c;
  int errflg=0, functionflag=-1, exit_code=0;
  extern char *optarg;
  extern int optind, optopt;
    
  while ((c = getopt(argc, argv, "vVbBrRhHf:F:")) != -1) {
    switch (c) {

    case 'V':
    case 'v':
      // Set verbose flag on:
      VERBOSE = 1;
      break;
    case 'B':
    case 'b':
      // Print out the build information:
      libtpctacio_print_build(stdout);
      return(0);
    case 'r':
      // Print out the Readme message:
      libtpctacio_print_readme(stdout);
      return(0);
    case 'R':
      // Print out the Readme message with Doxygen tags:
      libtpctacio_print_dreadme(stdout);
      return(0);
    case 'h':
      // Print out the History message:
      libtpctacio_print_history(stdout);
      return(0);
    case 'H':
      // Print out the History message with Doxygen tags:
      libtpctacio_print_dhistory(stdout);
      return(0);
    case 'f':
    case 'F':
      // Set function serial number to be tested:
      functionflag = atoi(optarg);
      break;
    case ':': // -f without argument
      fprintf(stderr, "Option -%c requires an argument\n", optopt);
      errflg++;
      break;
    case '?':
      fprintf(stderr, "Unrecognised option: -%c\n", optopt);
      errflg++;
    }
  }// End parsing command line options...

  if(errflg) {print_usage(); return(2);}

  if(functionflag<0) {print_usage(); return(0);}

  libtpctacio_print_build(stdout);

  // Choose function(s) to test:
  switch(functionflag){
    case ALLTESTS:
      exit_code = test_par1(); if(exit_code) exit(exit_code);
      exit_code = test_par2(); if(exit_code) exit(exit_code);
      exit_code = test_pario1(); if(exit_code) exit(exit_code);
      exit_code = test_pario2(); if(exit_code) exit(exit_code);
      exit_code = test_pario3(); if(exit_code) exit(exit_code);
      exit_code = test_tacio1(); if(exit_code) exit(exit_code);
      exit_code = test_tacio2(); if(exit_code) exit(exit_code);
      exit_code = test_tacio3(); if(exit_code) exit(exit_code);
      exit_code = test_tacio4(); if(exit_code) exit(exit_code);
      exit_code = test_tacid1(); if(exit_code) exit(exit_code);
      exit_code = test_tacid2(); if(exit_code) exit(exit_code);
      exit_code = test_tacnan1(); if(exit_code) exit(exit_code);
      exit_code = test_tacsort1(); if(exit_code) exit(exit_code);
      exit_code = test_tacsort2(); if(exit_code) exit(exit_code);
      exit_code = test_tacsel1(); if(exit_code) exit(exit_code);
      exit_code = test_tacsel2(); if(exit_code) exit(exit_code);
      break;
    case PAR1:
      exit_code = test_par1();
      break;
    case PAR2:
      exit_code = test_par2();
      break;
    case PARIO1:
      exit_code = test_pario1();
      break;
    case PARIO2:
      exit_code = test_pario2();
      break;
    case PARIO3:
      exit_code = test_pario3();
      break;
    case TACIO1:
      exit_code = test_tacio1();
      break;
    case TACIO2:
      exit_code = test_tacio2();
      break;
    case TACIO3:
      exit_code = test_tacio3();
      break;
    case TACIO4:
      exit_code = test_tacio4();
      break;
    case TACID1:
      exit_code = test_tacid1();
      break;
    case TACID2:
      exit_code = test_tacid2();
      break;
    case TACNAN1:
      exit_code = test_tacnan1();
      break;
    case TACSORT1:
      exit_code = test_tacsort1();
      break;
    case TACSORT2:
      exit_code = test_tacsort2();
      break;
    case TACSEL1:
      exit_code = test_tacsel1();
      break;
    case TACSEL2:
      exit_code = test_tacsel2();
      break;
    default:
      break;
  }
        
  exit(exit_code);
}
/******************************************************************************/

/******************************************************************************/
int test_par1()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret;
  
  /* Initiate PETPAR struct */
  PETPAR p;
  parInit(&p);
  /* Allocate memory for it */
  ret=parAllocate(&p, 1); if(ret!=TACIO_OK) error_code=1;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  /* Free the memory */
  parEmpty(&p);
  /* Allocate memory again for it */
  ret=parAllocate(&p, 6); if(ret!=TACIO_OK) error_code=2;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  /* Set VoiNr, too */
  p.voiNr=6;
  /* Print it */
  parPrint(&p);
  /* Free the memory */
  parEmpty(&p);
  /* Try to print it */
  parPrint(&p);

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_par2()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret;

  /* Initiate PETPAR struct */
  PETPAR p;
  parInit(&p);
  /* Fill it with test contents */
  ret=create_par(&p); if(ret!=TACIO_OK) error_code=1;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  /* Print it */
  parPrint(&p);
  /* Free the memory */
  parEmpty(&p);

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_pario1()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret;
  char filename[FILENAME_MAX];
  char testdir[]="test";


  /* Create subdirectory for test data */
  if(access(testdir, 0) == -1) {
    if(VERBOSE) fprintf(stdout, "Creating subdirectory '%s'\n", testdir);
#ifdef WIN32
    ret=mkdir(testdir);
#else
    ret=mkdir(testdir, 00775);
#endif
    if(ret!=0) {
      fprintf(stderr, "  Error: cannot create test subdirectory.\n");
      fflush(stderr); error_code=-1;
      printf("\n %s FAILED with error code: %i\n", __func__, error_code);
      return(error_code);
    }
  }

  /* Initiate PETPAR struct */
  PETPAR p, p2;
  parInit(&p);
  //VERBOSE=30;

  /* Fill it with test contents */
  ret=create_par(&p); if(ret!=TACIO_OK) error_code=1;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Save it in CSV UK format */
  p.format=PAR_FORMAT_CSV_UK;
  sprintf(filename, "%s\\testdata1.csv", testdir);
  ret=parWrite(&p, filename, TAC_FORMAT_UNKNOWN, VERBOSE);
  if(ret!=TACIO_OK) error_code=11;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Read the file */
  parInit(&p2);
  ret=parRead(&p2, filename, VERBOSE); if(ret!=TACIO_OK) error_code=12;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Check that data has not changed */
  ret=parMatch(&p, &p2, VERBOSE); if(ret!=0) error_code=13;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Free the memory */
  parEmpty(&p2);

  /* Save data in CSV INTL format */
  p.format=PAR_FORMAT_CSV_INT;
  sprintf(filename, "%s\\testdata2.csv", testdir);
  ret=parWrite(&p, filename, TAC_FORMAT_UNKNOWN, VERBOSE);
  if(ret!=TACIO_OK) error_code=21;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Read the file */
  ret=parRead(&p2, filename, VERBOSE); if(ret!=TACIO_OK) error_code=22;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Check that data has not changed */
  ret=parMatch(&p, &p2, VERBOSE); if(ret!=0) error_code=23;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Free the memory */
  parEmpty(&p); parEmpty(&p2);

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_pario2()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret;
  char filename[FILENAME_MAX];
  char testdir[]="test";


  /* Create subdirectory for test data */
  if(access(testdir, 0) == -1) {
    if(VERBOSE) fprintf(stdout, "Creating subdirectory '%s'\n", testdir);
#ifdef WIN32
    ret=mkdir(testdir);
#else
    ret=mkdir(testdir, 00775);
#endif
    if(ret!=0) {
      fprintf(stderr, "  Error: cannot create test subdirectory.\n");
      fflush(stderr); error_code=-1;
      printf("\n %s FAILED with error code: %i\n", __func__, error_code);
      return(error_code);
    }
  }

  /* Initiate PETPAR struct */
  PETPAR p, p2;
  parInit(&p);
  VERBOSE=30;

  /* Fill it with test contents */
  ret=create_par_for_fit(&p); if(ret!=TACIO_OK) error_code=1;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Save it in FIT format */
  p.format=PAR_FORMAT_FIT;
  sprintf(filename, "%s\\testdata2.fit", testdir);
  ret=parWrite(&p, filename, TAC_FORMAT_UNKNOWN, VERBOSE);
  if(ret!=TACIO_OK) error_code=11;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Read the file */
  parInit(&p2);
  ret=parRead(&p2, filename, VERBOSE); if(ret!=TACIO_OK) error_code=12;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Check that data has not changed */
  ret=parMatch(&p, &p2, VERBOSE); if(ret!=0) error_code=13;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Free the memory */
  parEmpty(&p); parEmpty(&p2);

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_pario3()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret;
  int pi;
  char filename[FILENAME_MAX];
  char testdir[]="test";


  /* Create subdirectory for test data */
  if(access(testdir, 0) == -1) {
    if(VERBOSE) fprintf(stdout, "Creating subdirectory '%s'\n", testdir);
#ifdef WIN32
    ret=mkdir(testdir);
#else
    ret=mkdir(testdir, 00775);
#endif
    if(ret!=0) {
      fprintf(stderr, "  Error: cannot create test subdirectory.\n");
      fflush(stderr); error_code=-1;
      printf("\n %s FAILED with error code: %i\n", __func__, error_code);
      return(error_code);
    }
  }

  /* Initiate PETPAR struct */
  PETPAR p, p2;
  parInit(&p);
  //VERBOSE=30;

  /* Fill it with test contents */
  ret=create_par_for_res(&p); if(ret!=TACIO_OK) error_code=1;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  for(pi=0; pi<p.voiNr; pi++) {
    p.voi[pi].start=p.voi[pi].end=nan("");
    p.voi[pi].parNr=0; p.voi[pi].dataNr=0;
  }

  /* Save it in RES format */
  p.format=PAR_FORMAT_RES;
  sprintf(filename, "%s\\testdata3.res", testdir);
  ret=parWrite(&p, filename, TAC_FORMAT_UNKNOWN, VERBOSE);
  if(ret!=TACIO_OK) error_code=11;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Read the file */
  parInit(&p2);
  ret=parRead(&p2, filename, VERBOSE); if(ret!=TACIO_OK) error_code=12;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Check that data has not changed */
  ret=parMatch(&p, &p2, VERBOSE); if(ret!=0) error_code=13;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

#if(0)
  /* Save it in RES HTML format */
  p.format=PAR_FORMAT_RES;
  sprintf(filename, "%s\\testdata3.htm", testdir);
  ret=parWrite(&p, filename, TAC_FORMAT_UNKNOWN, VERBOSE);
  if(ret!=TACIO_OK) error_code=11;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  /* Its ok if we did not get any errors */
#endif

  /* Free the memory */
  parEmpty(&p); parEmpty(&p2);

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_pario4()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;

  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }


  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_tacio1()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret;
  char filename[FILENAME_MAX];
  char testdir[]="test";


  /* Create subdirectory for test data */
  if(access(testdir, 0) == -1) {
    if(VERBOSE) fprintf(stdout, "Creating subdirectory '%s'\n", testdir);
#ifdef WIN32
    ret=mkdir(testdir);
#else
    ret=mkdir(testdir, 00775);
#endif
    if(ret!=0) {
      fprintf(stderr, "  Error: cannot create test subdirectory.\n");
      fflush(stderr); error_code=-1;
      printf("\n %s FAILED with error code: %i\n", __func__, error_code);
      return(error_code);
    }
  }

  /* Initiate TAC struct */
  TAC tac1, tac2;
  tacInit(&tac1); tacInit(&tac2);
  //VERBOSE=30;

  /* Fill it with test contents */
  ret=create_tac(&tac1); if(ret!=TACIO_OK) error_code=1;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  tac1.isframe=0;

  /* Save it in Simple format */
  tac1.format=TAC_FORMAT_SIMPLE;
  sprintf(filename, "%s\\testdata1.dat", testdir);
  ret=tacWrite(&tac1, filename, TAC_FORMAT_UNKNOWN, 0, VERBOSE);
  if(ret!=TACIO_OK) error_code=11;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Read the file */
  ret=tacRead(&tac2, filename, VERBOSE); if(ret!=TACIO_OK) error_code=12;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Check that data has not changed */
  error_code=100;
  //if(tacMatchUnit(&tac1, &tac2, VERBOSE)!=0) error_code+=1;
  //if(tacMatchTacID(&tac1, &tac2, 2, VERBOSE)!=0) error_code+=2;
  if(tacMatchConc(&tac1, &tac2, 1.0E-03, VERBOSE)!=0) error_code+=4;
  if(tacMatchFrames(&tac1, &tac2, 1.0E-03, VERBOSE)!=0) error_code+=8;
  if(error_code==100) error_code=0;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Free the memory */
  tacEmpty(&tac2);


  /* Free the memory */
  tacEmpty(&tac1); tacEmpty(&tac2);

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_tacio2()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret;
  char filename[FILENAME_MAX];
  char testdir[]="test";


  /* Create subdirectory for test data */
  if(access(testdir, 0) == -1) {
    if(VERBOSE) fprintf(stdout, "Creating subdirectory '%s'\n", testdir);
#ifdef WIN32
    ret=mkdir(testdir);
#else
    ret=mkdir(testdir, 00775);
#endif
    if(ret!=0) {
      fprintf(stderr, "  Error: cannot create test subdirectory.\n");
      fflush(stderr); error_code=-1;
      printf("\n %s FAILED with error code: %i\n", __func__, error_code);
      return(error_code);
    }
  }

  /* Initiate TAC struct */
  TAC tac1, tac2;
  tacInit(&tac1); tacInit(&tac2);
  //VERBOSE=30;

  /* Fill it with test contents */
  ret=create_tac(&tac1); if(ret!=TACIO_OK) error_code=1;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Save it in DFT format */
  tac1.format=TAC_FORMAT_DFT;
  sprintf(filename, "%s\\testdata1.dft", testdir);
  ret=tacWrite(&tac1, filename, TAC_FORMAT_DFT, 0, VERBOSE);
  if(ret!=TACIO_OK) error_code=11;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Read the file */
  ret=tacRead(&tac2, filename, VERBOSE); if(ret!=TACIO_OK) error_code=12;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Check that data has not changed */
  error_code=100;
  if(tacMatchUnit(&tac1, &tac2, VERBOSE)!=0) error_code+=1;
  if(tacMatchTacID(&tac1, &tac2, 2, VERBOSE)!=0) error_code+=2;
  if(tacMatchConc(&tac1, &tac2, 1.0E-03, VERBOSE)!=0) error_code+=4;
  if(tacMatchFrames(&tac1, &tac2, 1.0E-03, VERBOSE)!=0) error_code+=8;
  if(error_code==100) error_code=0;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Free the memory */
  tacEmpty(&tac2);


  /* Free the memory */
  tacEmpty(&tac1); tacEmpty(&tac2);

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_tacio3()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret;
  char filename[FILENAME_MAX];
  char testdir[]="test";


  /* Create subdirectory for test data */
  if(access(testdir, 0) == -1) {
    if(VERBOSE) fprintf(stdout, "Creating subdirectory '%s'\n", testdir);
#ifdef WIN32
    ret=mkdir(testdir);
#else
    ret=mkdir(testdir, 00775);
#endif
    if(ret!=0) {
      fprintf(stderr, "  Error: cannot create test subdirectory.\n");
      fflush(stderr); error_code=-1;
      printf("\n %s FAILED with error code: %i\n", __func__, error_code);
      return(error_code);
    }
  }

  /* Initiate TAC struct */
  TAC tac1, tac2;
  tacInit(&tac1); tacInit(&tac2);
  //VERBOSE=10;

  /* Fill it with test contents */
  ret=create_tac(&tac1); if(ret!=TACIO_OK) error_code=1;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Save it in PMOD format with frame start and end times */
  tac1.format=TAC_FORMAT_PMOD;
  tac1.isframe=1;
  sprintf(filename, "%s\\testdata1.tac", testdir);
  ret=tacWrite(&tac1, filename, TAC_FORMAT_PMOD, 0, VERBOSE);
  if(ret!=TACIO_OK) error_code=11;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Read the file */
  ret=tacRead(&tac2, filename, VERBOSE); if(ret!=TACIO_OK) error_code=12;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Check that data has not changed */
  error_code=100;
  if(tacMatchUnit(&tac1, &tac2, VERBOSE)!=0) error_code+=1;
  if(tacMatchTacID(&tac1, &tac2, 0, VERBOSE)!=0) error_code+=2;
  if(tacMatchConc(&tac1, &tac2, 1.0E-03, VERBOSE)!=0) error_code+=4;
  if(tacMatchFrames(&tac1, &tac2, 1.0E-03, VERBOSE)!=0) error_code+=8;
  if(error_code==100) error_code=0;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Free the memory */
  tacEmpty(&tac2);


  /* Save it in PMOD format with frame middle times */
  tac1.format=TAC_FORMAT_PMOD;
  tac1.isframe=0;
  sprintf(filename, "%s\\testdata2.tac", testdir);
  ret=tacWrite(&tac1, filename, TAC_FORMAT_PMOD, 0, VERBOSE);
  if(ret!=TACIO_OK) error_code=11;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Read the file */
  ret=tacRead(&tac2, filename, VERBOSE); if(ret!=TACIO_OK) error_code=12;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Check that data has not changed */
  error_code=100;
  if(tacMatchUnit(&tac1, &tac2, VERBOSE)!=0) error_code+=1;
  if(tacMatchTacID(&tac1, &tac2, 0, VERBOSE)!=0) error_code+=2;
  if(tacMatchConc(&tac1, &tac2, 1.0E-03, VERBOSE)!=0) error_code+=4;
  if(tacMatchFrames(&tac1, &tac2, 1.0E-03, VERBOSE)!=0) error_code+=8;
  if(error_code==100) error_code=0;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Free the memory */
  tacEmpty(&tac2);


  /* Free the memory */
  tacEmpty(&tac1); tacEmpty(&tac2);

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_tacio4()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret;
  char filename[FILENAME_MAX];
  char testdir[]="test";


  /* Create subdirectory for test data */
  if(access(testdir, 0) == -1) {
    if(VERBOSE) fprintf(stdout, "Creating subdirectory '%s'\n", testdir);
#ifdef WIN32
    ret=mkdir(testdir);
#else
    ret=mkdir(testdir, 00775);
#endif
    if(ret!=0) {
      fprintf(stderr, "  Error: cannot create test subdirectory.\n");
      fflush(stderr); error_code=-1;
      printf("\n %s FAILED with error code: %i\n", __func__, error_code);
      return(error_code);
    }
  }

  /* Initiate TAC struct */
  TAC tac1, tac2;
  tacInit(&tac1); tacInit(&tac2);
  //VERBOSE=30;

  /* Fill it with test contents */
  ret=create_tac(&tac1); if(ret!=TACIO_OK) error_code=1;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }


  /* Set format */
  tac1.format=TAC_FORMAT_CSV_INT;
  tac1.isframe=0;

  /* Save it */
  sprintf(filename, "%s\\testdata1.csv", testdir);
  ret=tacWrite(&tac1, filename, TAC_FORMAT_UNKNOWN, 0, VERBOSE);
  if(ret!=TACIO_OK) {
    tacEmpty(&tac1); tacEmpty(&tac2); error_code=100;
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }


  /* Set format */
  tac1.format=TAC_FORMAT_CSV_INT;
  tac1.isframe=1;

  /* Save it */
  sprintf(filename, "%s\\testdata2.csv", testdir);
  ret=tacWrite(&tac1, filename, TAC_FORMAT_UNKNOWN, 0, VERBOSE);
  if(ret!=TACIO_OK) {
    tacEmpty(&tac1); tacEmpty(&tac2); error_code=100;
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }


  /* Set format */
  tac1.format=TAC_FORMAT_TSV_UK;
  tac1.isframe=1;

  /* Save it */
  sprintf(filename, "%s\\testdata3.tsv", testdir);
  ret=tacWrite(&tac1, filename, TAC_FORMAT_UNKNOWN, 0, VERBOSE);
  if(ret!=TACIO_OK) {
    tacEmpty(&tac1); tacEmpty(&tac2); error_code=100;
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }



  /* Free the memory */
  tacEmpty(&tac1); tacEmpty(&tac2);

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_tacid1()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;

  /* Testing tacNameSplit */
  TACID id;
  int i, n;
  
  strcpy(id.name, "region");
  n=tacNameSplit(&id);
  if(VERBOSE) {
    printf("'%s' ->", id.name); for(i=0; i<n; i++) printf(" '%s'", id.sub[i]);
    printf("\n");
  }
  if(n!=1) error_code=10;
  else if(strcmp(id.sub[0], "region")!=0) error_code=11;
  else if(strcmp(id.sub[1], "")!=0) error_code=12;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  
  strcpy(id.name, "region dx pl10");
  n=tacNameSplit(&id);
  if(VERBOSE) {
    printf("'%s' ->", id.name); for(i=0; i<n; i++) printf(" '%s'", id.sub[i]);
    printf("\n");
  }
  if(n!=3) error_code=10;
  else if(strcmp(id.sub[0], "region")!=0) error_code=11;
  else if(strcmp(id.sub[1], "dx")!=0) error_code=12;
  else if(strcmp(id.sub[2], "pl10")!=0) error_code=13;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  
  strcpy(id.name, "region . pl10");
  n=tacNameSplit(&id);
  if(VERBOSE) {
    printf("'%s' ->", id.name); for(i=0; i<n; i++) printf(" '%s'", id.sub[i]);
    printf("\n");
  }
  if(n!=3) error_code=10;
  else if(strcmp(id.sub[0], "region")!=0) error_code=11;
  else if(strcmp(id.sub[1], "")!=0) error_code=12;
  else if(strcmp(id.sub[2], "pl10")!=0) error_code=13;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  
  strcpy(id.name, "region_sin");
  n=tacNameSplit(&id);
  if(VERBOSE) {
    printf("'%s' ->", id.name); for(i=0; i<n; i++) printf(" '%s'", id.sub[i]);
    printf("\n");
  }
  if(n!=2) error_code=10;
  else if(strcmp(id.sub[0], "region")!=0) error_code=11;
  else if(strcmp(id.sub[1], "sin")!=0) error_code=12;
  else if(strcmp(id.sub[2], "")!=0) error_code=13;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  
  strcpy(id.name, "QuiteLongName-dx 100");
  n=tacNameSplit(&id);
  if(VERBOSE) {
    printf("'%s' ->", id.name); for(i=0; i<n; i++) printf(" '%s'", id.sub[i]);
    printf("\n");
  }
  if(n!=3) error_code=10;
  else if(strcmp(id.sub[0], "QuiteLongName")!=0) error_code=11;
  else if(strcmp(id.sub[1], "dx")!=0) error_code=12;
  else if(strcmp(id.sub[2], "100")!=0) error_code=13;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  
  strcpy(id.name, ". sin Mean");
  n=tacNameSplit(&id);
  if(VERBOSE) {
    printf("'%s' ->", id.name); for(i=0; i<n; i++) printf(" '%s'", id.sub[i]);
    printf("\n");
  }
  if(n!=3) error_code=10;
  else if(strcmp(id.sub[0], "")!=0) error_code=11;
  else if(strcmp(id.sub[1], "sin")!=0) error_code=12;
  else if(strcmp(id.sub[2], "Mean")!=0) error_code=13;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  
  strcpy(id.name, "_sin_Mean");
  n=tacNameSplit(&id);
  if(VERBOSE) {
    printf("'%s' ->", id.name); for(i=0; i<n; i++) printf(" '%s'", id.sub[i]);
    printf("\n");
  }
  if(n!=3) error_code=10;
  else if(strcmp(id.sub[0], "")!=0) error_code=11;
  else if(strcmp(id.sub[1], "sin")!=0) error_code=12;
  else if(strcmp(id.sub[2], "Mean")!=0) error_code=13;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  
  strcpy(id.name, "__Mean");
  n=tacNameSplit(&id);
  if(VERBOSE) {
    printf("'%s' ->", id.name); for(i=0; i<n; i++) printf(" '%s'", id.sub[i]);
    printf("\n");
  }
  if(n!=3) error_code=10;
  else if(strcmp(id.sub[0], "")!=0) error_code=11;
  else if(strcmp(id.sub[1], "")!=0) error_code=12;
  else if(strcmp(id.sub[2], "Mean")!=0) error_code=13;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_tacid2()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;

  /* Testing tacNameMatch */
  TACID id;
  int n;
  char test_str[128];
  
  strcpy(id.name, ""); strcpy(test_str, "");
  n=tacNameMatch(&id, test_str, 0);
  if(VERBOSE) printf("tacNameMatch('%s', '%s') := %d\n", id.name, test_str, n);
  if(!n) error_code+=1; 
  
  strcpy(id.name, "region"); strcpy(test_str, "tappi");
  n=tacNameMatch(&id, test_str, 0);
  if(VERBOSE) printf("tacNameMatch('%s', '%s') := %d\n", id.name, test_str, n);
  if(n) error_code+=2; 
  
  strcpy(id.name, "region"); strcpy(test_str, "*");
  n=tacNameMatch(&id, test_str, 0);
  if(VERBOSE) printf("tacNameMatch('%s', '%s') := %d\n", id.name, test_str, n);
  if(!n) error_code+=4; 
  
  strcpy(id.name, "region"); strcpy(test_str, "reg*");
  n=tacNameMatch(&id, test_str, 0);
  if(VERBOSE) printf("tacNameMatch('%s', '%s') := %d\n", id.name, test_str, n);
  if(!n) error_code+=8; 
  
  strcpy(id.name, "region_dx"); strcpy(test_str, "region");
  n=tacNameMatch(&id, test_str, 0);
  if(VERBOSE) printf("tacNameMatch('%s', '%s') := %d\n", id.name, test_str, n);
  if(!n) error_code+=16; 
  
  strcpy(id.name, "region_dx"); strcpy(test_str, "dx");
  n=tacNameMatch(&id, test_str, 0);
  if(VERBOSE) printf("tacNameMatch('%s', '%s') := %d\n", id.name, test_str, n);
  if(!n) error_code+=32; 
  
  strcpy(id.name, "region_dx_Mean"); strcpy(test_str, "mean");
  n=tacNameMatch(&id, test_str, 0);
  if(VERBOSE) printf("tacNameMatch('%s', '%s') := %d\n", id.name, test_str, n);
  if(!n) error_code+=64; 
  
  strcpy(id.name, "region_dx_mean"); strcpy(test_str, "Mean");
  n=tacNameMatch(&id, test_str, 0);
  if(VERBOSE) printf("tacNameMatch('%s', '%s') := %d\n", id.name, test_str, n);
  if(!n) error_code+=128; 
  
  strcpy(id.name, "region_sin_Mean"); strcpy(test_str, "region_sin_mean");
  n=tacNameMatch(&id, test_str, 0);
  if(VERBOSE) printf("tacNameMatch('%s', '%s') := %d\n", id.name, test_str, n);
  if(!n) error_code+=256; 
  
  strcpy(id.name, "region_sin_Mean"); strcpy(test_str, "region_*_mean");
  n=tacNameMatch(&id, test_str, 0);
  if(VERBOSE) printf("tacNameMatch('%s', '%s') := %d\n", id.name, test_str, n);
  if(!n) error_code+=512; 
  
  strcpy(id.name, "region_sin_Mean"); strcpy(test_str, "region * mean");
  n=tacNameMatch(&id, test_str, 0);
  if(VERBOSE) printf("tacNameMatch('%s', '%s') := %d\n", id.name, test_str, n);
  if(!n) error_code+=1024; 


  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  } else
    if(VERBOSE) printf("first set of tests passed.\n");

  
  strcpy(id.name, "region__Mean"); strcpy(test_str, "region . mean");
  n=tacNameMatch(&id, test_str, 0);
  if(VERBOSE) printf("tacNameMatch('%s', '%s') := %d\n", id.name, test_str, n);
  if(!n) error_code+=1; 
  
  strcpy(id.name, "region_._Mean"); strcpy(test_str, "region . mean");
  n=tacNameMatch(&id, test_str, 0);
  if(VERBOSE) printf("tacNameMatch('%s', '%s') := %d\n", id.name, test_str, n);
  if(!n) error_code+=2; 
  
  strcpy(id.name, "region___"); strcpy(test_str, "region . mean");
  n=tacNameMatch(&id, test_str, 0);
  if(VERBOSE) printf("tacNameMatch('%s', '%s') := %d\n", id.name, test_str, n);
  if(n) error_code+=4; 
  
    
    
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_tacnan1()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret;


  /* Initiate TAC struct */
  TAC tac1, tac2;
  tacInit(&tac1); tacInit(&tac2);
  //VERBOSE=30;

  /* Fill it with test contents */
  ret=create_tac(&tac1); if(ret==TACIO_OK) create_tac(&tac2);
  if(ret!=TACIO_OK) error_code=-1;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  
  /* Now there should be no NaN's */
  if(tacNaNs(&tac1)!=0) error_code+=1;
  
  /* Set mid sample time to NaN */
  tac1.x[1]=nan("");
  tac1.isframe=0; if(tacNaNs(&tac1)!=1) error_code+=2;
  tac1.isframe=1; if(tacNaNs(&tac1)!=0) error_code+=4;
  tac1.x[1]=tac2.x[1];
  
  /* Set frame start and end time to NaN */
  tac1.x1[1]=tac1.x2[0]=nan("");
  tac1.isframe=0; if(tacNaNs(&tac1)!=0) error_code+=8;
  tac1.isframe=1; if(tacNaNs(&tac1)!=2) error_code+=16;
  tac1.x1[1]=tac2.x1[1]; tac1.x2[0]=tac2.x2[0];

  /* Set one y value to NaN */
  tac1.voi[1].y[1]=nan("");
  if(tacNaNs(&tac1)!=1) error_code+=32;
  
  /* Free the memory */
  tacEmpty(&tac1); tacEmpty(&tac2);

  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_tacsort1()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret;


  /* Initiate TAC struct */
  TAC tac1, tac2;
  tacInit(&tac1); tacInit(&tac2);
  //VERBOSE=30;

  /* Fill it with test contents */
  ret=create_tac(&tac1); if(ret==TACIO_OK) create_tac(&tac2);
  if(ret!=TACIO_OK) error_code=-1;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Data is sorted already, nothing should be changed */
  ret=tacSortByFrame(&tac1);
  if(ret!=TACIO_OK ||
     tacMatchFrames(&tac1, &tac2, 1.0E-20, VERBOSE)!=0 ||
     tacMatchConc(&tac1, &tac2, 1.0E-20, VERBOSE)!=0)
  {
    error_code=1;
    tacEmpty(&tac1); tacEmpty(&tac2);
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  
  /* Sorting by Frame times must give an error with missing frame times */
  /* Set some sample times to NaN */
  tac1.x[1]=nan(""); tac1.x1[1]=tac1.x2[0]=nan("");
  ret=tacSortByFrame(&tac1);
  if(ret!=TACIO_NAN) {
    error_code=2;
    tacEmpty(&tac1); tacEmpty(&tac2);
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  tac1.x[1]=tac2.x[1]; tac1.x1[1]=tac2.x1[1]; tac1.x2[0]=tac2.x2[0];

  /* Change the order of sample times, and sort it back */
  tac1.x[0]=tac2.x[2];   tac1.x[2]=tac2.x[0];
  tac1.x1[0]=tac2.x1[2]; tac1.x1[2]=tac2.x1[0];
  tac1.x2[0]=tac2.x2[2]; tac1.x2[2]=tac2.x2[0];
  for(int ri=0; ri<tac1.voiNr; ri++) {
    tac1.voi[ri].y[0]=tac2.voi[ri].y[2]; tac1.voi[ri].y[2]=tac2.voi[ri].y[0];
    tac1.w[0]=tac2.w[2]; tac1.w[2]=tac2.w[0];
  }
  tac1.isframe=1; tac2.isframe=1;
  ret=tacSortByFrame(&tac1);
  if(ret!=TACIO_OK ||
     tacMatchFrames(&tac1, &tac2, 1.0E-20, VERBOSE)!=0 ||
     tacMatchConc(&tac1, &tac2, 1.0E-20, VERBOSE)!=0)
  {
    error_code=3;
    tacEmpty(&tac1); tacEmpty(&tac2);
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  /* again with frame mid times */
  tac1.x[0]=tac2.x[2];   tac1.x[2]=tac2.x[0];
  tac1.x1[0]=tac2.x1[2]; tac1.x1[2]=tac2.x1[0];
  tac1.x2[0]=tac2.x2[2]; tac1.x2[2]=tac2.x2[0];
  for(int ri=0; ri<tac1.voiNr; ri++) {
    tac1.voi[ri].y[0]=tac2.voi[ri].y[2]; tac1.voi[ri].y[2]=tac2.voi[ri].y[0];
    tac1.w[0]=tac2.w[2]; tac1.w[2]=tac2.w[0];
  }
  tac1.isframe=0; tac2.isframe=0;
  ret=tacSortByFrame(&tac1);
  if(ret!=TACIO_OK ||
     tacMatchFrames(&tac1, &tac2, 1.0E-20, VERBOSE)!=0 ||
     tacMatchConc(&tac1, &tac2, 1.0E-20, VERBOSE)!=0)
  {
    error_code=4;
    tacEmpty(&tac1); tacEmpty(&tac2);
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Free the memory */
  tacEmpty(&tac1); tacEmpty(&tac2);

  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_tacsort2()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret;


  /* Initiate TAC struct */
  TAC tac1, tac2;
  tacInit(&tac1); tacInit(&tac2);
  //VERBOSE=30;

  /* Fill it with test contents */
  ret=create_tac(&tac1); if(ret==TACIO_OK) create_tac(&tac2);
  if(ret!=TACIO_OK) error_code=-1;
  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  if(VERBOSE) {
    tacWritePmod(&tac1, stdout, 0);
  }

  /* Data is sorted already, nothing should be changed */
  ret=tacSortByName(&tac1);
  if(ret!=TACIO_OK ||
     tacMatchFrames(&tac1, &tac2, 1.0E-20, VERBOSE)!=0 ||
     tacMatchConc(&tac1, &tac2, 1.0E-20, VERBOSE)!=0 ||
     tacMatchTacID(&tac1, &tac2, 0, VERBOSE)!=0)
  {
    error_code=1;
    tacEmpty(&tac1); tacEmpty(&tac2);
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  /* Change the order of 2nd and 3rd VOI, and sort back */
  ret=tacCopyTacvoi(tac2.voi+1, tac1.voi+2, tac1.frameNr);
  if(ret==TACIO_OK) ret=tacCopyTacvoi(tac2.voi+2, tac1.voi+1, tac1.frameNr);
  if(ret!=TACIO_OK) error_code=-2;
  if(error_code) {
    tacEmpty(&tac1); tacEmpty(&tac2);
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  if(VERBOSE) {
    tacWritePmod(&tac1, stdout, 0);
  }
  ret=tacSortByName(&tac1);
  if(ret!=TACIO_OK ||
     tacMatchFrames(&tac1, &tac2, 1.0E-20, VERBOSE)!=0 ||
     tacMatchConc(&tac1, &tac2, 1.0E-20, VERBOSE)!=0 ||
     tacMatchTacID(&tac1, &tac2, 0, VERBOSE)!=0)
  {
    error_code=2;
    tacEmpty(&tac1); tacEmpty(&tac2);
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }
  if(VERBOSE) {
    tacWritePmod(&tac1, stdout, 0);
  }


  /* Free the memory */
  tacEmpty(&tac1); tacEmpty(&tac2);

  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_tacsel1()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret, ri, n;
  char rname[128];


  /* Initiate TAC struct */
  TAC tac;
  tacInit(&tac);
  //VERBOSE=30;

  /* Fill it with test contents */
  ret=tacSetmem(&tac, 1, 10); if(ret!=TACIO_OK) return ret;
  tac.voiNr=10; tac.frameNr=1;

  /* Set region names */
  ri=0; strcpy(tac.voi[ri].id.name, "str_dx");
  ri=1; strcpy(tac.voi[ri].id.name, "str_sin");
  ri=2; strcpy(tac.voi[ri].id.name, "cau_dx");
  ri=3; strcpy(tac.voi[ri].id.name, "cau_sin");
  ri=4; strcpy(tac.voi[ri].id.name, "put_dx");
  ri=5; strcpy(tac.voi[ri].id.name, "put_sin");
  ri=6; strcpy(tac.voi[ri].id.name, "cer_dx");
  ri=7; strcpy(tac.voi[ri].id.name, "cer_sin");
  ri=8; strcpy(tac.voi[ri].id.name, "cer");
  ri=9; strcpy(tac.voi[ri].id.name, "wm");

  /* Select dx */
  strcpy(rname, "dx");
  n=tacSelectVOIs(&tac, rname, 1);
  if(VERBOSE) printf("tacSelectVOIs(tac, '%s', 1) := %d\n", rname, n);
  if(n!=4) error_code+=1; 
  
  /* Select sin in addition */
  strcpy(rname, "sin");
  n=tacSelectVOIs(&tac, rname, 0);
  if(VERBOSE) printf("tacSelectVOIs(tac, '%s', 0) := %d\n", rname, n);
  if(n!=4) error_code+=2;
  n=tacSelectedVOIs(&tac);
  if(VERBOSE) printf("tacSelectedVOIs(tac) := %d\n", n);
  if(n!=8) error_code+=4;

  /* Try to select something that does not exist */
  strcpy(rname, "Nemo");
  n=tacSelectVOIs(&tac, rname, 1);
  if(VERBOSE) printf("tacSelectVOIs(tac, '%s', 1) := %d\n", rname, n);
  if(n!=0) error_code+=8; 

  /* Select regions starting with c */
  strcpy(rname, "C*");
  n=tacSelectVOIs(&tac, rname, 1);
  if(VERBOSE) printf("tacSelectVOIs(tac, '%s', 1) := %d\n", rname, n);
  if(n!=5) error_code+=16; 

  /* Region name inside quotation marks */
  strcpy(rname, "'C*'");
  n=tacSelectVOIs(&tac, rname, 1);
  if(VERBOSE) printf("tacSelectVOIs(tac, '%s', 1) := %d\n", rname, n);
  if(n!=5) error_code+=32; 

  /* Region number as string */
  strcpy(rname, "3");
  n=tacSelectVOIs(&tac, rname, 1);
  if(VERBOSE) printf("tacSelectVOIs(tac, '%s', 1) := %d\n", rname, n);
  if(n!=1 || tac.voi[2].id.sw!=1) error_code+=64; 

  /* Free the memory */
  tacEmpty(&tac);

  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_tacsel2()
{
#if __STDC_VERSION__ < 199901L
# if __GNUC__ >= 2
#  define __func__ __FUNCTION__
# else
#  define __func__ "<unknown>"
# endif
#endif
  printf("\n=====================================\n");
  printf("\n%s\n", __func__);
  printf("\n=====================================\n");

  int error_code=0;
  int ret, ri, n;
  char rname[128];


  /* Initiate TAC struct */
  TAC tac;
  tacInit(&tac);
  //VERBOSE=30;

  /* Fill it with test contents */
  ret=tacSetmem(&tac, 1, 10); if(ret!=TACIO_OK) return ret;
  tac.voiNr=10; tac.frameNr=1;

  /* Set region names */
  ri=0; strcpy(tac.voi[ri].id.name, "str_dx");
  ri=1; strcpy(tac.voi[ri].id.name, "str_sin");
  ri=2; strcpy(tac.voi[ri].id.name, "cau_dx");
  ri=3; strcpy(tac.voi[ri].id.name, "cau_sin");
  ri=4; strcpy(tac.voi[ri].id.name, "put_dx");
  ri=5; strcpy(tac.voi[ri].id.name, "put_sin");
  ri=6; strcpy(tac.voi[ri].id.name, "cer_dx");
  ri=7; strcpy(tac.voi[ri].id.name, "cer_sin");
  ri=8; strcpy(tac.voi[ri].id.name, "cer");
  ri=9; strcpy(tac.voi[ri].id.name, "wm");

  /* Select Cer */
  strcpy(rname, "cer");
  n=tacSelectVOIs(&tac, rname, 1);
  if(VERBOSE) printf("tacSelectVOIs(tac, '%s', 1) := %d\n", rname, n);
  if(n!=3) error_code+=1; 

  /* Select the best Cer TAC as reference */
  n=tacSelectBestReference(&tac);
  if(VERBOSE) printf("tacSelectBestReference(tac) := %d\n", n);
  if(n!=8) error_code+=2; 

  /* Select WM */
  strcpy(rname, "WM");
  n=tacSelectVOIs(&tac, rname, 1);
  if(VERBOSE) printf("tacSelectVOIs(tac, '%s', 1) := %d\n", rname, n);
  if(n!=1) error_code+=4; 

  /* Select the best WM TAC (the only) as reference */
  n=tacSelectBestReference(&tac);
  if(VERBOSE) printf("tacSelectBestReference(tac) := %d\n", n);
  if(n!=9) error_code+=8; 

  /* When none are selected, we should get an error */
  strcpy(rname, "Nemo");
  n=tacSelectVOIs(&tac, rname, 1);
  if(VERBOSE) printf("tacSelectVOIs(tac, '%s', 1) := %d\n", rname, n);
  if(n!=0) error_code+=16; 
  n=tacSelectBestReference(&tac);
  if(VERBOSE) printf("tacSelectBestReference(tac) := %d\n", n);
  if(n>=0) error_code+=32; 


  /* Free the memory */
  tacEmpty(&tac);

  if(error_code) {
    printf("\n %s FAILED with error code: %i\n", __func__, error_code);
    return(error_code);
  }

  printf("\n %s SUCCESFULL\n", __func__);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
/** Create test PETPAR data */
int create_par(
  PETPAR *par
) {
  int ret, pi, ri;
  
  /* Allocate memory */
  ret=parAllocate(par, 3); if(ret!=TACIO_OK) return ret;
  par->voiNr=3;
  /* Set header fields */
  strcpy(par->studynr, "test0001");
  par->time=time(NULL);
  strcpy(par->program, "libtpctacio");
  strcpy(par->datafile, "C:\\My Documents\\PETData\\imaginary.dat");
  par->weighting=WEIGHTING_OFF;
  par->cunit=CUNIT_KBQ_PER_ML;
  par->tunit=TUNIT_MIN;
  /* Set parameter names */
  par->parNr=5;
  pi=0; strcpy(par->parname[pi], "K1"); par->parunit[pi]=CUNIT_ML_PER_ML_PER_MIN;
  pi++; strcpy(par->parname[pi], "K1/k2"); par->parunit[pi]=CUNIT_UNITLESS;
  pi++; strcpy(par->parname[pi], "k3"); par->parunit[pi]=CUNIT_PER_MIN;
  pi++; strcpy(par->parname[pi], "k4"); par->parunit[pi]=CUNIT_PER_MIN;
  pi++; strcpy(par->parname[pi], "Vb"); par->parunit[pi]=CUNIT_PERCENTAGE;
  //pi++; strcpy(par->parname[pi], "WSS"); par->parunit[pi]=CUNIT_UNITLESS;
  /* Set region names */
  for(ri=0; ri<par->voiNr; ri++) sprintf(par->voi[ri].id.name, "roi%d", 1+ri);
  /* Set regional fit options */
  for(ri=0; ri<par->voiNr; ri++) {
    par->voi[ri].parNr=4;
    par->voi[ri].dataNr=23;
    par->voi[ri].start=0.0;
    par->voi[ri].end=45.0;
    if(ri==0) par->voi[ri].wss=8.7654E+002;
    else par->voi[ri].wss=1.1*par->voi[ri-1].wss;
  }
  /* Set parameter values */
  ri=0; 
  pi=0; par->voi[ri].p[pi]=0.3;
  pi++; par->voi[ri].p[pi]=0.5;
  pi++; par->voi[ri].p[pi]=0.1;
  pi++; par->voi[ri].p[pi]=0.005;
  pi++; par->voi[ri].p[pi]=5.0;
  //pi++; par->voi[ri].p[pi]=9.8765E+002;
  for(ri=1; ri<par->voiNr; ri++)
    for(pi=0; pi<par->parNr; pi++)
      par->voi[ri].p[pi]=1.1*par->voi[ri-1].p[pi];
  /* Set SDs and CLs for certain parameters */
  for(pi=0; pi<par->parNr-1; pi++) {
    for(ri=0; ri<par->voiNr; ri++) {
      if(pi>0) par->voi[ri].sd[pi]=0.2*par->voi[ri].p[pi];
      if(pi==0) {
        par->voi[ri].cl1[pi]=0.9*par->voi[ri].p[pi];
        par->voi[ri].cl2[pi]=1.3*par->voi[ri].p[pi];
      }
      if(pi==2 || pi==3) {
        par->voi[ri].cl1[pi]=par->voi[ri].p[pi]-par->voi[ri].sd[pi];
        par->voi[ri].cl2[pi]=par->voi[ri].p[pi]+par->voi[ri].sd[pi];
      }
    }
  }
  
  return TACIO_OK;
}
/******************************************************************************/

/******************************************************************************/
/** Create PETPAR data for testing FIT struct */
int create_par_for_fit(
  PETPAR *par
) {
  int ret, pi, ri;
  
  /* Allocate memory */
  ret=parAllocate(par, 3); if(ret!=TACIO_OK) return ret;
  par->voiNr=3;
  /* Set header fields */
  par->time=time(NULL);
  strcpy(par->program, "libtpctacio");
  strcpy(par->datafile, "C:\\PETData\\imaginary.dat");
  par->weighting=WEIGHTING_UNKNOWN;
  par->cunit=CUNIT_KBQ_PER_ML;
  par->tunit=TUNIT_MIN;
  /* Set parameter names */
  par->parNr=4;
  for(pi=0; pi<par->parNr; pi++) sprintf(par->parname[pi], "p%d", pi+1);
  /* Set region names */
  for(ri=0; ri<par->voiNr; ri++) sprintf(par->voi[ri].id.name, "roi%d", 1+ri);
  /* Set regional fit options */
  for(ri=0; ri<par->voiNr; ri++) {
    par->voi[ri].dataNr=23;
    par->voi[ri].parNr=par->parNr;
    par->voi[ri].start=0.0;
    par->voi[ri].end=45.0;
    if(ri==0) par->voi[ri].wss=8.7654E+002;
    else par->voi[ri].wss=1.1*par->voi[ri-1].wss;
    /* Function id must be set */
    par->voi[ri].function=302;
  }
  /* Set parameter values */
  ri=0; 
  pi=0; par->voi[ri].p[pi]=0.3;
  pi++; par->voi[ri].p[pi]=0.5;
  pi++; par->voi[ri].p[pi]=0.1;
  pi++; par->voi[ri].p[pi]=0.005;
  for(ri=1; ri<par->voiNr; ri++)
    for(pi=0; pi<par->parNr; pi++)
      par->voi[ri].p[pi]=1.1*par->voi[ri-1].p[pi];
  
  return TACIO_OK;
}
/******************************************************************************/

/******************************************************************************/
/** Create PETPAR data for testing RES struct */
int create_par_for_res(
  PETPAR *par
) {
  int ret, pi, ri;
  
  /* Allocate memory */
  ret=parAllocate(par, 3); if(ret!=TACIO_OK) return ret;
  par->voiNr=3;
  par->parNr=5;
  /* Set header fields */
  strcpy(par->program, "libtpctacio");
  par->time=time(NULL);
  strcpy(par->studynr, "test0003");
  strcpy(par->datafile, "C:\\PETData\\imaginary.dat");
  strcpy(par->plasmafile, "C:\\PETData\\imaginary_ap.dat");
  strcpy(par->bloodfile, "C:\\PETData\\imaginary_ab.dat");
  par->weighting=WEIGHTING_ON;
  par->lc=1.0;
  par->Vb=5.0;
  /* Set parameter names */
  par->parNr=5;
  pi=0; strcpy(par->parname[pi], "K1"); par->parunit[pi]=CUNIT_ML_PER_ML_PER_MIN;
  pi++; strcpy(par->parname[pi], "K1/k2"); par->parunit[pi]=CUNIT_UNITLESS;
  pi++; strcpy(par->parname[pi], "k3"); par->parunit[pi]=CUNIT_PER_MIN;
  pi++; strcpy(par->parname[pi], "k4"); par->parunit[pi]=CUNIT_PER_MIN;
  pi++; strcpy(par->parname[pi], "Vb"); par->parunit[pi]=CUNIT_PERCENTAGE;
  /* Set region names */
  for(ri=0; ri<par->voiNr; ri++) sprintf(par->voi[ri].id.name, "roi%d", 1+ri);
  /* Set regional fit options */
  for(ri=0; ri<par->voiNr; ri++) {
    par->voi[ri].parNr=4;
    par->voi[ri].dataNr=23;
    par->voi[ri].start=0.0;
    par->voi[ri].end=45.0;
    if(ri==0) par->voi[ri].wss=8.7654E+002;
    else par->voi[ri].wss=1.1*par->voi[ri-1].wss;
  }
  /* Set parameter values */
  ri=0; 
  pi=0; par->voi[ri].p[pi]=0.3;
  pi++; par->voi[ri].p[pi]=0.5;
  pi++; par->voi[ri].p[pi]=0.1;
  pi++; par->voi[ri].p[pi]=0.005;
  pi++; par->voi[ri].p[pi]=5.0;
  for(ri=1; ri<par->voiNr; ri++)
    for(pi=0; pi<par->parNr; pi++)
      par->voi[ri].p[pi]=1.1*par->voi[ri-1].p[pi];
  /* Set SDs and CLs for certain parameters */
  for(pi=0; pi<par->parNr-1; pi++) {
    for(ri=0; ri<par->voiNr; ri++) {
      if(pi>0) par->voi[ri].sd[pi]=0.2*par->voi[ri].p[pi];
      if(pi==0) {
        par->voi[ri].cl1[pi]=0.9*par->voi[ri].p[pi];
        par->voi[ri].cl2[pi]=1.3*par->voi[ri].p[pi];
      }
      if(pi==2 || pi==3) {
        par->voi[ri].cl1[pi]=par->voi[ri].p[pi]-par->voi[ri].sd[pi];
        par->voi[ri].cl2[pi]=par->voi[ri].p[pi]+par->voi[ri].sd[pi];
      }
    }
  }
  
  return TACIO_OK;
}
/******************************************************************************/

/******************************************************************************/
/** Create test TAC data */
int create_tac(
  TAC *tac
) {
  int ret, fi, ri;
  
  /* Allocate memory */
  ret=tacSetmem(tac, 5, 3); if(ret!=TACIO_OK) return ret;
  tac->voiNr=3; tac->frameNr=5;
  /* Set header fields */
  strcpy(tac->studynr, "test0001");
  tac->scantime=time(NULL);
  tac->weighting=WEIGHTING_OFF;
  tac->cunit=CUNIT_KBQ_PER_ML;
  tac->tunit=TUNIT_MIN;
  tac->isframe=1;
  /* Set region names */
  for(ri=0; ri<tac->voiNr; ri++) sprintf(tac->voi[ri].id.name, "roi%d", 1+ri);
  /* Set data contents */
  for(fi=0; fi<tac->frameNr; fi++) {
    tac->x1[fi]=(double)fi; tac->x2[fi]=(double)(fi+1);
    tac->x[fi]=0.5*(tac->x1[fi]+tac->x2[fi]);
    for(ri=0; ri<tac->voiNr; ri++)
      tac->voi[ri].y[fi]=(double)((ri+1)*(fi+ri)-2);
  }
  /* Set volumes */
  for(ri=0; ri<tac->voiNr; ri++) tac->voi[ri].size=(double)(100*(ri+1));
  
  return TACIO_OK;
}
/******************************************************************************/

/******************************************************************************/
void print_usage(){

  libtpctacio_print_build(stdout);
  puts("\n"
  "  Methods for testing functions in the library "LIB_NAME"\n"
  "  and for printing out associated information, such as Readme,\n"
  "  History and build information. \n"
  "\n"
  "  Usage: "LIB_NAME" [-Options]\n"
  "\n"
  "  Options:\n"
  "  -h | H \n"
  "      Print out the "LIB_NAME" History message. Include Doxygen style tags\n"
  "      by using option '-H'.\n"
  "  -r | R \n"
  "      Print out the "LIB_NAME" Readme message. Include Doxygen style tags\n"
  "      by using option '-R'.\n"
  "  -b\n"
  "      Print out the "LIB_NAME" build information.\n"
  "  -f <function serial number>\n"
  "      Runs test procedures for the functions corresponding to given\n"
  "      'function serial number'. Serial numbers are defined in file\n"
  "      "LIB_NAME".c; enter 0 to execute them all.\n"
  "  -v\n"
  "      Run in verbose mode.\n"
  "\n"
  "  E.g.: "LIB_NAME" -h\n"
  "        "LIB_NAME" -v -f 1 \n"
  "\n"
  );
  fflush(stdout);
}
/******************************************************************************/

/******************************************************************************/
