/******************************************************************************
  Copyright (c) 2013 by Turku PET Centre

  File:        parconv.c
  Description: Functions for converting data in RES or FIT structs to
               PETPAR struct and vice versa.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Version history:
  2013-09-04 Vesa Oikonen
       First created.
     


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
//#include "curveio.h"
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/tacio.h"
/*****************************************************************************/

/*****************************************************************************/
#if(0)
/** Copy contents of FIT struct to PETPAR.
\return Returns PETPAR error code, PAR_OK (0) when successful.
 */ 
int parFit2Par(
  /** Pointer to filled FIT struct */
  FIT *fit,
  /** Pointer to initiated PETPAR struct */
  PETPAR *par,
  /** Verbose level; if zero, then only warnings are printed into stderr */
  int verbose
) {
  int pi, ri, n;
  
  if(verbose>0) printf("parFit2Par(fit, par, ...)\n");
  if(par==NULL || fit==NULL) return(PAR_ERROR);
  if(fit->voiNr<1) return(PAR_NOTABLE);

  /* Determine max parameter number in fits (used a bit later) */
  for(ri=0, n=0; ri<fit->voiNr; ri++)
    if(fit->voi[ri].parNr>n) n=fit->voi[ri].parNr;
  if(verbose>1) printf("maxParNr := %d\n", n);
  if(n<1) return(PAR_NOTABLE);
  if(n>MAX_PETPARAMS) return(PAR_TOOBIG);

  /* Allocate memory for par */
  parEmpty(par);
  if(parAllocate(par, fit->voiNr)!=0) return(PAR_OUTOFMEMORY);
  /* Copy header */
  par->fileformat=PETPAR_FIT;
  par->parNr=n; // n computed above, not needed after this
  par->voiNr=fit->voiNr;
  strcpy(par->datafile, fit->datafile);
  strcpy(par->studynr, fit->studynr);
  par->dataunit=petCunitId(fit->unit);
  par->timeunit=fit->timeunit;
  par->time=fit->time;
  if(strlen(fit->program)>0) strcpy(par->program, fit->program);
  /* Copy region names, etc */
  for(ri=0; ri<fit->voiNr; ri++) {
    rnameRmDots(fit->voi[ri].name, par->voi[ri].id.name);
    //strcpy(par->voi[ri].id.name, fit->voi[ri].name);
    par->voi[ri].function=fit->voi[ri].type;
    par->voi[ri].start=fit->voi[ri].start;
    par->voi[ri].end=fit->voi[ri].end;
    par->voi[ri].dataNr=fit->voi[ri].dataNr;
    //par->voi[ri].parNr=fit->voi[ri].parNr; // NOT applicable!
    par->voi[ri].wss=fit->voi[ri].wss;
  }
  /* Set parameter names */
  for(pi=0; pi<par->parNr; pi++) sprintf(par->parname[pi], "p%d", pi+1);
  /* Copy parameter values */
  for(ri=0; ri<par->voiNr; ri++) {
    for(pi=0; pi<par->parNr; pi++) {
      if(pi>=fit->voi[ri].parNr) par->voi[ri].p[pi]=nan("");
      else par->voi[ri].p[pi]=fit->voi[ri].p[pi];
    }
  }
  return(PAR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy contents of PETPAR struct to FIT.
\return Returns PETPAR error code, PAR_OK (0) when successful.
 */ 
int parPar2Fit(
  /** Pointer to filled PETPAR struct */
  PETPAR *par,
  /** Pointer to initiated FIT struct */
  FIT *fit,
  /** Verbose level; if zero, then only warnings are printed into stderr */
  int verbose
) {
  int pi, ri;
  
  if(verbose>0) printf("parPar2Fit(par, fit, %d)\n", verbose);
  if(par==NULL || fit==NULL) return(PAR_ERROR);
  if(par->voiNr<1 || par->parNr<1) return(PAR_NOTABLE);
  if(par->parNr>MAX_FITPARAMS) return(PAR_TOOBIG);

  /* Allocate memory for fit */
  fitEmpty(fit);
  if(fitSetmem(fit, par->voiNr)!=0) return(PAR_OUTOFMEMORY);
  /* Copy header */
  fit->voiNr=par->voiNr;
  strcpy(fit->datafile, par->datafile);
  strcpy(fit->studynr, par->studynr);
  strcpy(fit->unit, petCunit(par->dataunit));
  fit->timeunit=par->timeunit;
  fit->time=par->time;
  if(strlen(par->program)>0) strcpy(fit->program, par->program);
  /* Copy region names, etc */
  for(ri=0; ri<fit->voiNr; ri++) {
    strcpy(fit->voi[ri].name, par->voi[ri].id.name);
    rnameSplit(fit->voi[ri].name, fit->voi[ri].voiname, fit->voi[ri].hemisphere,
               fit->voi[ri].place, 6);
    fit->voi[ri].type=par->voi[ri].function;
    fit->voi[ri].start=par->voi[ri].start;
    fit->voi[ri].end=par->voi[ri].end;
    fit->voi[ri].dataNr=par->voi[ri].dataNr;
    fit->voi[ri].wss=par->voi[ri].wss;
    //fit->voi[ri].parNr=par->voi[ri].parNr; // NOT applicable!
    for(pi=0; pi<par->parNr; pi++) if(isnan(fit->voi[ri].p[pi])) break;
    fit->voi[ri].parNr=pi;
  }
  /* Copy parameter values */
  for(ri=0; ri<par->voiNr; ri++)
    for(pi=0; pi<fit->voi[ri].parNr; pi++)
      fit->voi[ri].p[pi]=par->voi[ri].p[pi];

  return(PAR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy contents of PETPAR struct to RES.
\return Returns PETPAR error code, PAR_OK (0) when successful.
 */ 
int parPar2Res(
  /** Pointer to filled PETPAR struct */
  PETPAR *par,
  /** Pointer to initiated RES struct */
  RES *res,
  /** Verbose level; if zero, then only warnings are printed into stderr */
  int verbose
) {
  int pi, ri;
  
  if(verbose>0) printf("parPar2Res(par, res, %d)\n", verbose);
  if(par==NULL || res==NULL) return(PAR_ERROR);
  if(par->voiNr<1 || par->parNr<1) return(PAR_NOTABLE);
  if(par->parNr>MAX_RESPARAMS) return(PAR_TOOBIG);

  /* Allocate memory for fit */
  resEmpty(res);
  if(resSetmem(res, par->voiNr)!=0) return(PAR_OUTOFMEMORY);
  res->voiNr=par->voiNr;
  res->parNr=par->parNr; // This nr may be increased later

  /* Copy header */
  if(strlen(par->program)>0) sprintf(res->program, "%s (c) 2013", par->program);
  else strcpy(res->program, "parToResult (c) 2013");
  res->time=par->time;
  strcpy(res->studynr, par->studynr);
  strcpy(res->datafile, par->datafile);
  strcpy(res->reffile, par->reffile);
  strcpy(res->plasmafile, par->plasmafile);
  strcpy(res->plasmafile2, par->plasmafile2);
  strcpy(res->bloodfile, par->bloodfile);
  strcpy(res->refroi, par->refname);
  if(par->weighting==WEIGHTING_ON) res->isweight=1;
  else if(par->weighting==WEIGHTING_OFF) res->isweight=0;
  else res->isweight=-1;
  if(!isnan(par->density)) res->density=par->density;
  if(!isnan(par->lc)) res->lc=par->lc;
  if(!isnan(par->beta)) res->beta=par->beta;
  if(!isnan(par->concentration)) res->concentration=par->concentration;
  if(!isnan(par->Vb)) res->Vb=par->Vb;
  if(!isnan(par->fA)) res->fA=par->fA;
  if(!isnan(par->E)) res->E=par->E;
  /* Copy region names, etc */
  for(ri=0; ri<res->voiNr; ri++) {
    strcpy(res->voi[ri].name, par->voi[ri].id.name);
    rnameSplit(res->voi[ri].name, res->voi[ri].voiname, res->voi[ri].hemisphere,
               res->voi[ri].place, 6);
  }
  /* Copy parameter names and units */
  for(pi=0; pi<par->parNr; pi++) {
    strcpy(res->parname[pi], par->parname[pi]);
    strcpy(res->parunit[pi], petCunit(par->parunit[pi]));
  }
  /* Copy parameter values */
  for(pi=0; pi<par->parNr; pi++) {
    for(ri=0; ri<par->voiNr; ri++)
      res->voi[ri].parameter[pi]=par->voi[ri].p[pi];
    if(parSDWithPar(par, pi))
      for(ri=0; ri<par->voiNr; ri++)
        res->voi[ri].sd[pi]=par->voi[ri].sd[pi];
    if(parCLWithPar(par, pi))
      for(ri=0; ri<par->voiNr; ri++) {
        res->voi[ri].cl1[pi]=par->voi[ri].cl1[pi];
        res->voi[ri].cl2[pi]=par->voi[ri].cl2[pi];
      }
  }
  /* Add extra parameters as parameters */
  if(parIsWSS(par)) {
    pi=res->parNr; res->parNr++;
    strcpy(res->parname[pi], "WSS");
    strcpy(res->parunit[pi], "");
    for(ri=0; ri<par->voiNr; ri++)
      res->voi[ri].parameter[pi]=par->voi[ri].wss;
  }
  if(parIsFitrange(par)) {
    pi=res->parNr; res->parNr+=2;
    strcpy(res->parname[pi], "Start"); strcpy(res->parname[pi+1], "End");
    strcpy(res->parunit[pi], ""); strcpy(res->parunit[pi+1], "");
    for(ri=0; ri<par->voiNr; ri++) {
      res->voi[ri].parameter[pi]=par->voi[ri].start;
      res->voi[ri].parameter[pi+1]=par->voi[ri].end;
    }
  }
  if(parHighestParnr(par)>0) {
    pi=res->parNr; res->parNr++;
    strcpy(res->parname[pi], "parNr");
    strcpy(res->parunit[pi], "");
    for(ri=0; ri<par->voiNr; ri++)
      res->voi[ri].parameter[pi]=par->voi[ri].parNr;
  }
  if(parHighestDatanr(par)>0) {
    pi=res->parNr; res->parNr++;
    strcpy(res->parname[pi], "dataNr");
    strcpy(res->parunit[pi], "");
    for(ri=0; ri<par->voiNr; ri++)
      res->voi[ri].parameter[pi]=par->voi[ri].dataNr;
  }
  return(PAR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy contents of RES struct to PETPAR.
\return Returns PETPAR error code, PAR_OK (0) when successful.
 */ 
int parRes2Par(
  /** Pointer to filled RES struct */
  RES *res,
  /** Pointer to initiated PETPAR struct */
  PETPAR *par,
  /** Verbose level; if zero, then only warnings are printed into stderr */
  int verbose
) {
  int pi, ri, n;
  char *cptr;
  
  if(verbose>0) printf("parRes2Par(fit, par, %d)\n", verbose);
  if(par==NULL || res==NULL) return(PAR_ERROR);
  if(res->voiNr<1 || res->parNr<1) return(PAR_NOTABLE);

  /* Allocate memory for par */
  parEmpty(par);
  if(parAllocate(par, res->voiNr)!=0) return(PAR_OUTOFMEMORY);
  par->parNr=res->parNr; // this may be decreased later
  par->voiNr=res->voiNr;

  /* Copy header */
  par->fileformat=PETPAR_RES;
  strcpy(par->program, res->program);
  cptr=strchr(par->program, ' '); if(cptr!=NULL) *cptr=(char)0;
  par->time=res->time;
  strcpy(par->studynr, res->studynr);
  strcpy(par->datafile, res->datafile);
  strcpy(par->reffile, res->reffile);
  strcpy(par->plasmafile, res->plasmafile);
  strcpy(par->plasmafile2, res->plasmafile2);
  strcpy(par->bloodfile, res->bloodfile);
  strcpy(par->refname, res->refroi);
  if(res->isweight==1) par->weighting=WEIGHTING_ON;
  else if(res->isweight==0) par->weighting=WEIGHTING_OFF;
  else par->weighting=WEIGHTING_UNKNOWN;
  if(res->density>0) par->density=res->density;
  if(res->lc>0) par->lc=res->lc;
  if(res->beta>0) par->beta=res->beta;
  if(res->concentration>0) par->concentration=res->concentration;
  if(res->Vb>=0) par->Vb=res->Vb;
  if(res->fA>=0) par->fA=res->fA;
  if(res->E>=0) par->E=res->E;
  /* Copy region names, etc */
  for(ri=0; ri<res->voiNr; ri++) {
    rnameRmDots(res->voi[ri].name, par->voi[ri].id.name);
  }
  /* Copy parameter names, units, and values */
  /* Take extra parameters into consideration */
  for(pi=n=0; pi<res->parNr; pi++) {
    if(strcasecmp(res->parname[pi], "WSS")==0) {
      for(ri=0; ri<par->voiNr; ri++)
        par->voi[ri].wss=res->voi[ri].parameter[pi];
      continue;
    }
    if(strcasecmp(res->parname[pi], "Start")==0) {
      for(ri=0; ri<par->voiNr; ri++)
        par->voi[ri].start=res->voi[ri].parameter[pi];
      continue;
    }
    if(strcasecmp(res->parname[pi], "End")==0) {
      for(ri=0; ri<par->voiNr; ri++)
        par->voi[ri].end=res->voi[ri].parameter[pi];
      continue;
    }
    if(strcasecmp(res->parname[pi], "parNr")==0) {
      for(ri=0; ri<par->voiNr; ri++)
        par->voi[ri].parNr=roundf(res->voi[ri].parameter[pi]);
      continue;
    }
    if(strcasecmp(res->parname[pi], "dataNr")==0) {
      for(ri=0; ri<par->voiNr; ri++)
        par->voi[ri].dataNr=roundf(res->voi[ri].parameter[pi]);
      continue;
    }
    strcpy(par->parname[pi], res->parname[pi]);
    par->parunit[pi]=petCunitId(res->parunit[pi]);
    for(ri=0; ri<par->voiNr; ri++) {
      par->voi[ri].p[n]=res->voi[ri].parameter[pi];
      par->voi[ri].sd[n]=res->voi[ri].sd[pi];
      par->voi[ri].cl1[n]=res->voi[ri].cl1[pi];
      par->voi[ri].cl2[n]=res->voi[ri].cl2[pi];
    }
    n++;
  }
  par->parNr=n;

  return(PAR_OK);
}
#endif
/*****************************************************************************/

/*****************************************************************************/
