/******************************************************************************
  Copyright (c) 2013 by Turku PET Centre

  File:        parcsv.c
  Description: I/O functions for PET model parameter files in CSV format.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Version history:
  2013-09-13 Vesa Oikonen
       First created.
     


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/tacio.h"
/*****************************************************************************/

/*****************************************************************************/
/** Save parameter data in PETPAR struct into specified CSV.
\return Returns 0 when successful, otherwise TACIO error code.
 */
int parWriteCSV(
  /** Pointer to PETPAR data */
  PETPAR *par,
  /** File pointer, opened for write */
  FILE *fp,
  /** Verbose level; if zero, then only warnings are printed into stderr */
  int verbose
) {
  int ri, pi;
  int tointl=0;
  char ss[2], tmp[FILENAME_MAX];
  struct tm *st;

  if(verbose>0) printf("parWriteCSV()\n");
  /* Check input */
  if(par==NULL || fp==NULL) return(TACIO_FAULT);
  if(par->voiNr<1) return(TACIO_NOTABLE);
  
  /* Set decimal and item separators */
  if(par->format==PAR_FORMAT_CSV_INT) {
    tointl=1; strcpy(ss, ";");
  } else {
    tointl=0; strcpy(ss, ",");
  }

  /* Write the title lines, if there are titles */
  if(verbose>1) printf("  writing header\n");
  if(csvWriteText(fp, "Parameters\n", 0)) return(TACIO_CANNOTWRITE);
  if(strlen(par->studynr)>0) {
    sprintf(tmp, "studynr%s%s\n", ss, par->studynr);
    if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
  }
  st=localtime(&par->time); if(st!=NULL) {
    if(csvWriteText(fp, "date,", tointl)) return(TACIO_CANNOTWRITE);
    //strftime(tmp, 256, "%Y-%m-%d %H:%M:%S\n", st);
    strftime(tmp, 256, "%Y-%m-%d\n", st);
    if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
  }
  if(strlen(par->program)>0) {
    sprintf(tmp, "program%s%s\n", ss, par->program);
    if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
  }
  if(strlen(par->datafile)>0) {
    sprintf(tmp, "datafile%s%s\n", ss, par->datafile);
    if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
  }
  if(strlen(par->reffile)>0) {
    sprintf(tmp, "reffile%s%s\n", ss, par->reffile);
    if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
  }
  if(strlen(par->plasmafile)>0) {
    sprintf(tmp, "plasmafile%s%s\n", ss, par->plasmafile);
    if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
  }
  if(strlen(par->plasmafile2)>0) {
    sprintf(tmp, "plasmafile2%s%s\n", ss, par->plasmafile2);
    if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
  }
  if(strlen(par->bloodfile)>0) {
    sprintf(tmp, "bloodfile%s%s\n", ss, par->bloodfile);
    if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
  }
  if(strlen(par->refname)>0) {
    sprintf(tmp, "refname%s%s\n", ss, par->refname);
    if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
  }
  if(strlen(par->fitmethod)>0) {
    sprintf(tmp, "fitmethod%s%s\n", ss, par->fitmethod);
    if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
  }
  if(par->cunit!=CUNIT_UNKNOWN) {
    sprintf(tmp, "dataunit%s%s\n", ss, petCunit(par->cunit));
    if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
  }
  if(par->tunit!=TUNIT_UNKNOWN) {
    sprintf(tmp, "timeunit%s%s\n", ss, petTunit(par->tunit));
    if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
  }
  if(par->weighting==WEIGHTING_ON) {
    sprintf(tmp, "weighting%syes\n", ss);
    if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
  } else if(par->weighting==WEIGHTING_OFF) {
    sprintf(tmp, "weighting%sno\n", ss);
    if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
  }
  if(!isnan(par->density)) {
    sprintf(tmp, "density,%g\n", par->density);
    if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
  }
  if(!isnan(par->lc)) {
    sprintf(tmp, "lc,%g\n", par->lc);
    if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
  }
  if(!isnan(par->beta)) {
    sprintf(tmp, "beta,%g\n", par->beta);
    if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
  }
  if(!isnan(par->concentration)){
    sprintf(tmp, "concentration,%g\n", par->concentration);
    if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
  }
  if(!isnan(par->Vb)) {
    sprintf(tmp, "Vb,%g\n", par->Vb);
    if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
  }
  if(!isnan(par->fA)) {
    sprintf(tmp, "fA,%g\n", par->fA);
    if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
  }
  if(!isnan(par->E)) {
    sprintf(tmp, "E,%g\n", par->E);
    if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
  }
  /* Write parameter table */
  if(verbose>1) printf("  writing parameter table\n");
  /* Parameter names */
  if(verbose>2) printf("  writing parameter names\n");
  if(csvWriteText(fp, "region", 0)) return(TACIO_CANNOTWRITE);
  for(pi=0; pi<par->parNr; pi++) {
    sprintf(tmp, "%s%s", ss, par->parname[pi]);
    if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
    if(parSDWithPar(par, pi))
      if(csvWriteText(fp, ",SD", tointl)) return(TACIO_CANNOTWRITE);
    if(parCLWithPar(par, pi))
      if(csvWriteText(fp, ",95%CLl,95%CLu", tointl)) return(TACIO_CANNOTWRITE);
  }
  /* Extra parameters, if available */
  if(parIsWSS(par))
    if(csvWriteText(fp, ",WSS", tointl)) return(TACIO_CANNOTWRITE);
  if(parIsFunction(par))
    if(csvWriteText(fp, ",Function", tointl)) return(TACIO_CANNOTWRITE);
  if(parIsModel(par))
    if(csvWriteText(fp, ",Model", tointl)) return(TACIO_CANNOTWRITE);
  if(parIsFitrange(par))
     if(csvWriteText(fp, ",Start,End", tointl)) return(TACIO_CANNOTWRITE);
  if(parHighestParnr(par)>0)
    if(csvWriteText(fp, ",parNr", tointl)) return(TACIO_CANNOTWRITE);
  if(parHighestDatanr(par)>0)
    if(csvWriteText(fp, ",dataNr", tointl)) return(TACIO_CANNOTWRITE);
  if(csvWriteText(fp, "\n", 0)) return(TACIO_CANNOTWRITE);
  /* Parameter units */
  if(parNrofUnits(par)>0) {
    if(verbose>2) printf("  writing parameter units\n");
    if(csvWriteText(fp, "units", 0)) return(TACIO_CANNOTWRITE);
    for(pi=0; pi<par->parNr; pi++) {
      sprintf(tmp, "%s%s", ss, petCunit(par->parunit[pi]));
      if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
      /* Units for SD and CLs, too, if needed */
      if(parSDWithPar(par, pi))
        if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
      if(parCLWithPar(par, pi)) {
        if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
        if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
      }
    }
    if(parIsWSS(par)) if(csvWriteText(fp, ",", tointl)) return(TACIO_CANNOTWRITE);
    if(parIsFunction(par))
      if(csvWriteText(fp, ",", tointl)) return(TACIO_CANNOTWRITE);
    if(parIsModel(par))
      if(csvWriteText(fp, ",", tointl)) return(TACIO_CANNOTWRITE);
    if(parIsFitrange(par))
      if(csvWriteText(fp, ",,", tointl)) return(TACIO_CANNOTWRITE);
    if(parHighestParnr(par)>0)
      if(csvWriteText(fp, ",", tointl)) return(TACIO_CANNOTWRITE);
    if(parHighestDatanr(par)>0)
      if(csvWriteText(fp, ",", tointl)) return(TACIO_CANNOTWRITE);
    if(csvWriteText(fp, "\n", 0)) return(TACIO_CANNOTWRITE);
  }
  /* Data table */
  if(verbose>2) printf("  writing parameter values\n");
  for(ri=0; ri<par->voiNr; ri++) {
    if(verbose>2) printf("    for region %d\n", 1+ri);
    /* Region names and parameter values */
    if(csvWriteText(fp, par->voi[ri].id.name, 0)) return(TACIO_CANNOTWRITE);
    for(pi=0; pi<par->parNr; pi++) {
      sprintf(tmp, ",%g", par->voi[ri].p[pi]);
      if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
      /* SD and CLs, too, if needed */
      if(parSDWithPar(par, pi)) {
        if(isnan(par->voi[ri].sd[pi])) strcpy(tmp, ",");
        else sprintf(tmp, ",%g", par->voi[ri].sd[pi]);
        if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
      }
      if(parCLWithPar(par, pi)) {
        if(isnan(par->voi[ri].cl1[pi]) || isnan(par->voi[ri].cl2[pi]))
          strcpy(tmp, ",,");
        else
          sprintf(tmp, ",%g,%g", par->voi[ri].cl1[pi], par->voi[ri].cl2[pi]);
        if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
      }
    }
    if(parIsWSS(par)) {
      if(isnan(par->voi[ri].wss)) strcpy(tmp, ",");
      else sprintf(tmp, ",%g", par->voi[ri].wss);
      if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
    }
    if(parIsFunction(par)) {
      sprintf(tmp, ",%d", par->voi[ri].function);
      if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
    }
    if(parIsModel(par)) {
      sprintf(tmp, ",%d", par->voi[ri].model);
      if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
    }
    if(parIsFitrange(par)) {
      if(isnan(par->voi[ri].start) || isnan(par->voi[ri].end)) strcpy(tmp, ",,");
      else sprintf(tmp, ",%g,%g", par->voi[ri].start, par->voi[ri].end);
      if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
    }
    if(parHighestParnr(par)>0) {
      sprintf(tmp, ",%d", par->voi[ri].parNr);
      if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
    }
    if(parHighestDatanr(par)>0) {
      sprintf(tmp, ",%d", par->voi[ri].dataNr);
      if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
    }

    if(csvWriteText(fp, "\n", 0)) return(TACIO_CANNOTWRITE);
  } // next parameter set (region)

  return(TACIO_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read parameter data into PETPAR struct from specified CSV file.
\return Returns 0 when successful, otherwise TACIO error code.
 */
int parReadCSV(
  /** Pointer to PETPAR data */
  PETPAR *par,
  /** Output filename */
  char *filename,
  /** Verbose level; if zero, then only warnings are printed into stderr */
  int verbose
) {
  int i, j, k, n, ret;
  CSV csv;
  int title_index, units_index, data_index, colNr, rowNr, voiNr, parNr;
  double f;
  char *cptr1, *cptr2;
  int col_type=0; // (0) SD (1), CL1 (2), or CL2 (3)
  //char tmp[FILENAME_MAX];


  if(verbose>0) printf("parReadCSV(par, %s, ...)\n", filename);
  /* Check input */
  if(par==NULL || filename==NULL || strlen(filename)<1) return(TACIO_FAULT);

  /* Try to read file as CSV */
  csvInit(&csv);
  ret=csvRead(&csv, filename, verbose-1);
  if(ret!=TACIO_OK) {
    if(verbose>0) printf("csvRead(%s) := %d\n", filename, ret);
    csvEmpty(&csv);
    if(ret==TACIO_CANNOTOPEN) return(ret);
    else return(TACIO_INVALIDFORMAT);
  }
  if(verbose>10) csvPrint(&csv);

  /* Find the title line, starting with 'Region' */
  title_index=-1;
  for(i=0; i<csv.nr; i++) {
    if(verbose>5)
      printf("col=%d row=%d content='%s'\n",
             csv.c[i].col, csv.c[i].row, csv.c[i].content);
    if(csv.c[i].col!=1) continue;
    if(strcasecmp(csv.c[i].content, "REGION")==0) {title_index=i;}
  }
  if(title_index<0) {
    if(verbose>0) printf("title line not found in %s\n", filename);
    csvEmpty(&csv); return(TACIO_INVALIDFORMAT);
  }
  /* Determine the nr of columns from title line */
  for(i=title_index+1, colNr=1; i<csv.nr; i++)
    if(csv.c[i].row==csv.c[i-1].row) colNr++; else break;
  if(colNr<2) {
    if(verbose>0) printf("invalid title line in %s\n", filename);
    csvEmpty(&csv); return(TACIO_INVALIDFORMAT);
  }
  if(verbose>1) printf("colNr := %d\n", colNr);
  /* Preliminary index for data start is right after the title line */
  data_index=title_index+colNr;
  /* Check if the line below titles contains the units */
  units_index=-1;
  if(strncasecmp(csv.c[i].content, "UNITS", 4)==0) {
    units_index=i;
    /* then also data is starting one row later */
    data_index+=colNr;
  }
  if(verbose>1) {
    if(units_index<0) printf("units not specified\n");
    else printf("units specified on line %d\n", csv.c[units_index].row);
  }
  /* Determine the nr of rows containing equal nr of columns as title line */
  for(i=title_index+colNr+1, rowNr=0, j=1; i<csv.nr; i++) {
    if(csv.c[i].row==csv.c[i-1].row) {j++; continue;}
    if(j!=colNr) break;
    j=1; rowNr++;
  }
  if(i==csv.nr && j==colNr) rowNr++;
  if(units_index>=0) rowNr--; // exclude units line from rowNr
  if(verbose>1) printf("rowNr := %d\n", rowNr);
  voiNr=rowNr;
  if(verbose>1) printf("voiNr := %d\n", voiNr);
  if(voiNr<1) {csvEmpty(&csv); return(TACIO_NOTABLE);}
  /* Check if table contains SD or CL columns */
  for(i=title_index+1, j=0; i<title_index+1+colNr; i++) {
    cptr1=csv.c[i].content;
    if(strcasecmp(cptr1, "SD")==0) {j++; continue;}
    if(strcasecmp(cptr1, "StDev")==0) {j++; continue;}
    if(strncasecmp(cptr1, "95%CL", 5)==0) {j++; continue;}
    if(strncasecmp(cptr1, "CL95", 4)==0) {j++; continue;}
  }
  if(verbose>4) printf("%d columns with SD or CL\n", j);
  parNr=colNr-1-j;
  /* or extra parameter columns */
  for(i=title_index+1, j=0; i<title_index+1+colNr; i++) {
    cptr1=csv.c[i].content;
    if(strcasecmp(cptr1, "WSS")==0) {j++; continue;}
    if(strcasecmp(cptr1, "START")==0) {j++; continue;}
    if(strcasecmp(cptr1, "END")==0) {j++; continue;}
    if(strcasecmp(cptr1, "PARNR")==0) {j++; continue;}
    if(strcasecmp(cptr1, "DATANR")==0) {j++; continue;}
    if(strcasecmp(cptr1, "FUNCTION")==0) {j++; continue;}
    if(strcasecmp(cptr1, "MODEL")==0) {j++; continue;}
  }
  if(verbose>4) printf("%d extra columns\n", j);
  parNr-=j;
  if(verbose>1) printf("parNr := %d\n", parNr);
  if(parNr<1) {csvEmpty(&csv); return(TACIO_NOTABLE);}

  /* Allocate PETPAR, delete any previous contents */
  if(verbose>2) printf("allocating memory for PAR\n");
  if(parNr>MAX_PETPARAMS) {
    if(verbose>0) printf("too many columns in data table in %s\n", filename);
    csvEmpty(&csv); return(TACIO_TOOBIG);
  }
  parEmpty(par);
  ret=parAllocate(par, voiNr);
  if(ret!=0) {
    if(verbose>0) printf("resSetmem(*res, %d) := %d\n", voiNr, ret);
    csvEmpty(&csv); return(TACIO_OUTOFMEMORY);
  }
  /* Fill PETPAR contents from CSV */
  if(verbose>1) printf("filling PAR\n");
  par->voiNr=voiNr;
  par->parNr=parNr;
  //for(i=0; i<voiNr; i++) par->voi[i].parNr=par->parNr;
  /* Region names from column 1 */
  for(j=0; j<voiNr; j++) {
    i=data_index+j*colNr;
    strncpy(par->voi[j].id.name, csv.c[i].content, MAX_TACNAME_LEN);
    rnameRmDots(par->voi[j].id.name, NULL);
  }
  /* Parameter names, units, and values from following columns */
  for(k=1, j=0; k<colNr; k++) {
    i=title_index+k; cptr1=csv.c[i].content;
    col_type=0; /* check if this is parameter (0) SD (1), CL1 (2), or CL2 (3),
    or WSS (4), start (5), end (6), parNr (7), dataNr (8), function id (9),
    or model id (10) */ 
    if(strcasecmp(cptr1, "SD")==0) col_type=1;
    else if(strcasecmp(cptr1, "StDev")==0) col_type=1;
    else if(strcasecmp(cptr1, "95%CLl")==0) col_type=2;
    else if(strcasecmp(cptr1, "CL95%l")==0) col_type=2;
    else if(strcasecmp(cptr1, "95%CLu")==0) col_type=3;
    else if(strcasecmp(cptr1, "CL95%u")==0) col_type=3;
    else if(strcasecmp(cptr1, "WSS")==0) col_type=4;
    else if(strcasecmp(cptr1, "START")==0) col_type=5;
    else if(strcasecmp(cptr1, "END")==0) col_type=6;
    else if(strcasecmp(cptr1, "PARNR")==0) col_type=7;
    else if(strcasecmp(cptr1, "DATANR")==0) col_type=8;
    else if(strcasecmp(cptr1, "FUNCTION")==0) col_type=9;
    else if(strcasecmp(cptr1, "MODEL")==0) col_type=10;
    if(verbose>6) printf("  col_type[%d] := %d\n", k, col_type);
    /* parameter name from first table row */
    if(col_type==0)
      strncpy(par->parname[j], cptr1, MAX_PETPARNAME_LEN);
    /* parameter unit from 2nd table row */
    if(units_index>=0 && col_type==0) {
      i=units_index+k; cptr1=csv.c[i].content;
      par->parunit[j]=petCunitId(cptr1);
    }
    /* parameter values or SD or CLs from following rows */
    for(n=0; n<voiNr; n++) {
      i=data_index+n*colNr+k; cptr1=csv.c[i].content;
      ret=atof_with_check(cptr1, &f); if(ret!=0) f=nan("");
      if(verbose>7) printf("    f[%d][%d] := %g\n", n, j, f);
      if(col_type==0) {par->voi[n].p[j]=f; continue;}
      /* Possible extra parameters */
      else if(col_type==4) par->voi[n].wss=f;
      else if(col_type==5) par->voi[n].start=f;
      else if(col_type==6) par->voi[n].end=f;
      else if(col_type==7) par->voi[n].parNr=roundf(f);
      else if(col_type==8) par->voi[n].dataNr=roundf(f);
      else if(col_type==9) par->voi[n].function=roundf(f);
      else if(col_type==10) par->voi[n].model=roundf(f);
      /* SD and CLs belong to the previous parameter */
      if(j<1) {
        if(verbose>0) printf("invalid table column order in %s\n", filename);
        csvEmpty(&csv); return(TACIO_INVALIDFORMAT);
      }
      else if(col_type==1) par->voi[n].sd[j-1]=f;
      else if(col_type==2) par->voi[n].cl1[j-1]=f;
      else if(col_type==3) par->voi[n].cl2[j-1]=f;
    } // next region
    if(col_type==0) j++; // next true parameter
  } // next column
  
  /* Try to find any extra information from lines before parameter table */
  for(i=0; i<title_index; i++) {
    /*printf("  content := '%s' col=%d row=%d\n", csv.c[i].content,
            csv.c[i].col, csv.c[i].row);*/
    /* Check that this is a first column */
    if(csv.c[i].col!=1) continue;
    if(csv.c[i].content==NULL) continue;
    /* Check that a second column with value exists */
    if(csv.c[i+1].row!=csv.c[i].row) continue;
    if(csv.c[i+1].content==NULL) {i++; continue;}
    if(verbose>5) printf("  info name := '%s' value := '%s'\n",
                         csv.c[i].content, csv.c[i+1].content);
    cptr1=csv.c[i].content; cptr2=csv.c[i+1].content;
    if(strcasecmp(cptr1, "STUDYNR")==0 || strcasecmp(cptr1, "STUDY")==0) {
      strncpy(par->studynr, cptr2, MAX_STUDYNR_LEN);
      i++; continue;
    }
    if(strcasecmp(cptr1, "DATE")==0 || strcasecmp(cptr1, "STUDY_DATE")==0) {
      struct tm st;
      if(get_datetime(cptr2, &st)==0) par->time=mktime(&st);
      else if(get_date(cptr2, &st)==0) par->time=mktime(&st);
      else par->time=(time_t)0;
      i++; continue;
    }
    if(strcasecmp(cptr1, "PROGRAM")==0) {
      strncpy(par->program, cptr2, MAX_STUDYNR_LEN);
      i++; continue;
    }
    if(strcasecmp(cptr1, "DATAFILE")==0) {
      strncpy(par->datafile, cptr2, MAX_STUDYNR_LEN);
      i++; continue;
    }
    if(strcasecmp(cptr1, "REFFILE")==0) {
      strncpy(par->reffile, cptr2, MAX_STUDYNR_LEN);
      i++; continue;
    }
    if(strcasecmp(cptr1, "PLASMAFILE")==0) {
      strncpy(par->plasmafile, cptr2, MAX_STUDYNR_LEN);
      i++; continue;
    }
    if(strcasecmp(cptr1, "PLASMAFILE2")==0) {
      strncpy(par->plasmafile2, cptr2, MAX_STUDYNR_LEN);
      i++; continue;
    }
    if(strcasecmp(cptr1, "BLOODFILE")==0) {
      strncpy(par->bloodfile, cptr2, MAX_STUDYNR_LEN);
      i++; continue;
    }
    if(strcasecmp(cptr1, "REFNAME")==0) {
      strncpy(par->refname, cptr2, MAX_STUDYNR_LEN);
      i++; continue;
    }
    if(strcasecmp(cptr1, "FITMETHOD")==0 || strcasecmp(cptr1, "METHOD")==0) {
      strncpy(par->fitmethod, cptr2, MAX_STUDYNR_LEN);
      i++; continue;
    }
    if(strcasecmp(cptr1, "DATAUNIT")==0) {
      par->cunit=petCunitId(cptr2);
      i++; continue;
    }
    if(strcasecmp(cptr1, "TIMEUNIT")==0) {
      par->tunit=petTunitId(cptr2);
      i++; continue;
    }
    if(strcasecmp(cptr1, "WEIGHTING")==0) {
      if(strncasecmp(cptr2, "YES", 1)==0 || strcasecmp(cptr2, "ON")==0)
        par->weighting=WEIGHTING_ON;
      else if(strncasecmp(cptr2, "NO", 1)==0 || strcasecmp(cptr2, "OFF")==0)
        par->weighting=WEIGHTING_OFF;
      else
        par->weighting=WEIGHTING_UNKNOWN;
      i++; continue;
    }
    if(strcasecmp(cptr1, "DENSITY")==0) {
      par->density=atof_dpi(cptr2);
      i++; continue;
    }
    if(strcasecmp(cptr1, "LC")==0) {
      par->lc=atof_dpi(cptr2);
      i++; continue;
    }
    if(strcasecmp(cptr1, "BETA")==0) {
      par->beta=atof_dpi(cptr2);
      i++; continue;
    }
    if(strcasecmp(cptr1, "CONCENTRATION")==0) {
      par->concentration=atof_dpi(cptr2);
      i++; continue;
    }
    if(strcasecmp(cptr1, "VB")==0) {
      par->Vb=atof_dpi(cptr2);
      i++; continue;
    }
    if(strcasecmp(cptr1, "FA")==0) {
      par->fA=atof_dpi(cptr2);
      i++; continue;
    }
    if(strcasecmp(cptr1, "E")==0) {
      par->E=atof_dpi(cptr2);
      i++; continue;
    }
    if(verbose>0) printf("  info title '%s' not identified\n", cptr1);
  } // next info line

  /* Set fileformat */
  if(csv.separator==';') par->format=PAR_FORMAT_CSV_INT;
  else par->format=PAR_FORMAT_CSV_UK;

  csvEmpty(&csv);
  return(TACIO_OK);
}
/*****************************************************************************/

/*****************************************************************************/
