/******************************************************************************
  Copyright (c) 2013 by Turku PET Centre

  File:        parfit.c
  Description: I/O functions for PET model parameter files in FIT format.
               Note that also older FIT files are read for compatibility.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Version history:
  2013-09-17 Vesa Oikonen
       First created.
     


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/tacio.h"
/*****************************************************************************/

/*****************************************************************************/
/** Save parameter data in PETPAR struct into specified FIT file.
\return Returns 0 when successful, otherwise TACIO error code.
 */
int parWriteFIT(
  /** Pointer to PETPAR data */
  PETPAR *par,
  /** File pointer, opened for write */
  FILE *fp,
  /** Verbose level; if zero, then only warnings are printed into stderr */
  int verbose
) {
  int ri, pi, n, savedNr;
  char tmp[256];
  struct tm *st;

  if(verbose>0) printf("parWriteFIT()\n");
  /* Check input */
  if(par==NULL || fp==NULL) return(TACIO_FAULT);
  if(par->voiNr<1) return(TACIO_NOTABLE);
  /* Check that there is some data to write */
  for(ri=0, savedNr=0; ri<par->voiNr; ri++)
    if(!isnan(par->voi[ri].wss) && par->voi[ri].function>0) savedNr++;
  if(verbose>1) printf("savedNr := %d\n", savedNr);
  if(savedNr<1) return(TACIO_NOTABLE);

  /* Make sure that region names are available */
  if(verbose>1) printf("constructing region names\n");
  parEnsureNames(par);
  /* Construct region subnames */
  for(ri=0; ri<par->voiNr; ri++)
    if(tacNameSplit(&par->voi[ri].id)<1) return TACIO_FAULT;

  /* Write header lines */
  if(verbose>1) printf("writing headers\n");

  /* Fit file format */
  if(strlen(par->program)>2) strcpy(tmp, par->program); else strcpy(tmp, "fit");
  n=fprintf(fp, "%-11.11s %s\n", "FIT1", tmp);
  if(n==0) return TACIO_CANNOTWRITE;

  /* Write calculation date and time */
  st=localtime(&par->time); 
  if(st!=NULL) strftime(tmp, 256, "%Y-%m-%d %H:%M:%S", st);
  else strcpy(tmp, "");
  n=fprintf(fp, "Date:\t%s\n", tmp); if(n==0) return TACIO_CANNOTWRITE;

  /* Write the names of the original datafile */
  n=fprintf(fp, "Data file:\t%s\n", par->datafile);
  if(n==0) return TACIO_CANNOTWRITE;
  
  /* Write the units */
  n=fprintf(fp, "Data unit:\t%s\n", petCunit(par->cunit));
  if(n==0) return TACIO_CANNOTWRITE;
  n=fprintf(fp, "Time unit:\t%s\n", petTunit(par->tunit));
  if(n==0) return TACIO_CANNOTWRITE;

  /* Write the voiNr to be saved */
  n=fprintf(fp, "Nr of VOIs:\t%d\n", savedNr);
  if(n==0) return TACIO_CANNOTWRITE;

  /* Write the table title */
  if(verbose>1) printf("writing table\n");
  n=fprintf(fp, "%s %s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\n",
     "Region", "Plane", "Start", "End", "dataNr", "WSS", "parNr", "Type",
     "Parameters");
  if(n==0) return TACIO_CANNOTWRITE;

  /* Write regional fits */
  for(ri=0; ri<par->voiNr; ri++) {
    if(isnan(par->voi[ri].wss) || par->voi[ri].function<=0) continue;
    /* VOI name */
    strcpy(tmp, par->voi[ri].id.sub[0]);
    strcat(tmp, " ");
    if(MAX_SUBTACID_NR<2 || strlen(par->voi[ri].id.sub[1])==0) strcat(tmp, ".");
    else strcat(tmp, par->voi[ri].id.sub[1]);
    strcat(tmp, " ");
    if(MAX_SUBTACID_NR<3 || strlen(par->voi[ri].id.sub[2])==0) strcat(tmp, ".");
    else strcat(tmp, par->voi[ri].id.sub[2]);
    n=fprintf(fp, "%s\t", tmp); if(n<1) return TACIO_CANNOTWRITE;
    /* parameter values */
    n=fprintf(fp, "%.3f\t%.3f\t%d\t%.2E\t%d\t%04d",
       par->voi[ri].start, par->voi[ri].end, par->voi[ri].dataNr,
       par->voi[ri].wss, par->voi[ri].parNr, par->voi[ri].function );
    if(n==0) return TACIO_CANNOTWRITE;
    for(pi=0; pi<par->voi[ri].parNr; pi++)
      fprintf(fp, "\t%.6E", par->voi[ri].p[pi]);
    n=fprintf(fp, "\n");
    if(n==0) return TACIO_CANNOTWRITE;
  } // next region

  return(TACIO_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read parameter data into PETPAR struct from specified FIT file.
 *  Reading older RES files are supported.
\return Returns 0 when successful, otherwise TACIO error code.
 */
int parReadFIT(
  /** Pointer to PETPAR data */
  PETPAR *par,
  /** Output filename */
  char *filename,
  /** Verbose level; if zero, then only warnings are printed into stderr */
  int verbose
) {
  int ret, lineNr, colNr;
  IFT ift;


  if(verbose>0) printf("parReadFIT(par, %s, ...)\n", filename);
  /* Check input */
  if(par==NULL || filename==NULL || strlen(filename)<1) return(TACIO_FAULT);

  /* Delete any previous data */
  parEmpty(par);

  /* Read file into IFT struct */
  if(verbose>1) printf("reading file\n");
  iftInit(&ift);
  ret=iftRead(&ift, filename, 0);
  if(ret!=IFT_OK) {
    if(verbose>0) printf("Error in reading: %s\n", ift.status);
    iftEmpty(&ift); if(ret==IFT_CANNOTREAD) return TACIO_CANNOTOPEN;
    return TACIO_INVALIDFORMAT;
  }
  if(verbose>1) printf("%d line(s).\n", ift.keyNr);
  
  /* Get data dimensions */
  if(verbose>1) printf("reading dimensions of the data\n");
  ret=iftGetDataDimensions(&ift, &lineNr, &colNr);
  if(ret!=IFT_OK) {
    if(verbose>0) printf("error %d in reading dimensions of the data\n", ret);
    iftEmpty(&ift); return TACIO_INVALIDFORMAT;
  }
  if(verbose>2) {
    printf("data_lines := %d\n", lineNr);
    printf("max_column_nr := %d\n", colNr);
  }

  /* Allocate memory for parameters */
  if(verbose>1) printf("allocating memory for parameters\n");
  ret=parAllocate(par, lineNr);
  if(ret!=TACIO_OK) {iftEmpty(&ift); parEmpty(par); return ret;}
  
  /* Move data from IFT */
  if(verbose>1) printf("moving data table from IFT\n");
  ret=parProcessFITData(par, &ift, verbose-10);

  if(verbose>4) parPrint(par);

  if(ret!=TACIO_OK) {iftEmpty(&ift); parEmpty(par); return ret;}
  iftEmpty(&ift);
  
  /* Set study number from filename, if necessary */
  if(strlen(par->studynr)==0) studynr_from_fname(filename, par->studynr);

  return(TACIO_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy FIT data from IFT to PETPAR struct.
\return Returns TACIO status.
 */
int parProcessFITData(
  /** Pointer to allocated PETPAR struct into where data is copied */
  PETPAR *par,
  /** Pointer to IFT struct from where the data is read */
  IFT *ift,
  /** Verbose level; if zero, nothing is printed into stdout */
  int verbose
) {
  int i, j, ri, pi, n, title_index, colNr;
  char tmp[256];
  double v;
  
  if(verbose>0) printf("parProcessFITData()\n");
  if(par==NULL || ift==NULL) return TACIO_FAULT;
  if(verbose>3) {
    printf("keyNr := %d\n", ift->keyNr);
  }

  /* Remove comment lines from IFT struct */
  i=0; while(i<ift->keyNr) {
    if(ift->item[i].type!='#' && ift->item[i].type!=';') {i++; continue;}
    iftDeleteItem(ift, i);
  }
  if(verbose>10) iftWrite(ift, "stdout");

  /* Check id, and read program name; must be the first line */
  if(verbose>2) printf("reading program name\n");
  i=0;
  if(strlen(ift->item[i].value)<1) return(TACIO_INVALIDFORMAT);
  if(strlen(ift->item[i].key)>0) return(TACIO_INVALIDFORMAT);
  if(strncasecmp(ift->item[i].value, "FIT1", 3)!=0)
    return(TACIO_INVALIDFORMAT);
  n=strTokenNCpy(ift->item[i].value, " \t", 2, par->program, 1024);
  if(n<1) return(TACIO_INVALIDFORMAT);

  /* Read header lines from IFT */
  if(verbose>2) printf("reading header lines\n");
  
  i=iftGetFrom(ift, 0, "Date");
  if(i>=0) {
    if(verbose>3) printf("date_str := '%s'\n", ift->item[i].value);
    struct tm st;
    if(get_datetime(ift->item[i].value, &st)==0) par->time=mktime(&st);
    else if(get_date(ift->item[i].value, &st)==0) par->time=mktime(&st);
    else par->time=(time_t)0;
  }
  
  i=iftGetFrom(ift, 0, "Study");
  if(i>=0) strncpy(par->studynr, ift->item[i].value, MAX_STUDYNR_LEN);
  
  i=iftGetFrom(ift, 0, "Data file");
  if(i>=0) strncpy(par->datafile, ift->item[i].value, FILENAME_MAX);
  
  i=iftGetFrom(ift, 0, "Data unit");
  if(i>=0) par->cunit=petCunitId(ift->item[i].value);
  
  i=iftGetFrom(ift, 0, "Time unit");
  if(i<0) i=iftGetFrom(ift, 0, "Distance unit");
  if(i>=0) par->tunit=petTunitId(ift->item[i].value);
  
  i=iftGetFrom(ift, 0, "Nr of VOIs");
  if(i>=0) n=atoi(ift->item[i].value);
  if(n<1) return(TACIO_NOTABLE);
  if(n>par->_voidataNr) return(TACIO_TOOBIG);
  par->voiNr=n;

  /* Read the result parameter title line */
  if(verbose>2) printf("reading parameter titles\n");
  title_index=-1;
  for(i=0; i<ift->keyNr; i++) {
    if(strlen(ift->item[i].key)>0) continue;
    if(strncasecmp(ift->item[i].value, "Region", 6)==0) {title_index=i; break;}
  }
  if(title_index<0) return(TACIO_NOTABLE);

  /* Read regional data */
  for(i=title_index+1, ri=0; i<ift->keyNr && ri<par->voiNr; i++, ri++) {
    if(verbose>7) printf("'%s'\n", ift->item[i].value);
    if(strlen(ift->item[i].key)>0) break;
    colNr=strTokenNr(ift->item[i].value, " \t");
    if(colNr<4) {if(verbose>7) printf("colNr=%d\n", colNr); break;}
    /* Check that we don't have too many parameters */
    if(colNr>MAX_PETPAR_NR) return(TACIO_TOOBIG);
    /* Read VOI name (the first three columns) */
    for(j=0; j<3 && j<MAX_SUBTACID_NR; j++) {
      n=strTokenNCpy(ift->item[i].value, " \t", j+1,
                       par->voi[ri].id.sub[j], MAX_TACNAME_LEN+1);
      if(n==0) return(TACIO_INVALIDFORMAT);
    }
    /* Construct full TAC name */
    tacNameCatenate(&par->voi[ri].id);
    if(verbose>8) printf("name := '%s'\n", par->voi[ri].id.name);

    /* Read fit information */
    n=strTokenNCpy(ift->item[i].value, " \t", 4, tmp, 256);
    if(n==0 || atof_with_check(tmp, &v)) return(TACIO_INVALIDFORMAT);
    par->voi[ri].start=v;
    n=strTokenNCpy(ift->item[i].value, " \t", 5, tmp, 256);
    if(n==0 || atof_with_check(tmp, &v)) return(TACIO_INVALIDFORMAT);
    par->voi[ri].end=v;
    n=strTokenNCpy(ift->item[i].value, " \t", 6, tmp, 256);
    if(n==0) return(TACIO_INVALIDFORMAT);
    par->voi[ri].dataNr=atoi(tmp);

    /* Fit error, parameter nr and function number (type) */
    n=strTokenNCpy(ift->item[i].value, " \t", 7, tmp, 256);
    if(n==0 || atof_with_check(tmp, &v)) return(TACIO_INVALIDFORMAT);
    par->voi[ri].wss=v;
    n=strTokenNCpy(ift->item[i].value, " \t", 8, tmp, 256);
    if(n==0) return(TACIO_INVALIDFORMAT);
    par->voi[ri].parNr=atoi(tmp);
    if(par->voi[ri].parNr<1) return(TACIO_INVALIDFORMAT);
    n=strTokenNCpy(ift->item[i].value, " \t", 9, tmp, 256);
    if(n==0) return(TACIO_INVALIDFORMAT);
    par->voi[ri].function=atoi(tmp);
    if(par->voi[ri].function<1) return(TACIO_INVALIDFORMAT);
    /* Set main parNr to max parNr */
    if(par->voi[ri].parNr>par->parNr) par->parNr=par->voi[ri].parNr;

    /* Read parameter values */
    for(pi=0; pi<par->voi[ri].parNr; pi++) {
      n=strTokenNCpy(ift->item[i].value, " \t", pi+10, tmp, 256);
      if(n==0 || atof_with_check(tmp, &v)) return(TACIO_INVALIDFORMAT);
      par->voi[ri].p[pi]=v;
    }

  } // next line (region)  

  /* Set simple parameter names */
  for(pi=0; pi<par->parNr; pi++) sprintf(par->parname[pi], "p%d", pi+1);

  return(TACIO_OK);
}
/*****************************************************************************/

/*****************************************************************************/
