/******************************************************************************
  Copyright (c) 2013 by Turku PET Centre

  File:        pario.c
  Description: General I/O functions for PET model parameter files.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Version history:
  2013-09-13 Vesa Oikonen
       First created.
     


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/tacio.h"
/*****************************************************************************/

/*****************************************************************************/
/** Save parameter data in PETPAR struct into specified file.
\return Returns 0 when successful, otherwise TACIO error code.
 */
int parWrite(
  /** Pointer to PETPAR data */
  PETPAR *par,
  /** Output filename */
  char *filename,
  /** File_format code, as specified in tacio.h; enter TAC_FORMAT_UNKNOWN
   *  to write data in the format specified inside PETPAR struct. */  
  int format,
  /** Verbose level; if zero, then only warnings are printed into stderr */
  int verbose
) {
  int ret;
  char is_stdout=0;
  FILE *fp;

  if(verbose>0) printf("parWrite(par, '%s', %s, %d)\n",
                       filename, tacFormattxt(format), verbose);
  /* Check input */
  if(par==NULL || filename==NULL || strlen(filename)<1) return(TACIO_FAULT);
  if(par->voiNr<1) return(TACIO_NOTABLE);

  /* Check if writing to stdout */
  if(!strcasecmp(filename, "stdout")) is_stdout=1;

  /* Check if file exists; backup, if necessary */
  if(!is_stdout) (void)backupExistingFile(filename, NULL, NULL);

  /* Open output file */
  if(is_stdout) fp=(FILE*)stdout;
  else {
    if(verbose>1) printf("opening file for write\n");
    if((fp=fopen(filename, "w")) == NULL) return TACIO_CANNOTWRITE;
  }

  /* If file format was not defined in function call, then get it from data */
  if(format==TAC_FORMAT_UNKNOWN) {
    format=par->format;
    if(verbose>1) printf("file_format := %s\n", tacFormattxt(format));
  }

  /* Write file in specified file format */
  ret=TACIO_INVALIDFORMAT;
  switch(par->format) {
    case PAR_FORMAT_CSV_UK:
    case PAR_FORMAT_CSV_INT:
    case TAC_FORMAT_UNKNOWN:
      ret=parWriteCSV(par, fp, verbose); break;
    case PAR_FORMAT_RES:
      ret=parWriteRES(par, fp, verbose); break;
    case PAR_FORMAT_FIT:
      ret=parWriteFIT(par, fp, verbose); break;
      break;
    default:
      ret=TACIO_INVALIDFORMAT;
      break;
  }

  /* Close file */
  fflush(fp); if(!is_stdout) fclose(fp);

  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read parameter file in PAR struct.
\return Returns 0 when successful, otherwise TACIO error code.
 */
int parRead(
  /** Pointer to PETPAR data */
  PETPAR *par,
  /** Input filename */
  char *filename,
  /** Verbose level; if zero, then only warnings are printed into stderr */
  int verbose
) {
  int ret, format;

  if(verbose>0) printf("parRead(par, %s, ...)\n", filename);
  if(par==NULL || filename==NULL || strlen(filename)<1) return(TACIO_FAULT);

  /* Try to identify the file format */
  format=tacFormatDetermine(filename, verbose);
  if(format==TAC_FORMAT_UNKNOWN) {
    if(verbose>0) printf("unknown file format");
    return TACIO_INVALIDFORMAT;
  }
  
  /* Read the supported formats */
  switch(format) {
    case PAR_FORMAT_CSV_INT:
    case PAR_FORMAT_CSV_UK:
      ret=parReadCSV(par, filename, verbose);
      return(ret);
    case PAR_FORMAT_RES:
      ret=parReadRES(par, filename, verbose);
      return(ret);
    case PAR_FORMAT_FIT:
      ret=parReadFIT(par, filename, verbose);
      return(ret);
      break;
  }

  /* The rest of formats are not supported */
  if(verbose>0)
    fprintf(stderr, "Error: format %s not supported as parameter file.\n",
            tacFormattxt(format));
  return TACIO_INVALIDFORMAT;
}
/*****************************************************************************/

/*****************************************************************************/
/** Determine whether the parameter should be printed as
    integer (0), float (1), or in exp format (2).
\return Returns the code, or <0 in case of an error.
 */
int parPrintType(
  /** Pointer to the result struct */
  PETPAR *par,
  /** Index of the parameter to test */
  int parIndex
) {
  int vi, partype=0;
  double x, m=0.0, pint;

  if(par==NULL || par->voiNr<1 || parIndex<0 || parIndex>=par->parNr)
    return(-1);
  for(vi=0; vi<par->voiNr; vi++) {
    x=par->voi[vi].p[parIndex];
    if(modf(x, &pint)!=0.0) partype=1;
    x=fabs(x); if(x>m) m=x;
  }
  if(partype==1 && (m>=10.0 || m<0.1)) partype=2;
  return(partype);
}
/*****************************************************************************/

/*****************************************************************************/
