/******************************************************************************
  Copyright (c) 2013 by Turku PET Centre

  File:        parres.c
  Description: I/O functions for PET model parameter files in RES format.
               Note that also older RES files are read for compatibility.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Version history:
  2013-09-17 Vesa Oikonen
       First created.
     


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/tacio.h"
/*****************************************************************************/

/*****************************************************************************/
/** Save parameter data in PETPAR struct into specified RES file.
\return Returns 0 when successful, otherwise TACIO error code.
 */
int parWriteRES(
  /** Pointer to PETPAR data */
  PETPAR *par,
  /** File pointer, opened for write */
  FILE *fp,
  /** Verbose level; if zero, then only warnings are printed into stderr */
  int verbose
) {
  int i, j, n;
  char tmp[1024];
  struct tm *st;
  int partype[MAX_PETPAR_NR]; /* 0=int, 1=double, 2=exp */
  double *p;


  if(verbose>0) printf("parWriteRES()\n");
  /* Check input */
  if(par==NULL || fp==NULL) return(TACIO_FAULT);
  if(par->voiNr<1) return(TACIO_NOTABLE);

  /* Make sure that region names are available */
  if(verbose>1) printf("constructing region names\n");
  parEnsureNames(par);
  /* Construct region subnames */
  for(i=0; i<par->voiNr; i++)
    if(tacNameSplit(&par->voi[i].id)<1) return TACIO_FAULT;

  /* Write header lines */
  if(verbose>1) printf("writing headers\n");

  /* Program name plus copyright (c) YYYY for file format identification */
  if(strstr(par->program, "(c)") || strstr(par->program, "(C)")) {
    n=fprintf(fp, "%s\n\n", par->program);
  } else {
    if(strlen(par->program)>3)
      n=fprintf(fp, "%s (c) 2013\n\n", par->program);
    else
      n=fprintf(fp, "results (c) 2013\n\n");
  }
  if(n==0) return TACIO_CANNOTWRITE;

  /* Write calculation date and time */
  st=localtime(&par->time); 
  if(st!=NULL) strftime(tmp, 256, "%Y-%m-%d %H:%M:%S", st);
  else strcpy(tmp, "1900-01-01 00:00:00");
  fprintf(fp, "Date:\t%s\n", tmp);

  /* Write the studynr */
  if(par->studynr[0]) fprintf(fp, "Study:\t%s\n", par->studynr);

  /* Write the names of the original datafiles */
  if(par->datafile[0])   fprintf(fp, "Data file:\t%s\n", par->datafile);
  if(par->plasmafile[0]) fprintf(fp, "Plasma file:\t%s\n", par->plasmafile);
  if(par->plasmafile2[0]) fprintf(fp,"2nd Plasma file:\t%s\n",par->plasmafile2);
  if(par->bloodfile[0])  fprintf(fp, "Blood file:\t%s\n", par->bloodfile);
  if(par->reffile[0])    fprintf(fp, "Reference file:\t%s\n", par->reffile);
  if(par->refname[0])    fprintf(fp, "Reference region:\t%s\n", par->refname);

  /* Data fit range, sample nr, and parameter nr will be later saved as
     extra table columns */
  
  /* Write constants */
  if(par->density>0.0)   fprintf(fp, "Tissue density:\t%g\n", par->density);
  if(par->lc>0.0)        fprintf(fp, "Lumped constant:\t%g\n", par->lc);
  if(par->concentration>0.0)
                         fprintf(fp, "Concentration:\t%g\n",par->concentration);
  if(par->beta>0.0)      fprintf(fp, "Beta:\t%g\n", par->beta);
  if(par->Vb>=0.0)       fprintf(fp, "Vb:\t%g %%\n", par->Vb);
  if(par->fA>=0.0)       fprintf(fp, "fA:\t%g %%\n", par->fA);
  if(par->E>=0.0)        fprintf(fp, "Extraction:\t%g\n", par->E);

  /* Weighting */
  if(par->weighting==WEIGHTING_ON) strcpy(tmp, "yes");
  else if(par->weighting==WEIGHTING_OFF) strcpy(tmp, "no");
  else strcpy(tmp, "unknown");
  fprintf(fp, "Weighting:\t%s\n", tmp);

  /* should column be printed as integers (0), floats (1) or with exp (2)? */
  for(j=0; j<par->parNr; j++) {
    partype[j]=parPrintType(par, j);
  }

  /* Title line */
  n=fprintf(fp, "\n%s\t", "Region"); if(n<1) return TACIO_CANNOTWRITE;
  for(i=0; i<par->parNr; i++) {
    if(strlen(par->parname[i])<1) strcpy(tmp, ".");
    else strcpy(tmp, par->parname[i]);
    n=fprintf(fp, "\t%s", tmp); if(n<1) return TACIO_CANNOTWRITE;
  }
  /* Extra parameters, if available */
  if(parIsWSS(par))
    n=fprintf(fp, "\t%s", "WSS"); if(n<1) return TACIO_CANNOTWRITE;
  n=fprintf(fp, "\n"); if(n<1) return TACIO_CANNOTWRITE;

  /* Write units, if they exist, currently as comment line */
  for(i=j=0; i<par->parNr; i++) if(par->parunit[i]!=CUNIT_UNKNOWN) j++;
  if(j>0) {
    fprintf(fp, "%s", "# Units:");
    for(i=0; i<par->parNr; i++) {
      if(par->parunit[i]==CUNIT_UNKNOWN) strcpy(tmp, ".");
      else strcpy(tmp, petCunit(par->parunit[i]));
      n=fprintf(fp, "\t%s", tmp); if(n<1) return TACIO_CANNOTWRITE;
    }
    /* Extra parameters, if available */
    if(parIsWSS(par))
      n=fprintf(fp, "\t%s", "."); if(n<1) return TACIO_CANNOTWRITE;
    n=fprintf(fp, "\n"); if(n<1) return TACIO_CANNOTWRITE;
  }

  /* Write regional results */
  for(i=0; i<par->voiNr; i++) {
    /* VOI name */
    strcpy(tmp, par->voi[i].id.sub[0]);
    strcat(tmp, " ");
    if(MAX_SUBTACID_NR<2 || strlen(par->voi[i].id.sub[1])==0) strcat(tmp, ".");
    else strcat(tmp, par->voi[i].id.sub[1]);
    strcat(tmp, " ");
    if(MAX_SUBTACID_NR<3 || strlen(par->voi[i].id.sub[2])==0) strcat(tmp, ".");
    else strcat(tmp, par->voi[i].id.sub[2]);
    n=fprintf(fp, "%s", tmp); if(n<1) return TACIO_CANNOTWRITE;
    /* parameter values */
    p=par->voi[i].p;
    for(j=0; j<par->parNr; j++) {
      if(isnan(p[j])) {fprintf(fp, "\t."); continue;}
      switch(partype[j]) {
        case 0: fprintf(fp, "\t%.0f", p[j]); break;
        case 1:
          if(p[j]>=0) n=4; else n=3;
          fprintf(fp, "\t%.*f", n, p[j]);
          break;
        default:
          if(p[j]>=0) n=4; else n=3;
          fprintf(fp, "\t%.*e", n, p[j]);
        break;
      }
    }
    /* Extra parameters, if available */
    if(parIsWSS(par))
      n=fprintf(fp, "\t%e", par->voi[i].wss); if(n<1) return TACIO_CANNOTWRITE;
    n=fprintf(fp, "\n"); if(n<1) return TACIO_CANNOTWRITE;
    /* Write SD's, if they exist */
    p=par->voi[i].sd; for(j=n=0; j<par->parNr; j++) if(!isnan(p[j])) n++;
    if(n>0) {
      fprintf(fp, "SD . .");
      for(j=0; j<par->parNr; j++) {
        if(!isnan(p[j])) {
          switch(partype[j]) {
            case 0: fprintf(fp, "\t%.0f", p[j]); break;
            case 1:
              if(p[j]>=0) n=4; else n=3;
              fprintf(fp, "\t%.*f", n, p[j]);
              break;
            default:
              if(p[j]>=0) n=4; else n=3;
              fprintf(fp, "\t%.*e", n, p[j]);
              break;
          }
        } else {
          fprintf(fp, "\t.");
        }
      }
      /* Extra parameters, if available */
      if(parIsWSS(par)) n=fprintf(fp, "\t."); if(n<1) return TACIO_CANNOTWRITE;
      n=fprintf(fp, "\n"); if(n<1) return TACIO_CANNOTWRITE;
    }
    /* Write lower confidence limits, if they exist */
    p=par->voi[i].cl1; for(j=n=0; j<par->parNr; j++) if(!isnan(p[j])) n++;
    if(n>0) {
      fprintf(fp, "CL 95%% Lower");
      for(j=0; j<par->parNr; j++) {
        if(!isnan(p[j])) {
          switch(partype[j]) {
            case 0: fprintf(fp, "\t%.0f", p[j]); break;
            case 1:
              if(p[j]>=0) n=4; else n=3;
              fprintf(fp, "\t%.*f", n, p[j]);
              break;
            default:
              if(p[j]>=0) n=4; else n=3;
              fprintf(fp, "\t%.*e", n, p[j]);
              break;
          }
        } else {
          fprintf(fp, "\t.");
        }
      }
      /* Extra parameters, if available */
      if(parIsWSS(par)) n=fprintf(fp, "\t."); if(n<1) return TACIO_CANNOTWRITE;
      n=fprintf(fp, "\n"); if(n<1) return TACIO_CANNOTWRITE;
    }
    /* Write upper confidence limits, if they exist */
    p=par->voi[i].cl2; for(j=n=0; j<par->parNr; j++) if(!isnan(p[j])) n++;
    if(n>0) {
      fprintf(fp, "CL 95%% Upper");
      for(j=0; j<par->parNr; j++) {
        if(!isnan(p[j])) {
          switch(partype[j]) {
            case 0: fprintf(fp, "\t%.0f", p[j]); break;
            case 1:
              if(p[j]>=0) n=4; else n=3;
              fprintf(fp, "\t%.*f", n, p[j]);
              break;
            default:
              if(p[j]>=0) n=4; else n=3;
              fprintf(fp, "\t%.*e", n, p[j]);
              break;
          }
        } else {
          fprintf(fp, "\t.");
        }
      }
      /* Extra parameters, if available */
      if(parIsWSS(par)) n=fprintf(fp, "\t."); if(n<1) return TACIO_CANNOTWRITE;
      n=fprintf(fp, "\n"); if(n<1) return TACIO_CANNOTWRITE;
    }
  } /* next region */

  return(TACIO_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read parameter data into PETPAR struct from specified RES file.
 *  Reading older RES files are supported.
\return Returns 0 when successful, otherwise TACIO error code.
 */
int parReadRES(
  /** Pointer to PETPAR data */
  PETPAR *par,
  /** Output filename */
  char *filename,
  /** Verbose level; if zero, then only warnings are printed into stderr */
  int verbose
) {
  int i, ret, lineNr, colNr;
  IFT ift;

  if(verbose>0) printf("parReadRES(par, %s, ...)\n", filename);
  /* Check input */
  if(par==NULL || filename==NULL || strlen(filename)<1) return(TACIO_FAULT);

  /* Delete any previous data */
  parEmpty(par);

  /* Read file into IFT struct */
  if(verbose>1) printf("reading file\n");
  iftInit(&ift);
  ret=iftRead(&ift, filename, 0);
  if(ret!=IFT_OK) {
    if(verbose>0) printf("Error in reading: %s\n", ift.status);
    iftEmpty(&ift); if(ret==IFT_CANNOTREAD) return TACIO_CANNOTOPEN;
    return TACIO_INVALIDFORMAT;
  }
  if(verbose>1) printf("%d line(s).\n", ift.keyNr);

  /* Get data dimensions */
  if(verbose>1) printf("reading dimensions of the data\n");
  ret=iftGetDataDimensions(&ift, &lineNr, &colNr);
  if(ret!=IFT_OK) {
    if(verbose>0) printf("error %d in reading dimensions of the data\n", ret);
    iftEmpty(&ift); return TACIO_INVALIDFORMAT;
  }
  if(verbose>2) {
    printf("data_lines := %d\n", lineNr);
    printf("max_column_nr := %d\n", colNr);
  }
  /* Subtract SD and CL lines */
  for(i=0; i<ift.keyNr; i++) {
    if(strncasecmp(ift.item[i].value, "SD ", 3)==0) {lineNr--; continue;}
    if(strncasecmp(ift.item[i].value, "CL 95 ", 5)==0) {lineNr--; continue;}
  }
  if(verbose>2) {
    printf("fixed_data_lines := %d\n", lineNr);
  }

  /* Allocate memory for parameters */
  if(verbose>1) printf("allocating memory for parameters\n");
  ret=parAllocate(par, lineNr);
  if(ret!=TACIO_OK) {iftEmpty(&ift); parEmpty(par); return ret;}
  
  /* Move data from IFT */
  if(verbose>1) printf("moving data table from IFT\n");
  ret=parProcessRESData(par, &ift, verbose-10);

  if(verbose>4) parPrint(par);

  if(ret!=TACIO_OK) {iftEmpty(&ift); parEmpty(par); return ret;}
  iftEmpty(&ift);
  
  /* Set study number from filename, if necessary */
  if(strlen(par->studynr)==0) studynr_from_fname(filename, par->studynr);

  //return TACIO_INVALIDFORMAT;
  return TACIO_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy RES data from IFT to PETPAR struct.
\return Returns TACIO status.
 */
int parProcessRESData(
  /** Pointer to allocated PETPAR struct into where data is copied */
  PETPAR *par,
  /** Pointer to IFT struct from where the data is read */
  IFT *ift,
  /** Verbose level; if zero, nothing is printed into stdout */
  int verbose
) {
  int ret, i, j, n, pi, title_index, colNr;
  char *cptr, tmp[256];
  double v;
  
  if(verbose>0) printf("parProcessRESData()\n");
  if(par==NULL || ift==NULL) return TACIO_FAULT;
  if(verbose>3) {
    printf("keyNr := %d\n", ift->keyNr);
  }

  /* Remove comment lines from IFT struct, except units */
  i=0; while(i<ift->keyNr) {
    if(ift->item[i].type!='#' && ift->item[i].type!=';') {i++; continue;}
    if(strcasecmp(ift->item[i].key, "Units")==0) {i++; continue;}
    iftDeleteItem(ift, i);
  }

  /* Read program name, must be the first line */
  if(verbose>2) printf("reading program name\n");
  for(i=0; i<ift->keyNr; i++) {
    if(strlen(ift->item[i].value)<1) return(TACIO_INVALIDFORMAT);
    if(strlen(ift->item[i].key)>0) return(TACIO_INVALIDFORMAT);
    cptr=ift->item[i].value;
    if(strstr(cptr, "(c)") || strstr(cptr, "(C)")) {
      strncpy(par->program, cptr, 1023); par->program[1023]=(char)0;
    } else {
      return(TACIO_INVALIDFORMAT);
    }
    break;
  }
    
  /* Read header lines from IFT */
  if(verbose>2) printf("reading header lines\n");

  i=iftGetFrom(ift, 0, "Date");
  if(i>=0) {
    if(verbose>3) printf("date_str := '%s'\n", ift->item[i].value);
    struct tm st;
    if(get_datetime(ift->item[i].value, &st)==0) par->time=mktime(&st);
    else if(get_date(ift->item[i].value, &st)==0) par->time=mktime(&st);
    else par->time=(time_t)0;
  }
  
  i=iftGetFrom(ift, 0, "Study");
  if(i>=0) strncpy(par->studynr, ift->item[i].value, MAX_STUDYNR_LEN);
  
  i=iftGetFrom(ift, 0, "Data file");
  if(i>=0) strncpy(par->datafile, ift->item[i].value, FILENAME_MAX);
  i=iftGetFrom(ift, 0, "ROI file");
  if(i>=0) strncpy(par->datafile, ift->item[i].value, FILENAME_MAX);
  
  i=iftGetFrom(ift, 0, "Plasma file");
  if(i>=0) strncpy(par->plasmafile, ift->item[i].value, FILENAME_MAX);
  
  i=iftGetFrom(ift, 0, "2nd Plasma file");
  if(i>=0) strncpy(par->plasmafile2, ift->item[i].value, FILENAME_MAX);
  
  i=iftGetFrom(ift, 0, "Blood file");
  if(i>=0) strncpy(par->bloodfile, ift->item[i].value, FILENAME_MAX);
  
  i=iftGetFrom(ift, 0, "Reference file");
  if(i>=0) strncpy(par->reffile, ift->item[i].value, FILENAME_MAX);
  
  i=iftGetFrom(ift, 0, "Reference region");
  if(i>=0) strncpy(par->refname, ift->item[i].value, MAX_REGIONNAME_LEN);
  
  i=iftGetFrom(ift, 0, "Fit method");
  if(i>=0) strncpy(par->fitmethod, ift->item[i].value, FILENAME_MAX);
  
  i=iftGetFrom(ift, 0, "Tissue density");
  if(i>=0) {
    ret=atof_with_check(ift->item[i].value, &v);
    if(ret==0) par->density=v;
  }
  
  i=iftGetFrom(ift, 0, "Lumped constant");
  if(i>=0) {
    ret=atof_with_check(ift->item[i].value, &v);
    if(ret==0) par->lc=v;
  }
  
  i=iftGetFrom(ift, 0, "Concentration");
  if(i>=0) {
    ret=atof_with_check(ift->item[i].value, &v);
    if(ret==0) par->concentration=v;
  }
  
  i=iftGetFrom(ift, 0, "Beta");
  if(i>=0) {
    ret=atof_with_check(ift->item[i].value, &v);
    if(ret==0) par->beta=v;
  }
  
  i=iftGetFrom(ift, 0, "Vb");
  if(i>=0) {
    ret=atof_with_check(ift->item[i].value, &v);
    if(ret==0) par->Vb=v;
  }
  
  i=iftGetFrom(ift, 0, "fA");
  if(i>=0) {
    ret=atof_with_check(ift->item[i].value, &v);
    if(ret==0) par->fA=v;
  }
  
  i=iftGetFrom(ift, 0, "Extraction");
  if(i>=0) {
    ret=atof_with_check(ift->item[i].value, &v);
    if(ret==0) par->E=v;
  }

  par->weighting=WEIGHTING_UNKNOWN;
  if(iftGetFrom(ift, 0, "Data was weighted")>=0)
    par->weighting=WEIGHTING_ON;
  if(iftGetFrom(ift, 0, "Data was not weighted")>=0)
    par->weighting=WEIGHTING_OFF;
  i=iftGetFrom(ift, 0, "Weighting");
  if(i>=0) {
    if(strncasecmp(ift->item[i].value, "yes", 1)==0)
      par->weighting=WEIGHTING_ON;
    else if(strncasecmp(ift->item[i].value, "no", 1)==0)
      par->weighting=WEIGHTING_OFF;
  }

  /* Read the result parameter title line */
  if(verbose>2) printf("reading parameter titles\n");
  title_index=-1;
  for(i=0; i<ift->keyNr; i++) {
    if(strlen(ift->item[i].key)>0) continue;
    if(strncasecmp(ift->item[i].value, "Region", 6)==0) {title_index=i; break;}
  }
  if(title_index<0) return(TACIO_NOTABLE);
  /* Read the nr of columns from the title line */
  colNr=strTokenNr(ift->item[title_index].value+6, " \t");
  if(colNr<1) return(TACIO_INVALIDFORMAT);
  if(verbose>2) printf("colNr := %d\n", colNr);
  /* Allocate array for column types */
  int columns[colNr];
  char parname[MAX_PARNAME_LEN];
  /* Read parameter names, identify special columns, and set parNr
     accordingly */
  for(pi=0, par->parNr=0; pi<colNr; pi++) {
    columns[pi]=0;
    ret=strTokenNCpy(ift->item[title_index].value+6, " \t", pi+1,
                     parname, MAX_PARNAME_LEN);
    if(ret==0) return(TACIO_INVALIDFORMAT);
    if(ret==1 && strcmp(parname, ".")==0) strcpy(parname, "");
    if(strcasecmp(parname, "WSS")==0) {columns[pi]=1; continue;}
    if(strcasecmp(parname, "Start")==0) {columns[pi]=2; continue;}
    if(strcasecmp(parname, "End")==0) {columns[pi]=3; continue;}
    if(strcasecmp(parname, "parNr")==0) {columns[pi]=4; continue;}
    if(strcasecmp(parname, "dataNr")==0) {columns[pi]=5; continue;}
    if(strcasecmp(parname, "SD")==0) {columns[pi]=6; continue;}
    if(strcasecmp(parname, "95%CLl")==0) {columns[pi]=7; continue;}
    if(strcasecmp(parname, "95%CLu")==0) {columns[pi]=8; continue;}
    /* this is actual parameter column; check that there are not too many */
    if(par->parNr>=MAX_PETPAR_NR) return(TACIO_TOOBIG);
    strcpy(par->parname[par->parNr], parname);
    par->parNr++;
  }
  if(verbose>2) printf("parNr := %d\n", par->parNr);

  /* Read the result parameter unit line */
  i=iftGetFrom(ift, title_index+1, "Units");
  if(i>title_index+1) return(TACIO_INVALIDFORMAT);
  if(i<0) {
    if(verbose>5) printf("  no units line identified.\n");
  } else {
    title_index+=1;
    if(verbose>2) printf("reading parameter units\n");
    n=strTokenNr(ift->item[title_index].value, " \t");
    if(n!=colNr) {
      if(verbose>0) printf("invalid units line\n");
      return(TACIO_INVALIDFORMAT);
    }
    for(pi=n=0; pi<colNr; pi++) if(columns[pi]==0) {
      ret=strTokenNCpy(ift->item[title_index].value, " \t", pi+1, tmp, 256);
      if(ret==0) return(TACIO_INVALIDFORMAT);
      par->parunit[n++]=petCunitId(tmp);
    }
  }

  /* Read regional results */
  if(verbose>2) printf("reading results table\n");
  par->voiNr=0;
  for(i=title_index+1; i<ift->keyNr; i++) {
    if(verbose>7) printf("'%s'\n", ift->item[i].value);
    if(strlen(ift->item[i].key)>0) break;
    n=strTokenNr(ift->item[i].value, " \t");
    if(n!=(colNr+3)) {if(verbose>7) printf("n=%d\n", n); break;}
    /* Check that we don't have too many ROIs */
    if(par->voiNr>=par->_voidataNr) return(TACIO_TOOBIG);
    /* Read VOI name (the first three columns) */
    for(j=0; j<3 && j<MAX_SUBTACID_NR; j++) {
      ret=strTokenNCpy(ift->item[i].value, " \t", j+1,
                       par->voi[par->voiNr].id.sub[j], MAX_TACNAME_LEN+1);
      if(ret==0) return(TACIO_INVALIDFORMAT);
    }
    /* Construct full TAC name */
    tacNameCatenate(&par->voi[par->voiNr].id);
    if(verbose>8) printf("name := '%s'\n", par->voi[par->voiNr].id.name);

    /* Read parameter values */
    for(pi=n=0; pi<colNr; pi++) {
      ret=strTokenNCpy(ift->item[i].value, " \t", 3+pi+1, tmp, 256);
      if(ret==0) return(TACIO_INVALIDFORMAT);
      if(ret==1 && tmp[0]=='.') v=nan("");
      else if(atof_with_check(tmp, &v)) return(TACIO_INVALIDFORMAT);
      if(columns[pi]==0) par->voi[par->voiNr].p[n]=v;
      else if(columns[pi]==1) par->voi[par->voiNr].wss=v;
      else if(columns[pi]==2) par->voi[par->voiNr].start=v;
      else if(columns[pi]==3) par->voi[par->voiNr].end=v;
      else if(columns[pi]==4) par->voi[par->voiNr].parNr=v;
      else if(columns[pi]==5) par->voi[par->voiNr].dataNr=v;
      else if(columns[pi]==6) par->voi[par->voiNr].sd[n]=v;
      else if(columns[pi]==7) par->voi[par->voiNr].cl1[n]=v;
      else if(columns[pi]==8) par->voi[par->voiNr].cl2[n]=v;
      n++;
    }

    /* If 'region' name implies that this was confidence limit or sd, then */
    /* move the values into correct places, and do not increase the voiNr */
    if(par->voiNr==0) {par->voiNr++; continue;}
    if(strcasecmp(par->voi[par->voiNr].id.sub[0], "CL")==0) {
      if(strcmp(par->voi[par->voiNr].id.sub[1], "95%")==0) {
        if(strcasecmp(par->voi[par->voiNr].id.sub[2], "Lower")==0)
          for(pi=0; pi<par->parNr; pi++)
            par->voi[par->voiNr-1].cl1[pi]=par->voi[par->voiNr].p[pi];
        else if(strcasecmp(par->voi[par->voiNr].id.sub[2], "Upper")==0)
          for(pi=0; pi<par->parNr; pi++)
            par->voi[par->voiNr-1].cl2[pi]=par->voi[par->voiNr].p[pi];
        continue;
      }
    } else if(strcasecmp(par->voi[par->voiNr].id.sub[0], "SD")==0) {
      for(pi=0; pi<par->parNr; pi++)
        par->voi[par->voiNr-1].sd[pi]=par->voi[par->voiNr].p[pi];
      continue;
    }
    par->voiNr++;
  }

  /* Check the data table size the was read */
  if(par->parNr<1 || par->voiNr<1) return TACIO_NOTABLE;

  return TACIO_OK;
}
/*****************************************************************************/

/*****************************************************************************/
