/******************************************************************************
  Copyright (c) 2013 by Turku PET Centre

  File:        taccomp.c
  Description: Contains functions for comparing TAC data.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  2013-09-11 Vesa Oikonen
       First created, based on libtpccurveio.


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/tacio.h"
/*****************************************************************************/

/*****************************************************************************/
/** Check whether time and concentration units are the same in two TAC data.
\return Returns 0 in case of match, and 1 if concentration unit is not matching,
    2 if time unit is not matching, and 3 if neither is matching.
 */
int tacMatchUnit(
  /** Pointer to TAC struct */
  TAC *d1,
  /** Pointer to TAC struct */
  TAC *d2,
  /** Verbose level; if zero, then nothing is printed into stdout or stderr */
  int verbose
) {
  if(verbose>0) printf("tacMatchUnit()\n");
  int ret=0;
  if(d1==NULL || d2==NULL) return(3);
  if(d1->cunit!=d2->cunit) ret+=1;
  if(d1->tunit!=d2->tunit) ret+=2;
  if(ret>0 && verbose>0) {
    printf("tac1.cunit := %s\n", petCunit(d1->cunit));
    printf("tac2.cunit := %s\n", petCunit(d2->cunit));
    printf("tac1.tunit := %s\n", petTunit(d1->tunit));
    printf("tac2.tunit := %s\n", petTunit(d2->tunit));
  }
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether TAC names are the same in two TAC data.
\return Returns 0 in case of match, or 1 if name is not matching.
 */
int tacMatchTacID(
  /** Pointer to TAC struct */
  TAC *d1,
  /** Pointer to TAC struct */
  TAC *d2,
  /** Test only name field (0), test only sub names (1), or test all names (2)*/
  int whatname,
  /** Verbose level; if zero, then nothing is printed into stdout or stderr */
  int verbose
) {
  int ri, si;
  
  if(verbose>0) printf("tacMatchTacID()\n");
  if(d1==NULL || d2==NULL) return(1);
  if(d1->voiNr!=d2->voiNr) {
    if(verbose>0) printf("different TAC nr.\n");
    return(1);
  }
  if(whatname!=1) { // test name
    for(ri=0; ri<d1->voiNr; ri++) {
      if(strcmp(d1->voi[ri].id.name, d2->voi[ri].id.name)!=0) {
        if(verbose>0) {
          printf("d1.voi[%d].id.name := '%s'\n", ri, d1->voi[ri].id.name);
          printf("d2.voi[%d].id.name := '%s'\n", ri, d2->voi[ri].id.name);
        }
        return(1);
      }
    }
  }
  if(whatname!=0) { // test sub names
    for(ri=0; ri<d1->voiNr; ri++) for(si=0; si<MAX_SUBTACID_NR; si++) {
      if(strcmp(d1->voi[ri].id.sub[si], d2->voi[ri].id.sub[si])!=0) {
        if(verbose>0) {
          printf("d1.voi[%d].id.sub[%d] := '%s'\n",
	          ri, si, d1->voi[ri].id.sub[si]);
          printf("d2.voi[%d].id.sub[%d] := '%s'\n",
	          ri, si, d2->voi[ri].id.sub[si]);
        }
        return(1);
      }
    }
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether TAC concentrations (y values) are the same in two TAC data.
\return Returns 0 in case of match.
 */
int tacMatchConc(
  /** Pointer to TAC struct */
  TAC *d1,
  /** Pointer to TAC struct */
  TAC *d2,
  /** Limit for accepted absolute difference */
  double test_abs,
  /** Verbose level; if zero, then nothing is printed into stdout or stderr */
  int verbose
) {
  int ri, fi;
  
  if(verbose>0) printf("tacMatchConc(tac1, tac2, %g, ...)\n", test_abs);
  if(d1==NULL || d2==NULL) return(1);
  if(d1->voiNr!=d2->voiNr) {
    if(verbose>0) printf("different TAC nr.\n"); return(2);}
  if(d1->frameNr!=d2->frameNr) {
    if(verbose>0) printf("different sample nr.\n"); return(3);}
  for(ri=0; ri<d1->voiNr; ri++) {
    for(fi=0; fi<d1->frameNr; fi++) {
      if(doubleMatch(d1->voi[ri].y[fi], d2->voi[ri].y[fi], test_abs)==0)
        continue;
      if(verbose>0) {
        double s;
        s=fabs(d1->voi[ri].y[fi]-d2->voi[ri].y[fi]);
        printf("tac1.voi[%d].y[%d] := %g\n", ri, fi, d1->voi[ri].y[fi]);
        printf("tac2.voi[%d].y[%d] := %g\n", ri, fi, d2->voi[ri].y[fi]);
        printf("|diff| := %g\n", s);
        printf("diff_limit := %g\n", test_abs);
      }
      return(10);
    }
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether TAC sample times (x or x1 and x2) are the same in two TACs.
\return Returns 0 in case of match.
 */
int tacMatchFrames(
  /** Pointer to TAC struct */
  TAC *d1,
  /** Pointer to TAC struct */
  TAC *d2,
  /** Limit for accepted absolute difference */
  double test_abs,
  /** Verbose level; if zero, then nothing is printed into stdout or stderr */
  int verbose
) {
  int fi;
  
  if(verbose>0) printf("tacMatchFrames(tac1, tac2, %g, ...)\n", test_abs);
  if(d1==NULL || d2==NULL) return(1);
  if(d1->frameNr!=d2->frameNr) {
    if(verbose>0) printf("different sample nr.\n"); return(2);}
  if(d1->isframe!=d2->isframe) {
    if(verbose>0) printf("different sampling type.\n"); return(3);}
  for(fi=0; fi<d1->frameNr; fi++) {
    if(d1->isframe==0) {
      if(doubleMatch(d1->x[fi], d2->x[fi], test_abs)==0)
        continue;
      if(verbose>0) {
        printf("tac1.x[%d] := %g\n", fi, d1->x[fi]);
        printf("tac2.x[%d] := %g\n", fi, d2->x[fi]);
        printf("diff_limit := %g\n", test_abs);
      }
      return(10);
    } else {
      if(doubleMatch(d1->x1[fi], d2->x1[fi], test_abs)==0 &&
         doubleMatch(d1->x2[fi], d2->x2[fi], test_abs)==0)
        continue;
      if(verbose>0) {
        printf("tac1.x1[%d] := %g\n", fi, d1->x1[fi]);
        printf("tac2.x1[%d] := %g\n", fi, d2->x1[fi]);
        printf("tac1.x2[%d] := %g\n", fi, d1->x2[fi]);
        printf("tac2.x2[%d] := %g\n", fi, d2->x2[fi]);
        printf("diff_limit := %g\n", test_abs);
      }
      return(11);
    }
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
