/******************************************************************************

  Copyright (c) 2013 Turku PET Centre

  File:        taccsv.c
  Description: Functions for TAC files in CSV formats.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

  Turku PET Centre hereby disclaims all copyright interest in the program.

  Modification history:
  2013-09-11 Vesa Oikonen
       First created.

******************************************************************************/

/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/tacio.h"
/*****************************************************************************/

/*****************************************************************************/
/** Write TAC data into specified file pointer in CSV or TSV format.
 *  Column and decimal separator is written as stated by TAC format field.
\return Returns TACIO status.
 */
int tacWriteCSV(
  /** Pointer to TAC struct, contents of which are to be written */
  TAC *tac,
  /** File pointer */
  FILE *fp,
  /** Verbose level; if zero, nothing extra is printed into stdout */
  int verbose
) {
  char tmp[512];
  double v;

  if(verbose>0) printf("tacWriteCSV(tac, fp, ...)\n");
  /* Check input */
  if(tac==NULL || fp==NULL) return(TACIO_FAULT);
  if(tac->voiNr<1 || tac->frameNr<1) return(TACIO_NOTABLE);

  /* Set decimal and item separators */
  int tointl=0; // Decimal point (0) or comma (1)
  char ss[2];
  if(tac->format==TAC_FORMAT_CSV_INT) {
    tointl=1; strcpy(ss, ";");
  } else if(tac->format==TAC_FORMAT_TSV_INT) {
    tointl=1; strcpy(ss, "\t");
  } else if(tac->format==TAC_FORMAT_TSV_UK) {
    tointl=0; strcpy(ss, "\t");
  } else { // default
    tointl=0; strcpy(ss, ",");
  }

  /* Write file format ID and study number */
  if(verbose>1) printf("  writing header\n");
  sprintf(tmp, "TAC%s%s\n", ss, tac->studynr);
  if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);

  /* Write obligatory table title line */
  if(verbose>1) printf("  writing data table title\n");
  if(tac->isframe==0) sprintf(tmp, "time[%s]", petTunit(tac->tunit));
  else sprintf(tmp, "start[%s]%send[%s]", petTunit(tac->tunit), ss,
               petTunit(tac->tunit));
  if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
  for(int ri=0; ri<tac->voiNr; ri++) {
    /* Column separator */
    if(csvWriteText(fp, ss, 0)) return(TACIO_CANNOTWRITE);
    /* write TAC names, must not include spaces */
    strcpy(tmp, tac->voi[ri].id.name);
    strReplaceChar(tmp, ' ', '_');
    strReplaceChar(tmp, '\t', '_');
    if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
    /* First TAC name should also include the concentration unit */
    if(ri==0) {
      sprintf(tmp, "[%s]", petCunit(tac->cunit));
      if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
    }  
  }
  if(tac->weighting==WEIGHTING_ON) {
    sprintf(tmp, "%sweight", ss);
    if(csvWriteText(fp, tmp, 0)) return(TACIO_CANNOTWRITE);
  }
  if(csvWriteText(fp, "\n", 0)) return(TACIO_CANNOTWRITE);

  /* Write data */
  if(verbose>1) printf("writing data table\n");
  for(int fi=0; fi<tac->frameNr; fi++) {
    /* Time(s) (x, or x1 and x2) */
    if(tac->isframe==0) v=tac->x[fi]; else v=tac->x1[fi];
    if(isnan(v)) strcpy(tmp, ""); else sprintf(tmp, "%g", v);
    if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
    if(tac->isframe!=0) {
      v=tac->x2[fi];
      if(isnan(v)) strcpy(tmp, ss); else sprintf(tmp, "%s%g", ss, v);
      if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
    }
    /* Concentrations (y values) */
    for(int ri=0; ri<tac->voiNr; ri++) {
      v=tac->voi[ri].y[fi];
      if(isnan(v)) strcpy(tmp, ss); else sprintf(tmp, "%s%g", ss, v);
      if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
    }
    /* Weight */
    if(tac->weighting==WEIGHTING_ON) {
      v=tac->w[fi];
      if(isnan(v)) strcpy(tmp, ss); else sprintf(tmp, "%s%g", ss, v);
      if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
    }
    if(csvWriteText(fp, "\n", 0)) return(TACIO_CANNOTWRITE);
  }
  
  /* Write VOI volumes, if available */
  if(tacIsSize(tac)) {
    strcpy(tmp, "volume"); if(tac->isframe) strcat(tmp, ss);
    if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
    for(int ri=0; ri<tac->voiNr; ri++) {
      v=tac->voi[ri].size;
      if(isnan(v)) strcpy(tmp, ss); else sprintf(tmp, "%s%g", ss, v);
      if(csvWriteText(fp, tmp, tointl)) return(TACIO_CANNOTWRITE);
    }
    if(csvWriteText(fp, "\n", 0)) return(TACIO_CANNOTWRITE);
  }
  
  return(TACIO_OK);
}
/*****************************************************************************/

/*****************************************************************************/


#if(0)
/** Reads different CSV formats into DFT struct
\return Returns 0 when successful, otherwise an error code.
 */
int csv2dft(
  /** Pointer to CSV data to be converted */
  CSV *csv,
  /** Pointer to empty DFT struct which will be allocated and filled here */
  DFT *dft
) {
  int ret, i, u, n, m;

  if(CSV_TEST>2) {printf("csv2dft()\n"); fflush(stdout);}
  if(csv==NULL || dft==NULL) return CSV_ERROR;
  if(csv->row_nr<1 || csv->col_nr<1) return CSV_INVALIDFORMAT;

  /* Many CSV formats are impossible to identify, therefore we will just
     try to convert different formats until one succeeds or all are failed */
  if(CSV_TEST>2) printf("trying to read 1st CSV format\n");
  ret=csv2dft_a(csv, dft);
  if(ret!=CSV_OK) {
    if(CSV_TEST>2) printf("reading 1st CSV format failed; trying 2nd format\n");
    ret=csv2dft_b(csv, dft);
  }
  if(ret!=CSV_OK) {
    if(CSV_TEST>2) printf("2nd CSV format failed\n");
  }
  if(ret!=CSV_OK) return ret;
  /* Make sure that TAC names are filled */
  u=dft->voiNr; n=1; while((u/=10)>=1) n++;
  if(n>MAX_REGIONSUBNAME_LEN) n=MAX_REGIONSUBNAME_LEN;
  for(i=0, m=0; i<dft->voiNr; i++) {
    if(strlen(dft->voi[i].voiname)<1 || strcmp(dft->voi[i].voiname, ".")==0) {
      sprintf(dft->voi[i].voiname, "%0*d", n, i+1);
      strcpy(dft->voi[i].name, dft->voi[i].voiname);
      m++;
    }
  }
  /* If none of TACs had a name, then set DFT plain format */
  if(m==dft->voiNr) dft->_type=DFT_FORMAT_PLAIN;
  return CSV_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Reads simple and Inveon type 1 data into DFT struct
\return Returns 0 when successful, otherwise an error code.
 */
int csv2dft_a(
  /** Pointer to CSV data to be converted */
  CSV *csv,
  /** Pointer to empty DFT struct which will be allocated and filled here */
  DFT *dft
) {
  int ri, ci, sci, ii, ret;
  char *cptr;

  if(CSV_TEST>2) {printf("csv2dft_a()\n"); fflush(stdout);}
  if(csv==NULL || dft==NULL) return CSV_ERROR;
  if(csv->row_nr<1 || csv->col_nr<1) return CSV_INVALIDFORMAT;
  dftEmpty(dft);
  /* Allocate memory for DFT */
  if(CSV_TEST>2) {
    printf("frame_nr=%d voi_nr=%d\n", csv->row_nr, csv->col_nr-1); 
    fflush(stdout);
  }
  ret=dftSetmem(dft, csv->row_nr, csv->col_nr-1);
  if(ret!=0) return CSV_OUTOFMEMORY;
  /* Set DFT defaults */
  dft->timetype=DFT_TIME_MIDDLE;
  dft->_type=DFT_FORMAT_STANDARD;
  dft->isweight=0;
  dftUnitToDFT(dft, CUNIT_UNKNOWN);
  dft->timeunit=TUNIT_UNKNOWN;
  for(ri=0; ri<csv->row_nr; ri++) dft->w[ri]=1.0;
  for(ci=0; ci<csv->col_nr-1; ci++) dft->voi[ci].sw=0;

  /* Fill DFT */
  ri=0;
  for(ii=0; ii<csv->nr;) {
    // goto start of row
    for(; ii<csv->nr && csv->c[ii].col!=1; ii++) {}
    if(ii==csv->nr) break;
    if(CSV_TEST>10) {
      printf("\nline start at %d\n", ii); 
      printf("  ri=%d\n", ri); 
      fflush(stdout);
    }
    // ignore line with empty first column
    if(csv->c[ii].content==NULL) {
      if(CSV_TEST>11) {printf("  empty first column\n"); fflush(stdout);}
      ii++; continue;
    }
    // ignore comment line
    if(csv->c[ii].content[0]=='#') {
      if(CSV_TEST>11) {printf("  comment line\n"); fflush(stdout);}
      ii++; continue;
    }
    // ignore line that does not start with number
    // unless it contains TAC titles
    if(!isdigit(csv->c[ii].content[0]) && csv->c[ii].content[0]!='-') {
      if(strstr(csv->c[ii].content, "Time")==NULL &&
         strstr(csv->c[ii].content, "TIME")==NULL &&
         strstr(csv->c[ii].content, "time")==NULL) {
        if(CSV_TEST>11) {
          printf("  not a numerical value or title\n"); fflush(stdout);}
        ii++; continue;
      }
      dft->_type=DFT_FORMAT_STANDARD;
      if(strncasecmp(csv->c[ii].content, "Start time", 10)==0 &&
         strncasecmp(csv->c[ii+1].content, "End time", 8)==0) {
        dft->timetype=DFT_TIME_STARTEND;
        if(CSV_TEST>6) printf("timetype := %d\n", dft->timetype);
      }
      if(CSV_TEST>7) {
        printf("first title field := '%s'\n", csv->c[ii].content);
        fflush(stdout);
      }
      if(strstr(csv->c[ii].content, "min")!=NULL) dft->timeunit=TUNIT_MIN;
      else if(strstr(csv->c[ii].content, "sec")!=NULL) dft->timeunit=TUNIT_SEC;
      else dft->timeunit=TUNIT_UNKNOWN;
      ii++;
      
      if(dft->timetype==DFT_TIME_MIDDLE) sci=2; else {sci=3; ii++;}
      for(ci=sci; ci<=csv->col_nr && ii<csv->nr; ci++, ii++) {
        if(CSV_TEST>2) {
          printf("col=%d row=%d\n", csv->c[ii].col, csv->c[ii].row);
          if(CSV_TEST>3) printf("ci=%d ii=%d\n", ci, ii);
          fflush(stdout);
        }
        if(csv->c[ii].col!=ci) {dftEmpty(dft); return CSV_NOTABLE;}
        if(csv->c[ii].content!=NULL) {
          /* Check if this column should be omitted from DFT */
          if(strstr(csv->c[ii].content, " - Time")!=NULL) {
            if(CSV_TEST>2) printf("  ignored time column.\n");
            dft->voi[ci-sci].sw=1; continue;
          }
          if(strstr(csv->c[ii].content, "(upper bound)")!=NULL) {
            if(CSV_TEST>2) printf("  ignored upper bound column.\n");
            dft->voi[ci-sci].sw=2; continue;
          }
          if(strstr(csv->c[ii].content, "(lower bound)")!=NULL) {
            if(CSV_TEST>2) printf("  ignored lower bound column.\n");
            dft->voi[ci-sci].sw=3; continue;
          }
          if(strstr(csv->c[ii].content, "(standard deviation)")!=NULL) {
            if(CSV_TEST>2) printf("  ignored s.d. column.\n");
            dft->voi[ci-sci].sw=4; continue;
          }
          /* Search calibration unit from name */
          if(petCunitId(dft->unit)==CUNIT_UNKNOWN) {
            if(strstr(csv->c[ii].content, "(Bq/ml)")!=NULL)
              dftUnitToDFT(dft, CUNIT_BQ_PER_ML);
            else if(strstr(csv->c[ii].content, "(kBq/ml)")!=NULL)
              dftUnitToDFT(dft, CUNIT_KBQ_PER_ML);
            else if(strstr(csv->c[ii].content, "(MBq/ml)")!=NULL)
              dftUnitToDFT(dft, CUNIT_MBQ_PER_ML);
            else if(strstr(csv->c[ii].content, "(% ID/g)")!=NULL)
              dftUnitToDFT(dft, CUNIT_PIDM);
            else if(strstr(csv->c[ii].content, "Bq/ml")!=NULL)
              dftUnitToDFT(dft, CUNIT_BQ_PER_ML);
            else if(strstr(csv->c[ii].content, "kBq/ml")!=NULL)
              dftUnitToDFT(dft, CUNIT_KBQ_PER_ML);
            else if(strstr(csv->c[ii].content, "MBq/ml")!=NULL)
              dftUnitToDFT(dft, CUNIT_MBQ_PER_ML);
            else if(strstr(csv->c[ii].content, "% ID/g")!=NULL)
              dftUnitToDFT(dft, CUNIT_PIDM);
          }
        }
        if(csv->c[ii].content==NULL) {
          sprintf(dft->voi[ci-sci].name, "%d", ci-sci+1);
          strcpy(dft->voi[ci-sci].voiname, dft->voi[ci-sci].name);
        } else {
          cptr=strstr(csv->c[ii].content, " - "); if(cptr!=NULL) *cptr=(char)0;
          strncpy(dft->voi[ci-sci].name, csv->c[ii].content, MAX_REGIONNAME_LEN);
          rnameSplit(csv->c[ii].content, dft->voi[ci-sci].voiname,
  	  dft->voi[ci-sci].hemisphere, dft->voi[ci-sci].place,
          MAX_REGIONSUBNAME_LEN);
        }
        if(CSV_TEST>8) printf("name[%d]=%s\n", ci-sci, dft->voi[ci-sci].name);
      }
    }
    
    // check that allocated DFT frame nr is not exceeded
    if(ri>=csv->row_nr) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
    // read the sample time
    if(dft->timetype==DFT_TIME_MIDDLE) {
      dft->x[ri]=atof_dpi(csv->c[ii++].content);
      if(CSV_TEST>3) printf("x[%d]=%g\n", ri, dft->x[ri]);
    } else {
      dft->x1[ri]=atof_dpi(csv->c[ii++].content);
      dft->x2[ri]=atof_dpi(csv->c[ii++].content);
      dft->x[ri]=0.5*(dft->x1[ri]+dft->x2[ri]);
      if(CSV_TEST>3)
        printf("x1[%d]=%g x2[%d]=%g\n", ri, dft->x1[ri], ri, dft->x2[ri]);
    }
    // read the sample values
    if(dft->timetype==DFT_TIME_MIDDLE) sci=2; else sci=3;
    for(ci=sci; ci<=csv->col_nr && ii<csv->nr; ci++, ii++) {
      if(CSV_TEST>2) {
        printf("  col=%d row=%d\n", csv->c[ii].col, csv->c[ii].row);
        if(CSV_TEST>3) printf("  ci=%d ii=%d\n", ci, ii);
        fflush(stdout);
      }
      if(csv->c[ii].col!=ci) {dftEmpty(dft); return CSV_NOTABLE;}
      if(csv->c[ii].content==NULL) dft->voi[ci-sci].y[ri]=nan("");
      else dft->voi[ci-sci].y[ri]=atof_dpi(csv->c[ii].content);
      if(CSV_TEST>4)
        printf("  y[%d][%d]=%g\n", ri, ci-sci, dft->voi[ci-sci].y[ri]);
    }
    ri++; //if(ri>csv->row_nr) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
  }
  if(CSV_TEST>1) printf("  %d frame(s) read from CSV\n", ri);
  if(ri<1) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
  dft->frameNr=ri;
  dft->voiNr=csv->col_nr-1; if(dft->timetype==DFT_TIME_STARTEND) dft->voiNr--;

  /* Remove those DFT VOIs which where above set to be deleted (where sw!=0) */
  for(ci=dft->voiNr-1, ret=0; ci>=0; ci--) if(dft->voi[ci].sw!=0) {
    ret=dftDelete(dft, ci); if(ret!=0) break;}
  if(ret!=0) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
  if(dft->voiNr<1) {dftEmpty(dft); return CSV_INVALIDFORMAT;}

  /* Calculate frame mid times, or frame start and end times;
     this works only if time units are known */
  dftFrametimes(dft);

  return CSV_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Reads Inveon type 2 data into DFT struct
\return Returns 0 when successful, otherwise an error code.
 */
int csv2dft_b(
  /** Pointer to CSV data to be converted */
  CSV *csv,
  /** Pointer to empty DFT struct which will be allocated and filled here */
  DFT *dft
) {
  int ri, fi, fip, ii, ret;
  char *cptr, *cptr2, tmp[256];
  double v1, v2;

  if(CSV_TEST>2) {printf("csv2dft_b()\n"); fflush(stdout);}
  if(csv==NULL || dft==NULL) return CSV_ERROR;
  //printf("row_nr=%d col_nr=%d\n", csv->row_nr, csv->col_nr);
  if(csv->row_nr<4 || csv->col_nr!=9) return CSV_INVALIDFORMAT;
  dftEmpty(dft);

  /* Check the format; first line (containing titles) */
  if(strcasecmp(csv->c[0].content, "#Subject ID")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[1].content, "Subject Weight")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[2].content, "Subject Sex")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[3].content, "Unique Series ID")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[4].content, "Series Date")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[5].content, "Series Description")!=0) return CSV_INVALIDFORMAT;

  /* Check the format; third line (containing titles) */
  if(strcasecmp(csv->c[12].content, "#Name")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[13].content, "Volume (mm^3)")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[14].content, "Mean")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[15].content, "SD")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[16].content, "Min")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[17].content, "Max")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[18].content, "Frame Index")!=0) return CSV_INVALIDFORMAT;
  if(strncasecmp(csv->c[19].content, "Mid time (sec)", 10)!=0) return CSV_INVALIDFORMAT;
  if(strncasecmp(csv->c[20].content, "Duration (sec)", 10)!=0) return CSV_INVALIDFORMAT;

  /* Calculate the number of ROIs and time frames */
  ri=1; fi=0; fip=-1; ii=21; cptr=csv->c[ii].content;
  //printf("cell[%d] := '%s'\n", ii, cptr); fflush(stdout);
  for(; ii<csv->nr; ii+=9) {
    cptr2=csv->c[ii].content;
    //printf("cell[%d] := '%s'\n", ii, cptr2); fflush(stdout);
    if(strcmp(cptr, cptr2)==0) fi++;
    else {
      ri++; cptr=cptr2;
      if(fip<0) fip=fi; else if(fi!=fip) return CSV_INVALIDFORMAT;
      fi=1;
    }
    //printf("ri=%d fi=%d fip=%d\n", ri, fi, fip);
  }
  //printf("ri=%d fi=%d\n", ri, fi);

  /* Allocate memory for DFT */
  if(CSV_TEST>2) {printf("frame_nr=%d voi_nr=%d\n", fi, ri); fflush(stdout);}
  ret=dftSetmem(dft, fi, ri);
  if(ret!=0) return CSV_OUTOFMEMORY;
  dft->voiNr=ri; dft->frameNr=fi;
  dft->timetype=DFT_TIME_STARTEND;
  dft->_type=DFT_FORMAT_STANDARD;
  dft->isweight=0;
  dftUnitToDFT(dft, CUNIT_UNKNOWN); dft->timeunit=TUNIT_UNKNOWN;
  for(fi=0; fi<dft->frameNr; fi++) dft->w[fi]=1.0;

  /* Fill DFT */
  /* time unit */
  ii=19;
  if(strstr(csv->c[ii].content, "min")!=NULL) dft->timeunit=TUNIT_MIN;
  else if(strstr(csv->c[ii].content, "sec")!=NULL) dft->timeunit=TUNIT_SEC;
  else dft->timeunit=TUNIT_UNKNOWN;
  /* study number */
  ii=6;
  cptr=csv->c[ii].content; strncpy(dft->studynr, cptr, MAX_STUDYNR_LEN);
  dft->studynr[MAX_STUDYNR_LEN]=(char)0;
  cptr=strchr(dft->studynr, '.'); if(cptr!=NULL) *cptr=(char)0;
  cptr=strchr(dft->studynr, ','); if(cptr!=NULL) *cptr=(char)0;
  cptr=strchr(dft->studynr, ' '); if(cptr!=NULL) *cptr=(char)0;
  /* subject weight */
  ii=7; if(ii>=csv->nr) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
  v1=atof_dpi(csv->c[ii].content);
  if(v1>0.0) sprintf(dft->comments, "# weight := %g\n", v1);
  /* scan start time */
  ii=10; if(ii>=csv->nr) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
  if(strlen(csv->c[ii].content)>9) {
    sprintf(tmp, "# scan_start_time := %s\n", csv->c[ii].content);
    strcat(dft->comments, tmp);
  }
  /* frame times */
  for(fi=0; fi<dft->frameNr; fi++) {
    ii= 21 + fi*9 + 7; if(ii>csv->nr-2) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
    //printf("cell[%d] := '%s'\n", ii, csv->c[ii].content); fflush(stdout);
    v1=atof_dpi(csv->c[ii].content); v2=atof_dpi(csv->c[ii+1].content);
    dft->x[fi]=v1; dft->x1[fi]=v1-0.5*v2; dft->x2[fi]=v1+0.5*v2;
  }
  /* region names, volumes, and concentrations */
  for(ri=0; ri<dft->voiNr; ri++) {
    /* ROI name */
    ii= 21 + ri*dft->frameNr*9;
    if(ii>=csv->nr) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
    //printf("ri=%d cell[%d] := '%s'\n", ri, ii, csv->c[ii].content); fflush(stdout);
    strncpy(dft->voi[ri].name, csv->c[ii].content, MAX_REGIONNAME_LEN);
    dft->voi[ri].name[MAX_REGIONNAME_LEN]=(char)0;
    rnameSplit(csv->c[ii].content, dft->voi[ri].voiname,
  	       dft->voi[ri].hemisphere, dft->voi[ri].place,
               MAX_REGIONSUBNAME_LEN);
    /* Volume */
    ii++; if(ii>=csv->nr) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
    dft->voi[ri].size=atof_dpi(csv->c[ii].content);
    /* Frame concentrations */
    ii++; for(fi=0; fi<dft->frameNr; fi++) {
      if((ii+6)>=csv->nr) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
      /* Get concentration */
      dft->voi[ri].y[fi]=atof_dpi(csv->c[ii].content);
      /* Get concentration SD (probably not needed) */
      dft->voi[ri].y2[fi]=atof_dpi(csv->c[ii+1].content);
      /* check that frame times are correct */
      v1=atof_dpi(csv->c[ii+5].content);
      if(dft->x[fi]!=v1) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
      ii+=9;
    } // next frame
  } // next region

  return CSV_OK;
}
#endif
/*****************************************************************************/

/*****************************************************************************/
