/******************************************************************************

  Copyright (c) 2013 Turku PET Centre

  Library:      tacdecay
  Description:  Functions for working with physical decay and isotopes in TAC.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  2013-09-18 Vesa Oikonen
       First created, based on previous code.

    
******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
#include <ctype.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/tacio.h"
/*****************************************************************************/

/*****************************************************************************/
/** Corrects TAC data for physical decay, or removes the correction.
\return Returns TACIO status code.
 */   
int tacDecayCorrection(
  /** Pointer to existing TAC data; status of decay correction in TAC
   *  is not verified, but set in this function;
   *  TAC must contain valid sample time unit. */
  TAC *d,
  /** Half-life of isotope in minutes; enter <=0, if correct isotope code
   *  is given in TAC */
  double hl,
  /** 0=Remove decay correction; 1=Correct for decay */
  int mode,
  /** Apply (1) or do not apply (0) correction to y[] data in TAC */
  int y,
  /** Apply (1) or do not apply (0) correction to y2[] data in TAC */
  int y2,
  /** Apply (1) or do not apply (0) correction to y3[] data in TAC */
  int y3,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout */
  int verbose
) {
  int ri, fi, isotopeID=-1;
  double lambda, dc;

  if(verbose>0) printf("tacDecayCorrection(tac, %g, %d, %d, %d, %d, ...)\n",
    hl, mode, y, y2, y3);

  /* Check the input */
  if(d==NULL) return(TACIO_FAULT);
  if(d->voiNr<1 || d->frameNr<1) return(TACIO_NOTAC);
  /* Check that DFT contains time unit */
  if(d->tunit!=TUNIT_SEC && d->tunit!=TUNIT_MIN) {
    if(verbose>0) printf("timeunit := %s\n", petTunit(d->tunit));
    return(TACIO_NOTUNIT);
  }
  
  /* Check and set isotope code and half-life */
  if(hl>1.0E-10) { 
    /* Try to identify the isotope from given half-life */
    isotopeID=hlIsotopeFromHalflife(hl);
    if(isotopeID>=0) {
      /* If identified, then set the code in TAC */
      d->isocode=isotopeID;
      if(verbose>1) printf("  isotope := %s\n", hlIsotopeCode(isotopeID));
    } else {
      /* If not identified, that may just be because isotope is not yet listed
         in TPC library, but give a warning */
      if(verbose>0)
        fprintf(stderr, "Warning: halflife %g min is not identified.\n", hl);
    }
  } else {
    /* Check that valid isotope code is found in TAC */
    hl=hlFromIsotope(hlIsotopeCode(d->isocode));
    if(hl<=0.0) return(TACIO_INVALIDISOTOPE);
    if(verbose>1) printf("  half-life := %g min\n", hl);
  }
  
  /*
   *  Calculate the lambda
   */
  if(d->tunit==TUNIT_SEC) hl*=60.0;
  lambda=hl2lambda(hl); if(mode==0) lambda=-lambda;
  if(verbose>1) printf("  lambda := %e\n", lambda);

  /*
   *  Decay correction / removal
   */
  if(verbose>2) {
    if(mode!=0) printf("decay correction\n");
    else printf("removing decay correction\n");
  }
  for(fi=0; fi<d->frameNr; fi++) {
    /* Calculate decay factor */
    if(d->isframe) {
      if(isnan(d->x1[fi]) || isnan(d->x2[fi])) {dc=1.0; continue;}
      dc=hlLambda2factor(lambda, d->x1[fi], d->x2[fi]-d->x1[fi]);
    } else {
      if(isnan(d->x[fi])) {dc=1.0; continue;}
      dc=hlLambda2factor(lambda, d->x[fi], 0.0);
    }
    if(verbose>4) printf("  %10.4f  ->  %e\n", d->x[fi], dc);
    /* Correct all regions */
    for(ri=0; ri<d->voiNr; ri++) {
      if(y!=0  && !isnan(d->voi[ri].y[fi]))  d->voi[ri].y[fi]*=dc;
      if(y2!=0 && !isnan(d->voi[ri].y2[fi])) d->voi[ri].y2[fi]*=dc;
      if(y3!=0 && !isnan(d->voi[ri].y3[fi])) d->voi[ri].y3[fi]*=dc;
    }
  }
  if(mode!=0) d->decayCorrected=DECAY_CORRECTED;
  else d->decayCorrected=DECAY_NOTCORRECTED;
  return(TACIO_OK);
} 
/*****************************************************************************/

/*****************************************************************************/
