/******************************************************************************
  Copyright (c) 2013 by Turku PET Centre

  File:        tacdft.c
  Description: Functions for TAC files in DFT format.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  2013-09-10 Vesa Oikonen
       First created.
     


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/tacio.h"
/*****************************************************************************/

/*****************************************************************************/
/** Write TAC data into specified file pointer in DFT format.
    Number of decimals can be determined by changing global variable
    TAC_NR_OF_DECIMALS.
\return Returns TACIO status.
 */
int tacWriteDFT(
  /** Pointer to TAC struct, contents of which are to be written */
  TAC *tac,
  /** File pointer */
  FILE *fp,
  /** Write (1) or do not write (0) also extra header fields found in IFT */
  int extra,
  /** Verbose level; if zero, nothing extra is printed into stdout */
  int verbose
) {
  int fi, ri, n, prec;
  double v;
  char *cptr, tmp[1024];

  if(verbose>0) printf("tacWriteDFT(tac, fp, %d, ...)\n", extra);

  /* Check that there is some data to write */
  if(tac==NULL) return TACIO_FAULT;
  if(tac->voiNr<1 || tac->frameNr<1) return TACIO_NOTABLE;

  prec=0; if(TAC_NR_OF_DECIMALS>prec) prec=TAC_NR_OF_DECIMALS;

  /* Make sure that TAC names are available */
  if(verbose>1) printf("constructing TAC names\n");
  tacEnsureNames(tac);
  /* Construct TAC subnames */
  for(ri=0; ri<tac->voiNr; ri++)
    if(tacNameSplit(&tac->voi[ri].id)<1) return TACIO_FAULT;

  /* Write obligatory header lines */
  if(verbose>1) printf("writing obligatory title lines\n");
  
  /* 1st line with filetype identification string and region names */
  n=fprintf(fp, "DFT1"); if(n<1) return TACIO_CANNOTWRITE;
  for(ri=0; ri<tac->voiNr; ri++)
    fprintf(fp, "\t%s", tac->voi[ri].id.sub[0]);
  if(tac->weighting==WEIGHTING_ON) fprintf(fp, "\t%s", "weight");
  n=fprintf(fp, "\n"); if(n<1) return TACIO_CANNOTWRITE;
  
  /* 2nd line with study identification and 2nd subname */
  cptr=tac->studynr; if(strlen(cptr)) strcpy(tmp, cptr); else strcpy(tmp, ".");
  n=fprintf(fp, "%s", tmp); if(n<1) return TACIO_CANNOTWRITE;
  for(ri=0; ri<tac->voiNr; ri++) {
    cptr=tac->voi[ri].id.sub[1];
    if(strlen(cptr)) strcpy(tmp, cptr); else strcpy(tmp, ".");
    n=fprintf(fp, "\t%s", tmp); if(n<1) return TACIO_CANNOTWRITE;
  }
  if(tac->weighting==WEIGHTING_ON) fprintf(fp,"\t%s", ".");
  n=fprintf(fp, "\n"); if(n<1) return TACIO_CANNOTWRITE;
  
  /* 3rd line with calibration unit and 3rd subname */
  if(tac->cunit==CUNIT_UNKNOWN) strcpy(tmp, ".");
  else strcpy(tmp, petCunit(tac->cunit));
  n=fprintf(fp, "%s", tmp); if(n<1) return TACIO_CANNOTWRITE;
  for(ri=0; ri<tac->voiNr; ri++) {
    cptr=tac->voi[ri].id.sub[2];
    if(strlen(cptr)) strcpy(tmp, cptr); else strcpy(tmp, ".");
    n=fprintf(fp, "\t%s", tmp); if(n<1) return TACIO_CANNOTWRITE;
  }
  if(tac->weighting==WEIGHTING_ON) fprintf(fp,"\t%s", ".");
  n=fprintf(fp, "\n"); if(n<1) return TACIO_CANNOTWRITE;

  /* 4th line with time type & unit and region volumes */
  switch(tac->tunit) {
    case TUNIT_UM:
    case TUNIT_MM:
    case TUNIT_CM:
      if(tac->isframe==0) strcpy(tmp, "Distance");
      else strcpy(tmp, "Distances");
      break;
    default:
      if(tac->isframe==0) strcpy(tmp, "Time");
      else strcpy(tmp, "Times");
    break;
  }
  n=fprintf(fp, "%s (%s)", tmp, petTunit(tac->tunit));
  if(n<1) return TACIO_CANNOTWRITE;
  for(ri=0; ri<tac->voiNr; ri++) {
    if(tac->voi[ri].size>=0.0) sprintf(tmp, "%.*e", prec, tac->voi[ri].size);
    else strcpy(tmp, ".");
    n=fprintf(fp, "\t%s", tmp); if(n<1) return TACIO_CANNOTWRITE;
  }
  if(tac->weighting==WEIGHTING_ON) fprintf(fp,"\t%s", ".");
  n=fprintf(fp, "\n"); if(n<1) return TACIO_CANNOTWRITE;

  /* Write data */
  if(verbose>1) printf("writing data table\n");
  for(fi=0; fi<tac->frameNr; fi++) {
    /* Time(s) (x, x1, x2) */
    if(tac->isframe==0) v=tac->x[fi]; else v=tac->x1[fi];
    if(isnan(v)) n=fprintf(fp, "."); else n=fprintf(fp, "%.5f", v);
    if(n<1) return TACIO_CANNOTWRITE;
    if(tac->isframe!=0) {
      v=tac->x2[fi];
      if(isnan(v)) n=fprintf(fp, "\t."); else n=fprintf(fp, "\t%.5f", v);
      if(n<1) return TACIO_CANNOTWRITE;
    }
    /* Concentrations (y values) */
    for(ri=0; ri<tac->voiNr; ri++) {
      if(isnan(tac->voi[ri].y[fi])) n=fprintf(fp, "\t.");
      else fprintf(fp, "\t%.*e", prec, tac->voi[ri].y[fi]);
      if(n<1) return TACIO_CANNOTWRITE;
    }
    /* Weight */
    if(tac->weighting==WEIGHTING_ON) {
      if(isnan(tac->w[fi])) n=fprintf(fp, "\t.");
      else n=fprintf(fp, "\t%.*e", prec, tac->w[fi]);
      if(n<1) return TACIO_CANNOTWRITE;
    }
    n=fprintf(fp, "\n"); if(n<1) return TACIO_CANNOTWRITE;
  }
  
  return(TACIO_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read DFT TAC file contents into specified TAC data structure.
\return Returns TACIO status.
 */
int tacReadDFT(
  /** Pointer to initiated TAC struct where TAC data will be written;
   *  any old content is deleted. */
  TAC *d,
  /** Name of file to be read */
  char *filename,
  /** Verbose level; if zero, nothing is printed into stdout */
  int verbose
) {
  int ret, voiNr=0, frameNr=0;

  if(verbose>0) printf("tacReadDFT(tac, '%s', ...)\n", filename);
  if(strlen(filename)<1 || d==NULL) return TACIO_FAULT;

  /* Delete any previous data */
  tacEmpty(d);

  /* Read file into IFT struct */
  if(verbose>1) printf("reading file\n");
  ret=iftRead(&d->ift, filename, 0);
  if(ret!=IFT_OK) {
    if(verbose>0) printf("Error in reading: %s\n", d->ift.status);
    iftEmpty(&d->ift); if(ret==IFT_CANNOTREAD) return TACIO_CANNOTOPEN;
    return TACIO_INVALIDFORMAT;
  }
  if(verbose>1) printf("%d line(s).\n", d->ift.keyNr);

  /* Get data dimensions */
  if(verbose>1) printf("reading dimensions of the data\n");
  ret=iftGetDataDimensions(&d->ift, &frameNr, &voiNr);
  if(ret!=IFT_OK) {
    if(verbose>0) printf("error %d in reading dimensions of the data\n", ret);
    tacEmpty(d); return TACIO_INVALIDFORMAT;
  }
  if(verbose>2) {
    printf("data_lines := %d\n", frameNr);
    printf("max_column_nr := %d\n", voiNr);
  }

  /* Allocate memory for TAC */
  if(verbose>1) printf("allocating memory for TACs\n");
  voiNr--; // first column should be time
  ret=tacSetmem(d, frameNr, voiNr);
  if(ret!=TACIO_OK) {tacEmpty(d); return ret;}
  
  /* Move TAC data from IFT */
  if(verbose>1) printf("moving data table from IFT\n");
  ret=tacProcessDFTData(d, verbose-10);
  if(ret!=TACIO_OK) {tacEmpty(d); return ret;}

  /* Set study number from filename, if necessary */
  if(strlen(d->studynr)==0) studynr_from_fname(filename, d->studynr);
  
  /* Set file format */
  d->format=TAC_FORMAT_DFT;

  //return TACIO_INVALIDFORMAT;
  return TACIO_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Move DFT TAC data table from IFT inside TAC struct file.
\return Returns TACIO status.
 */
int tacProcessDFTData(
  /** Pointer to TAC struct where TAC table is in IFT;
   *  corresponding parts of IFT are deleted. */
  TAC *d,
  /** Verbose level; if zero, nothing is printed into stdout */
  int verbose
) {
  int ret, i, ri, fi, ti;
  char *cptr, *lptr;
  double v;
  
  if(verbose>0) printf("tacProcessDFTData()\n");
  if(d==NULL) return TACIO_FAULT;
  if(verbose>3) {
    printf("_framedataNr := %d\n", d->_framedataNr);
    printf("_voidataNr := %d\n", d->_voidataNr);
    printf("keyNr := %d\n", d->ift.keyNr);
  }

  /* Go through the IFT 'lines' */
  i=0; ri=0; fi=0; ti=0;
  while(i<d->ift.keyNr && fi<d->_framedataNr) {
    if(verbose>2) printf("table line := '%s'\n", d->ift.item[i].value);
    if(d->ift.item[i].type=='#') {i++; continue;}
    if(d->ift.item[i].type==';') {i++; continue;}
    if(strlen(d->ift.item[i].value)<1) {i++; continue;}
    if(strlen(d->ift.item[i].key)>0) {i++; continue;}
    /* First title line */
    if(ti==0) {
      /* The first token in the line is the magic number */
      lptr=d->ift.item[i].value; cptr=strtok(lptr, " \t");
      if(strncasecmp(cptr, "DFT1", 3)!=0) return TACIO_INVALIDFORMAT;
      /* Next tokens are TAC names */
      ri=0;
      while((cptr=strtok(NULL, " \t"))!=NULL && ri<d->_voidataNr) {
        if(strcmp(cptr, ".")==0) strcpy(d->voi[ri].id.sub[0], "");
        else strncpy(d->voi[ri].id.sub[0], cptr, MAX_TACNAME_LEN);
        d->voi[ri++].id.sub[0][MAX_TACNAME_LEN]=(char)0;
      }
      /* If we did not get anything, then that is bad */
      if(ri==0) return TACIO_BADTABLE;
      /* Check that we get the same nr of y values on each line */
      d->voiNr=ri; // first sample line
      if(verbose>2) printf("voiNr based on title line := %d\n", d->voiNr);
      /* Remove this line from IFT */
      ret=iftDeleteItem(&d->ift, i);
      ti++; continue; 
    }
    /* 2nd title line */
    if(ti==1) {
      /* The first token in the line is the study number */
      lptr=d->ift.item[i].value; cptr=strtok(lptr, " \t");
      if(strcmp(cptr, ".")==0) strcpy(d->studynr, "");
      else strncpy(d->studynr, cptr, MAX_STUDYNR_LEN);
      d->studynr[MAX_STUDYNR_LEN]=(char)0;
      /* Next tokens are TAC name subparts */
      ri=0;
      while((cptr=strtok(NULL, " \t"))!=NULL && ri<d->_voidataNr) {
        if(strcmp(cptr, ".")==0) strcpy(d->voi[ri].id.sub[1], "");
        else strncpy(d->voi[ri].id.sub[1], cptr, MAX_TACNAME_LEN);
        d->voi[ri++].id.sub[1][MAX_TACNAME_LEN]=(char)0;
      }
      /* If we did not get anything, then that is bad */
      if(ri==0) return TACIO_BADTABLE;
      /* Check that we get the same nr of y values on each line */
      if(ri!=d->voiNr) return TACIO_BADTABLE;
      /* Remove this line from IFT */
      ret=iftDeleteItem(&d->ift, i);
      ti++; continue; 
    }
    /* 3rd title line */
    if(ti==2) {
      /* The first token in the line is the calibration unit */
      lptr=d->ift.item[i].value; cptr=strtok(lptr, " \t");
      d->cunit=petCunitId(cptr);
      /* Next tokens are TAC name subparts */
      ri=0;
      while((cptr=strtok(NULL, " \t"))!=NULL && ri<d->_voidataNr) {
        if(strcmp(cptr, ".")==0) strcpy(d->voi[ri].id.sub[2], "");
        else strncpy(d->voi[ri].id.sub[2], cptr, MAX_TACNAME_LEN);
        d->voi[ri++].id.sub[2][MAX_TACNAME_LEN]=(char)0;
      }
      /* If we did not get anything, then that is bad */
      if(ri==0) return TACIO_BADTABLE;
      /* Check that we get the same nr of y values on each line */
      if(ri!=d->voiNr) return TACIO_BADTABLE;
      /* All subnames are now collected, lets make the full names */
      for(ri=0; ri<d->voiNr; ri++)
        {ret=tacNameCatenate(&d->voi[ri].id); if(ret!=TACIO_OK) return ret;}
      /* Remove this line from IFT */
      ret=iftDeleteItem(&d->ift, i);
      ti++; continue; 
    }
    /* 4th title line */
    if(ti==3) {
      /* The first token in the line is the time (x) type */
      lptr=d->ift.item[i].value; cptr=strtok(lptr, " \t");
      if(verbose>3) printf("  1st token := '%s'\n", cptr);
      if(strcasecmp(cptr, "Time")==0) d->isframe=0;
      else if(strcasecmp(cptr, "Times")==0) d->isframe=1;
      else if(strcasecmp(cptr, "Distance")==0) d->isframe=0;
      else if(strcasecmp(cptr, "Distances")==0) d->isframe=1;
      else return TACIO_INVALIDFORMAT;
      /* Next token is the x unit in parenthesis */
      cptr=strtok(NULL, " \t()[]{}");
      if(verbose>3) printf("  next token := '%s'\n", cptr);
      if(cptr==NULL) return TACIO_INVALIDFORMAT;
      d->tunit=petTunitId(cptr);
      /* Next tokens are TAC region volumes */
      ri=0;
      while((cptr=strtok(NULL, " \t"))!=NULL && ri<d->_voidataNr) {
        if(verbose>3) printf("  next token := '%s'\n", cptr);
        ret=atof_with_check(cptr, &v);
        if(ret==0) d->voi[ri++].size=v; else d->voi[ri++].size=0.0;
      }
      /* If we did not get anything, then that is bad */
      if(ri==0) return TACIO_BADTABLE;
      /* Check that we get the same nr of y values on each line */
      if(ri!=d->voiNr) return TACIO_BADTABLE;
      /* Remove this line from IFT */
      ret=iftDeleteItem(&d->ift, i);
      ti++; continue; 
    }
    if(verbose>2 && ti==5 && fi==0)
      printf("reading DFT title lines finished; reading TAC data table\n");
    /* The first token in the line is the sample time (x or x1) */
    lptr=d->ift.item[i].value; cptr=strtok(lptr, " \t");
    if(verbose>3) printf("  1st token := '%s'\n", cptr);
    if(strcmp(cptr, ".")==0) v=nan("");
    else if(atof_with_check(cptr, &v)!=0) return TACIO_INVALIDFORMAT;
    if(d->isframe==0) {
      d->x[fi]=v;
    } else {
      d->x1[fi]=v;
      /* The 2nd token in the line is the frame end time (x2) */
      cptr=strtok(NULL, " \t");
      if(verbose>3) printf("  2nd token := '%s'\n", cptr);
      if(strcmp(cptr, ".")==0) v=nan("");
      else if(atof_with_check(cptr, &v)!=0) return TACIO_INVALIDFORMAT;
      d->x2[fi]=v; d->x[fi]=0.5*(d->x1[fi]+d->x2[fi]);
    }
    /* Next tokens are concentration values (y) */
    ri=0;
    while((cptr=strtok(NULL, " \t"))!=NULL && ri<d->_voidataNr) {
      if(verbose>3) printf("  next token := '%s'\n", cptr);
      if(strcmp(cptr, ".")==0) {ret=0; v=nan("");}
      else ret=atof_with_check(cptr, &v);
      if(ret==0) d->voi[ri++].y[fi]=v; else return TACIO_INVALIDFORMAT;
    }
    /* If we did not get any y values, then that is bad */
    if(ri==0) return TACIO_BADTABLE;
    /* Check that we get the same nr of y values on each line */
    if(ri!=d->voiNr) return TACIO_BADTABLE;
    fi++;
    /* Remove this line from IFT */
    ret=iftDeleteItem(&d->ift, i);
  }
  d->frameNr=fi;
  /* Check the data table size */
  if(d->frameNr<1 || d->voiNr<1) return TACIO_NOTABLE;

  return TACIO_OK;
}
/*****************************************************************************/

/*****************************************************************************/
