/******************************************************************************
  Copyright (c) 2013 by Turku PET Centre

  File:        tacid.c
  Description: Functions for processing TAC names in TAC struct.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  2013-09-26 Vesa Oikonen
       First created.
     


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/tacio.h"
/*****************************************************************************/

/*****************************************************************************/
/** Copy TACID struct contents from tacid1 to tacid2.
\return Returns TACIO status.
 */
int tacCopyTacid(
  /** Pointer to TACID struct */
  TACID *d1,
  /** Pointer to TACID struct */
  TACID *d2
) {
  int i;

  /* Check that required data exists */
  if(d1==NULL || d2==NULL) return TACIO_FAULT;
  /* Copy */
  strcpy(d2->name, d1->name);
  for(i=0; i<MAX_SUBTACID_NR; i++) strcpy(d2->sub[i], d1->sub[i]);
  d2->side=d1->side;
  d2->plane=d1->plane;
  d2->input=d1->input;
  d2->sw=d1->sw; d2->sw2=d1->sw2; d2->sw3=d1->sw3;
  return TACIO_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Split TAC ID name into subparts.
\return Returns the number of last filled subpart.
 */
int tacNameSplit(
  /** Pointer to TACID struct */
  TACID *d
) {
  char temp[MAX_TACNAME_LEN+1], temp2[MAX_TACNAME_LEN+1];
  char *cptr, space[64];
  int i, nr, m;

  if(d==NULL) return(0);
  for(i=0; i<MAX_SUBTACID_NR; i++) strcpy(d->sub[i], "");
  strncpy(temp, d->name, MAX_TACNAME_LEN); temp[MAX_TACNAME_LEN]=(char)0;
  m=strlen(temp); for(i=0; i<m; i++)
    if(temp[i]=='\n' || temp[i]=='\r') {temp[i]=(char)0; break;}
  strcpy(space, " \t");
  nr=rnameRmDots(temp, temp2);
#if(0)  
  printf("temp = '%s'\n", temp);  
  printf("temp2 = '%s'\n", temp2);  
#endif
  if(nr<MAX_SUBTACID_NR) {
    if(strChrCount(temp2, space)<MAX_SUBTACID_NR-1) strcat(space, "_");
    if(strChrCount(temp2, space)<MAX_SUBTACID_NR-1) strcat(space, "-");
  }

/*
  for(nr=0; nr<MAX_SUBTACID_NR; nr++) {
    if(nr==0) cptr=strtok(temp, space); else cptr=strtok(NULL, space); 
    if(cptr==NULL) return(nr);
    if(strcmp(cptr, ".")==0) continue;
    strncpy(d->sub[nr], cptr, MAX_TACNAME_LEN);
    d->sub[nr][MAX_TACNAME_LEN]=(char)0;
  }
*/
  cptr=temp;
  for(nr=0; nr<MAX_SUBTACID_NR && strlen(cptr)>0; nr++) {
    //printf("cptr='%s'\n", cptr);
    /* If we have space character next, we assume that subname is empty */
    i=strspn(cptr, space); if(i>0) {cptr++; continue;}
    /* Non-space characters for a subname */
    i=m=strcspn(cptr, space); if(i==0) break;
    if(m>MAX_TACNAME_LEN) m=MAX_TACNAME_LEN; strncpy(d->sub[nr], cptr, m);
    d->sub[nr][m]=(char)0;
    cptr+=(i+1); // jump over token and the next space character
    /* If name was '.' then replace it with empty string */
    if(strcmp(d->sub[nr], ".")==0) strcpy(d->sub[nr], "");
  }
  return(nr);
}
/*****************************************************************************/

/*****************************************************************************/
/** Construct full TAC name from the subnames.
\return Returns TACIO_OK when successful.
 */
int tacNameCatenate(
  /** Pointer TACID struct */  
  TACID *id
) {
  int si, n, len, rlen;

  if(id==NULL) return TACIO_FAULT;
  strcpy(id->name, "");
  for(si=n=0; si<MAX_SUBTACID_NR; si++) {
    // no use to add empty subname
    len=strlen(id->sub[si]); if(len==0) continue;
    if(len==1 && id->sub[si][0]=='.') continue;
    // do not overflow target string
    rlen=strlen(id->name); if((len+rlen+1)>=MAX_TACNAME_LEN) continue;
    // if this is not the first addition then add '_'
    if(n>0) strcat(id->name, "_");
    // add subname
    strcat(id->name, id->sub[si]);
    n++;
  }
  if(strlen(id->name)<1) return TACIO_FAULT;
  return TACIO_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Test whether region name or number matches with a test string.
 *  Test string can contain wildcards. 
 *  If both names contain separators, the corresponding subnames are tested
 *  against each other.
 *  If test str does not contain separators (' ', '_', or '-') but TAC name
 *  does, then names are considered matching if test_str matches with any of
 *  subnames.
 *
\return Returns 1, in case of match, or 0 if not matched.
 */
int tacNameMatch(
  /** Pointer to TACID struct containing full name to be tested */
  TACID *d,
  /** Test string; string of max length MAX_PARNAME_LEN */
  char *test_str,
  /** Verbose level; if zero, then nothing is printed into stdout or stderr */
  int verbose
) {
  TACID test;
  int   ni, rtoknr=0, ttoknr=0;

  /* Check the input */
  if(d==NULL || test_str==NULL) return(0);
  /* If both are empty, that is a match */
  if(strlen(d->name)<1 && strlen(test_str)<1) return(1);
  /* Split full names into subnames */
  rtoknr=tacNameSplit(d);
  strncpy(test.name, test_str, MAX_PARNAME_LEN+1);
  test.name[MAX_PARNAME_LEN]=(char)0;
  ttoknr=tacNameSplit(&test);
  if(verbose>0) {
    printf("rtoknr := %d\nsubnames :=", rtoknr);
    for(ni=0; ni<rtoknr; ni++) printf(" '%s'", d->sub[ni]); printf("\n");
    printf("ttoknr := %d\nsubnames :=", ttoknr);
    for(ni=0; ni<ttoknr; ni++) printf(" '%s'", test.sub[ni]); printf("\n");
  }
  if(rtoknr==0 && ttoknr==0) return(1);
  if(rtoknr==0 || ttoknr==0) return(0);
  /* If more than one subname to test for, then test against corresponding
     subnames with wildcards */
  if(ttoknr>1) {
    if(rtoknr!=ttoknr) return(0);
    for(ni=0; ni<ttoknr; ni++) {
      if(strlen(test.sub[ni])==0 && strlen(d->sub[ni])==0) continue;
      if( strlen(test.sub[ni])<1 || fncasematch(test.sub[ni], d->sub[ni])==0 ) {
        return(0);}
    }
    return(1);
  }
  /* We have just one subname in the test string */
  /* If only one subname also in TAC name, then test them with wildcards */
  if(rtoknr==1) {
    if(fncasematch(test.sub[0], d->sub[0])==0 ) return(0); else return(1);
  }
  /* If it is found in any of sub names, then we consider it a match */
  for(ni=0; ni<rtoknr; ni++) if(fncasematch(test.sub[0], d->sub[ni])) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
