/******************************************************************************
  Copyright (c) 2013 by Turku PET Centre

  File:        tacio.c
  Description: Functions for TAC file reading and writing.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  2013-09-11 Vesa Oikonen
       First created.
     


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/tacio.h"
/*****************************************************************************/

/*****************************************************************************/
/** Nr of decimals for sample concentration (y) values */
int TAC_NR_OF_DECIMALS = 4;
/*****************************************************************************/

/*****************************************************************************/
static const char *tacio_format[] = {
  "Unknown",    // TAC_FORMAT_UNKNOWN
  "Simple",     // TAC_FORMAT_SIMPLE
  "DFT",        // TAC_FORMAT_DFT
  "IFT",        // TAC_FORMAT_IFT
  "FIT",        // PAR_FORMAT_FIT
  "NCI",        // TAC_FORMAT_NCI
  "PMOD",       // TAC_FORMAT_PMOD
  "TAC CSV-INT",// TAC_FORMAT_CSV_INT
  "TAC CSV-UK", // TAC_FORMAT_CSV_UK
  "TAC TSV-INT",// TAC_FORMAT_TSV_INT
  "TAC TSV-UK", // TAC_FORMAT_TSV_UK
  "CPT",        // TAC_FORMAT_CPT
  "IDWC",       // TAC_FORMAT_IDWC
  "IF",         // TAC_FORMAT_IF
  "XML",        // TAC_FORMAT_XML
  "HTML table", // TAC_FORMAT_HTML
  "Xeleris",    // TAC_FORMAT_XELERIS
  "Amide",      // TAC_FORMAT_AMIDE
  "Binary",     // TAC_FORMAT_BINARY
  "RES",        // PAR_FORMAT_RES
  "PAR CSV-INT",// PAR_FORMAT_CSV_INT
  "PAR CSV-UK", // PAR_FORMAT_CSV_UK
  "PAR TSV-INT",// PAR_FORMAT_TSV_INT
  "PAR TSV-UK", // PAR_FORMAT_TSV_UK
  0
  };
/** Return pointer to string describing the TAC format */
char *tacFormattxt(
  int format_index
) {
  int n=0;
  while(tacio_format[n]!=0) n++;
  if(format_index<0 || format_index>n-1) format_index=TAC_FORMAT_UNKNOWN;
  return((char*)tacio_format[format_index]);
}
/*****************************************************************************/

/*****************************************************************************/
/** Determine the format of TAC and parameter file.
 *  Note that not all formats are identified, and identification does not mean
 *  that reading the format would be supported.
\return Returns the format number, or TAC_FORMAT_UNKNOWN if not identified or
    in case of an error.
 */
int tacFormatDetermine(
  /** Pointer to filename; this string is not modified. */
  const char *fname,
  /** Verbose level; if zero, nothing is printed into stdout */
  int verbose
) {
  FILE *fp;
  char tmp[256], *cptr;
  int c, format;

  if(verbose>0) printf("tacFormat('%s', ...)\n", fname);
  format=TAC_FORMAT_UNKNOWN;

  /* Open file; note that 'b' is required for fgetpos() and fsetpos() to work
     correctly */
  fp=fopen(fname, "rb");
  if(fp==NULL) {
    if(verbose>0) printf("cannot open file '%s'\n", fname);
    return format;
  }

  /* Binary data? */
  while((c=fgetc(fp))!=EOF) if(!isalnum(c) && !isspace(c) && !isgraph(c)) {
    format=TAC_FORMAT_BINARY;
    if(verbose>0) printf("identified as %s file\n", tacio_format[format]);
    fclose(fp); return format;
  }
  rewind(fp);

  /* Read the first line that is not empty or comment line */
  while(fgets(tmp, 32, fp) != NULL) {
    if(strlen(tmp)==0) continue;
    if(tmp[0]=='#') continue;
    if(strncmp(tmp, "//", 2)==0) continue;
    break;
  }
  rewind(fp);

  /* Check for identification strings */
  if(verbose>1) printf("checking for magic number\n");
  if(strncasecmp(tmp, "DFT", 3)==0) {
    format=TAC_FORMAT_DFT;
  } else if(strncasecmp(tmp, "FIT1", 3)==0) {
    format=PAR_FORMAT_FIT;
  } else if(strncasecmp(tmp, "cpt", 3)==0) {
    format=TAC_FORMAT_NCI;
  } else if(strncasecmp(tmp, "Parameters", 10)==0) {
    /* now determine the field and decimal separators */
    char fs, ds;
    csvSeparator(fp, &fs, &ds, verbose-6);
    if(fs==';') format=PAR_FORMAT_CSV_INT;
    else if(fs==',') format=PAR_FORMAT_CSV_UK;
    else if(ds==',') format=PAR_FORMAT_TSV_INT;
    else format=PAR_FORMAT_TSV_UK;
  }
  if(format!=TAC_FORMAT_UNKNOWN) {
    if(verbose>0) printf("identified as %s file\n", tacio_format[format]);
    fclose(fp); return format;
  }

  /* Identify certain filename extensions */
  if(verbose>1) printf("checking file name extension\n");
  cptr=strrchr(fname, '.'); if(cptr!=NULL) cptr++;
  if(verbose>1 && cptr!=NULL) printf("extension := %s\n", cptr);
  if(strcasecmp(cptr, "IDWC")==0 || strcasecmp(cptr, "IDW")==0) {
    format=TAC_FORMAT_IDWC;
  } else if(strcasecmp(cptr, "IF")==0) {
    format=TAC_FORMAT_IF;
  } else if(strcasecmp(cptr, "CSV")==0) {
    // Parameter CSV was identified abobe, so this should be TAC
    if(verbose>0) printf("identified as CSV TAC file\n");
#if(0)
    /* If ';' is found in file, then CSV is in international format,
       otherwise in UK format */
    while((c=fgetc(fp))!=EOF) if(c==';') break;
    if(c==';') format=TAC_FORMAT_CSV_INT;
    else format=TAC_FORMAT_CSV_UK;
#else
    char fs, ds;
    csvSeparator(fp, &fs, &ds, verbose-6);
    if(fs==';') format=TAC_FORMAT_CSV_INT;
    else if(fs==',') format=TAC_FORMAT_CSV_UK;
    else if(ds==',') format=TAC_FORMAT_TSV_INT;
    else format=TAC_FORMAT_TSV_UK;
#endif
  } else if(strcasecmp(cptr, "TSV")==0) {
    // Parameter TSV was identified abobe, so this should be TAC
    if(verbose>0) printf("identified as TSV TAC file\n");
#if(0)
    /* If ',' is found in file, then TSV is in international format,
       otherwise in UK format */
    while((c=fgetc(fp))!=EOF) if(c==',') break;
    if(c==',') format=TAC_FORMAT_TSV_INT;
    else format=TAC_FORMAT_TSV_UK;
#else
    char fs, ds;
    csvSeparator(fp, &fs, &ds, verbose-6);
    if(fs==';') format=TAC_FORMAT_CSV_INT;
    else if(fs==',') format=TAC_FORMAT_CSV_UK;
    else if(ds==',') format=TAC_FORMAT_TSV_INT;
    else format=TAC_FORMAT_TSV_UK;
#endif
  } else if(strcasecmp(cptr, "XML")==0) {
    format=TAC_FORMAT_XML;
  } 
  if(format!=TAC_FORMAT_UNKNOWN) {
    if(verbose>0) printf("identified as %s file\n", tacio_format[format]);
    fclose(fp); return format;
  }

  /* File with one title line without comment mark? */
  while(fgets(tmp, 255, fp) != NULL) {if(strlen(tmp)) break;}
  if(strncasecmp(tmp, "Time", 4)==0) {
    format=TAC_FORMAT_PMOD;
  } else if(strncasecmp(tmp, "Start", 5)==0) {
    format=TAC_FORMAT_PMOD;
  } else if((cptr=strstr(tmp, " (c)"))!=NULL
            || (cptr=strstr(tmp, " (C)"))!=NULL)
  { // this may be result (RES) file, if preceded by program name
    c=strlen(tmp)-1; while(c>0 && isspace(tmp[c])) c--; tmp[c+1]=(char)0;
    if(c>3) format=PAR_FORMAT_RES;
  }
  if(format!=TAC_FORMAT_UNKNOWN) {
    if(verbose>0) printf("identified as %s file\n", tacio_format[format]);
    fclose(fp); return format;
  }

  format=TAC_FORMAT_SIMPLE;
  if(verbose>0) printf("assumed to be in %s format\n", tacio_format[format]);
  fclose(fp);
  return format; 
}
/*****************************************************************************/

/*****************************************************************************/
/** Read TAC file contents into specified TAC data structure.
 *  This function reads simple data files (time + data columns), DFT format,
 *  PMOD and Hammersmith format (IDWC and IF) and some CSV formats.
\return Returns TACIO status.
 */
int tacRead(
  /** Pointer to initiated TAC struct where TAC data will be written;
   *  any old content is deleted. */
  TAC *d,
  /** Name of file to be read */
  char *filename,
  /** Verbose level; if zero, nothing is printed into stdout */
  int verbose
) {
  int ret, format;

  if(verbose>0) printf("tacRead(tac, '%s', ...)\n", filename);
  if(strlen(filename)<1 || d==NULL) return TACIO_FAULT;

  /* Delete any previous data */
  tacEmpty(d);

  /* Try to identify the file format */
  format=tacFormatDetermine(filename, verbose);
  if(format==TAC_FORMAT_UNKNOWN) {
    if(verbose>0) printf("unknown file format");
    return TACIO_INVALIDFORMAT;
  }
  
  /* Read the supported formats */
  if(format==TAC_FORMAT_SIMPLE) {
    ret=tacReadSimple(d, filename, verbose-3);
    return(ret);
  } else if(format==TAC_FORMAT_DFT) {
    ret=tacReadDFT(d, filename, verbose-3);
    return(ret);
  } else if(format==TAC_FORMAT_PMOD) {
    ret=tacReadPmod(d, filename, verbose-3);
    return(ret);
  }


  /* The rest of formats are not supported */
  if(verbose>0) fprintf(stderr, "Error: format %s not yet supported.\n",
                        tacio_format[format]);
  return TACIO_INVALIDFORMAT;
}
/*****************************************************************************/

/*****************************************************************************/
/** Determine data part dimensions in an IFT struct.
\return Returns IFT error code and sets ift->status text.
 */
int iftGetDataDimensions(
  /** Pointer to IFT struct */
  IFT *ift,
  /** Nr of lines, excluding empty lines; NULL if not needed. */
  int *lineNr,
  /** Nr of columns (max nr of tokens/line); space and tab characters
   *  are used as column separators; NULL if not needed. */
  int *colNr
) {
  int    i, c, lnr=0, toknr=0;
//  char  *p1, *p2;

  if(lineNr!=NULL) *lineNr=0;
  if(colNr!=NULL) *colNr=0;
  if(ift==NULL) {return IFT_FAULT;}

  /* Go through the IFT 'lines' */
  for(i=0; i<ift->keyNr; i++) {
    //printf("i=%d\n", i);
    if(strlen(ift->item[i].value)<1) continue;
    /* Item with key is not counted */
    if(strlen(ift->item[i].key)>0) continue;
    /* Comment line is not counted */
    if(ift->item[i].type=='#') continue;
    if(ift->item[i].type==';') continue;
    lnr++;
    /* Calculate the nr of tokens in the line */
    c=strTokenNr(ift->item[i].value, " \t");
#if(0)
    p1=ift->item[i].value; c=0;
    while(1) {   //printf("p1='%s'\n", p1);
      j=strcspn(p1, " \t"); if(j>0) c++;
      p2=p1+j; j=strspn(p2, " \t"); if(j==0) break; p1=p2+j;
    }
#endif
    if(c>toknr) toknr=c;   
  }
  if(lineNr!=NULL) *lineNr=lnr;
  if(colNr!=NULL) *colNr=toknr;
  iftSetStatus(ift, IFT_OK); return IFT_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Write TAC data into specified file in specified format.
    Number of decimals can be determined by changing global variable
    TAC_NR_OF_DECIMALS.
\return Returns TACIO status.
 */
int tacWrite(
  /** Pointer to TAC struct, contents of which are to be written */
  TAC *tac,
  /** Name of file to write TAC contents in. If file exists, original file
      is renamed to a backup file. If string 'stdout' is given, then
      contents are written in stdout. */
  char *filename,
  /** File_format code, as specified in tacio.h; enter TAC_FORMAT_UNKNOWN
   *  to write data in the format specified inside TAC struct. */  
  int format,
  /** Write (1) or do not write (0) also extra header fields found in IFT;
   *  only effective with DFT and Simple formats   */
  int extra,
  /** Verbose level; if zero, nothing extra is printed into stdout */
  int verbose
) {
  char is_stdout=0;
  int ret;
  FILE *fp;

  if(verbose>0) printf("tacWrite(tac, '%s', %s, %d, ...)\n",
    filename, tacFormattxt(format), extra);

  /* Check that there is some data to write */
  if(tac==NULL) return TACIO_FAULT;
  if(verbose>1) {
    printf("frameNr := %d\n", tac->frameNr);
    printf("voiNr := %d\n", tac->voiNr);
  }
  if(tac->voiNr<1 || tac->frameNr<1) return TACIO_NOTABLE;

  /* Check if writing to stdout */
  if(!strcasecmp(filename, "stdout")) is_stdout=1;

  /* Check if file exists; backup, if necessary */
  if(!is_stdout) (void)backupExistingFile(filename, NULL, NULL);

  /* If file format was not defined in function call, then get it from data */
  if(format==TAC_FORMAT_UNKNOWN) {
    format=tac->format;
    if(verbose>1) printf("file_format := %s\n", tacFormattxt(format));
  }
  /* Check that format is known before erasing file */
  if(format==TAC_FORMAT_UNKNOWN) return TACIO_INVALIDFORMAT;

  /* Open output file */
  if(is_stdout) fp=(FILE*)stdout;
  else {
    if(verbose>1) printf("opening file for write\n");
    if((fp=fopen(filename, "w")) == NULL) return TACIO_CANNOTWRITE;
  }

  /* Write file */
  switch(format) {
    case TAC_FORMAT_SIMPLE:
      ret=tacWriteSimple(tac, fp, extra, verbose);
      break;
    case TAC_FORMAT_DFT:
      ret=tacWriteDFT(tac, fp, extra, verbose);
      break;
    case TAC_FORMAT_PMOD:
      ret=tacWritePmod(tac, fp, verbose);
      break;
    case TAC_FORMAT_CSV_INT:
    case TAC_FORMAT_CSV_UK:
    case TAC_FORMAT_TSV_INT:
    case TAC_FORMAT_TSV_UK:
      ret=tacWriteCSV(tac, fp, verbose);
      break;
    default:
      ret=TACIO_INVALIDFORMAT;
  }

  /* Close file */
  fflush(fp); if(!is_stdout) fclose(fp);
  
  /* Quit */
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
#if(0)
  int i, j, n, sw, mfw, prec;
  char tmp[1024], tmp2[128], is_stdout=0;
  FILE *fp;


  /* Check that there is some data to write */
  if(data->voiNr<1 || data->frameNr<1) {
    strcpy(dfterrmsg, "no data"); return 1;}
  /* If format if DFT_FORMAT_HTML or extension is *.HTM(L),
     write in HTML format */
  if(data->_type==DFT_FORMAT_HTML || fncasematch("*.htm", filename)==1 ||
     fncasematch("*.html", filename)==1)
  {
    return(dftWriteHTML(data, filename, 1));
  }

  /* Check if writing to stdout */
  if(!strcasecmp(filename, "stdout")) is_stdout=1;

  /* Set minimum field width and precision for concentrations */
  mfw=11; if(DFT_NR_OF_DECIMALS>3) mfw+=DFT_NR_OF_DECIMALS-3;
  prec=0; if(DFT_NR_OF_DECIMALS>prec) prec=DFT_NR_OF_DECIMALS;

  /* Check if file exists; backup, if necessary */
  if(!is_stdout) (void)backupExistingFile(filename, NULL, NULL);

  /* Open output file */
  if(is_stdout) fp=(FILE*)stdout;
  else if((fp = fopen(filename, "w")) == NULL) {
    strcpy(dfterrmsg, "cannot open file"); return 2;}

  /* Write title lines */
  if(data->_type==DFT_FORMAT_STANDARD) {
    /* 1st line with filetype identification string and region names */
    n=fprintf(fp, "%s", DFT_VER);
    if(n==0) {strcpy(dfterrmsg, "disk full"); if(!is_stdout) fclose(fp); return 3;}
    for(i=0; i<data->voiNr; i++)
      fprintf(fp,"\t%s", data->voi[i].voiname);
    if(data->isweight) fprintf(fp,"\t%s", "weight");
    fprintf(fp, "\n");
    /* 2nd line with study identification and 2nd name (hemisphere) */
    if(strlen(data->studynr)) strcpy(tmp, data->studynr); else strcpy(tmp, ".");
    fprintf(fp, "%s", tmp);
    for(i=0; i<data->voiNr; i++) {
      if(strlen(data->voi[i].hemisphere)) strcpy(tmp, data->voi[i].hemisphere);
      else strcpy(tmp, ".");
      fprintf(fp, "\t%s", tmp);
    }
    if(data->isweight) fprintf(fp,"\t%s", ".");
    fprintf(fp, "\n");
    /* 3rd line with unit and plane names */
    if(strlen(data->unit)) strcpy(tmp, data->unit); else strcpy(tmp, ".");
    fprintf(fp, "%s", tmp);
    for(i=0; i<data->voiNr; i++) {
      if(strlen(data->voi[i].place)) strcpy(tmp, data->voi[i].place);
      else strcpy(tmp, ".");
      fprintf(fp, "\t%s", tmp);
    }
    if(data->isweight) fprintf(fp,"\t%s", ".");
    fprintf(fp, "\n");
    /* 4th line with time type & unit and region volumes */
    switch(data->timetype) {
      case DFT_TIME_MIDDLE:
        if(data->timeunit==TUNIT_MM || data->timeunit==TUNIT_UM ||
           data->timeunit==TUNIT_CM)
	  strcpy(tmp, "Distance ");
        else
	  strcpy(tmp, "Time ");
        break;
      case DFT_TIME_START: strcpy(tmp, "Start "); break;
      case DFT_TIME_END: strcpy(tmp, "End "); break;
      case DFT_TIME_STARTEND:
        if(data->timeunit==TUNIT_MM || data->timeunit==TUNIT_UM ||
           data->timeunit==TUNIT_CM)
	  strcpy(tmp, "Distances ");
        else
	  strcpy(tmp, "Times ");
        break;
    }
    snprintf(tmp2, 128, "(%s)", petTunit(data->timeunit)); strcat(tmp, tmp2);
    fprintf(fp, "%s", tmp);
    for(i=0; i<data->voiNr; i++) {
      if(data->voi[i].size>=0.0) sprintf(tmp, "%.*e", prec, data->voi[i].size);
      else strcpy(tmp, ".");
      fprintf(fp, "\t%s", tmp);
    }
    if(data->isweight) fprintf(fp,"\t%s", ".");
    fprintf(fp, "\n");
  } else if(data->_type==DFT_FORMAT_PMOD) { // write PMOD title line
    char *cptr, tunit[128], cunit[128]; // Set units to format accepted by PMOD
    if(data->timeunit==TUNIT_SEC) strcpy(tunit, "seconds");
    else if(data->timeunit==TUNIT_MIN) strcpy(tunit, "minutes");
    else strcpy(tunit, petTunit(data->timeunit));
    strcpy(cunit, data->unit); cptr=strstr(cunit, "mL");
    if(cptr!=NULL) {*cptr='c'; cptr++; *cptr='c';}
    if(data->timetype==DFT_TIME_STARTEND)
      fprintf(fp, "start[%s]\tend[%s]", tunit, cunit);
    else
      fprintf(fp, "time[%s]", tunit);
    for(i=0; i<data->voiNr; i++) {
      /* write roi names, must not include spaces */
      if(strchr(data->voi[i].name, ' ')==NULL) {
        fprintf(fp, "\t%s", data->voi[i].name);
      } else {
        fprintf(fp, "\t%s", data->voi[i].voiname);
        if(strlen(data->voi[i].hemisphere)>0)
          fprintf(fp, "-%s", data->voi[i].hemisphere);
        if(strlen(data->voi[i].place)>0)
          fprintf(fp, "-%s", data->voi[i].place);
      }
      /* write calibration unit when necessary */
      if(i==0 && data->timetype!=DFT_TIME_STARTEND)
        fprintf(fp, "[%s]", cunit);
    }
    fprintf(fp, "\n");
  }

  /* Make sure that only frame mid times are saved without titles */
  if(data->_type==DFT_FORMAT_PLAIN && data->timetype!=DFT_TIME_MIDDLE) sw=0;
  else sw=data->timetype;

  /* Write data */
  for(j=0; j<data->frameNr; j++) {
    /* Time(s) */
    switch(sw) {
      case 0: fprintf(fp, "%.5f", data->x[j]); break;
      case 1: fprintf(fp, "%.5f", data->x1[j]); break;
      case 2: fprintf(fp, "%.5f", data->x2[j]); break;
      case 3: fprintf(fp, "%.5f\t%.5f", data->x1[j], data->x2[j]); break;
    }
    /* y values */
    for(i=0; i<data->voiNr; i++) {
      if(!isnan(data->voi[i].y[j])) sprintf(tmp, "%.*e", prec, data->voi[i].y[j]);
      else strcpy(tmp, ".");
      fprintf(fp, "\t%s", tmp);
    }
    if(data->isweight && data->_type==DFT_FORMAT_STANDARD) {
      sprintf(tmp, "%.*e", prec, data->w[j]); fprintf(fp,"\t%s", tmp);}
    n=fprintf(fp, "\n");
    if(n==0) {strcpy(dfterrmsg, "disk full"); if(!is_stdout) fclose(fp); return 3;}
  }

  /* Write comments */
  if(data->_type!=DFT_FORMAT_PMOD) {
    n=strlen(data->comments);
    if(n) for(i=0; i<n; i++) {
      if(i>0 && data->comments[i]=='#' &&
         (data->comments[i-1]!='\n' && data->comments[i-1]!='\r'))
        fputc('\n', fp);
      fputc(data->comments[i], fp);
    }
  }

  /* Close file */
  if(!is_stdout) {fflush(fp); fclose(fp);}

  return 0;
}
#endif
/*****************************************************************************/

/*****************************************************************************/
