/******************************************************************************
  Copyright (c) 2013 by Turku PET Centre

  File:        tacnan.c
  Description: Functions for working with NaN's in TAC or PAR struct.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  2013-09-25 Vesa Oikonen
       First created.
     


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/tacio.h"
/*****************************************************************************/

/*****************************************************************************/
/** Check TAC struct for NaN's in sample times (x, or x1 and x2).
\return Returns the number of NaN's that were found.
 */
int tacFrameNaNs(
  /** Pointer to TAC struct */
  TAC *tac
) {
  if(tac==NULL || tac->frameNr<1) return(0);
  
  int fi, nan_nr=0;
  if(tac->isframe) {
    for(fi=0; fi<tac->frameNr; fi++) {
      if(isnan(tac->x1[fi])) nan_nr++;
      if(isnan(tac->x2[fi])) nan_nr++;
    }
  } else {
    for(fi=0; fi<tac->frameNr; fi++) {
      if(isnan(tac->x[fi])) nan_nr++;
    }
  }
  return(nan_nr);
}
/*****************************************************************************/
/** Check TAC struct for NaN's in sample values (y).
\return Returns the number of NaN's that were found.
 */
int tacValueNaNs(
  /** Pointer to TAC struct */
  TAC *tac
) {
  if(tac==NULL || tac->frameNr<1 || tac->voiNr<1) return(0);
  
  int ri, fi, nan_nr=0;
  for(fi=0; fi<tac->frameNr; fi++)
    for(ri=0; ri<tac->voiNr; ri++)
      if(isnan(tac->voi[ri].y[fi])) nan_nr++;
  return(nan_nr);
}
/*****************************************************************************/
/** Check TAC struct for NaN's in sample times and values.
\return Returns the number of NA's that were found.
 */
int tacNaNs(
  /** Pointer to TAC struct */
  TAC *tac
) {
  return(tacFrameNaNs(tac)+tacValueNaNs(tac));
}
/*****************************************************************************/

/*****************************************************************************/
#if(0)
/** Replace NA's in basic DFT data with interpolated values.
    If extrapolation is necessary, then the values (0,0) and
    (Infinity,last measured) are assumed.
\return Returns 0, if NA's could be filled with sensible values.
 */
int dftNAfill(
  /** Pointer to DFT struct */
  DFT *dft
) {
  int ri, fi, fj;
  double x1, x2, y1, y2, x, y;

  if(dft==NULL || dft->voiNr<1 || dft->frameNr<1) return(1);
  for(ri=0; ri<dft->voiNr; ri++) for(fi=0; fi<dft->frameNr; fi++) {
    if(isnan(dft->x[fi])) return(2);
    if(isnan(dft->voi[ri].y[fi])) {
      /* NA's before zero time are always replaced with 0 */
      if(dft->x[fi]<0.0) {dft->voi[ri].y[fi]=0.0; continue;}
      x=dft->x[fi];
      /* Get the previous data that is not NA */
      for(x1=y1=nan(""), fj=fi-1; fj>=0; fj--) if(!isnan(dft->voi[ri].y[fj])) {
        x1=dft->x[fj]; y1=dft->voi[ri].y[fj]; break;
      }
      if(isnan(x1) || isnan(y1)) x1=y1=0.0;
      /* Get the following data that is not NA */
      for(x2=y2=nan(""), fj=fi+1; fj<dft->frameNr; fj++) if(!isnan(dft->voi[ri].y[fj])) {
        x2=dft->x[fj]; y2=dft->voi[ri].y[fj]; break;
      }
      if(isnan(x2) || isnan(y2)) for(fj=fi-1; fj>=0; fj--) if(!isnan(dft->voi[ri].y[fj])) {
        x2=dft->x[fj]; y2=dft->voi[ri].y[fj]; break;
      }
      if(isnan(x2) || isnan(y2)) return(2);
      /* Calculate new value */
      if(x2==x1) y=0.5*(y1+y2); else y=y2-(x2-x)*(y2-y1)/(x2-x1);
      dft->voi[ri].y[fi]=y;
    }
  }
  return(0);
}
#endif
