/******************************************************************************
  Copyright (c) 2013 by Turku PET Centre

  File:        tacpmod.c
  Description: Functions for TAC files in PMOD format.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  2013-09-11 Vesa Oikonen
       First created.
     


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/tacio.h"
/*****************************************************************************/

/*****************************************************************************/
/** Write TAC data into specified file pointer in PMOD format.
    Number of decimals can be determined by changing global variable
    TAC_NR_OF_DECIMALS.
\return Returns TACIO status.
 */
int tacWritePmod(
  /** Pointer to TAC struct, contents of which are to be written */
  TAC *tac,
  /** File pointer */
  FILE *fp,
  /** Verbose level; if zero, nothing extra is printed into stdout */
  int verbose
) {
  int fi, ri, n, prec;
  double v;
  char *cptr, tunit[128], cunit[128];

  if(verbose>0) printf("tacWritePmod(tac, fp, ...)\n");

  /* Check that there is some data to write */
  if(tac==NULL) return TACIO_FAULT;
  if(tac->voiNr<1 || tac->frameNr<1) return TACIO_NOTABLE;

  prec=0; if(TAC_NR_OF_DECIMALS>prec) prec=TAC_NR_OF_DECIMALS;

  /* Make sure that TAC names are available */
  if(verbose>1) printf("constructing TAC names\n");
  tacEnsureNames(tac);
  
  /* Set units to format accepted by PMOD */
  if(tac->tunit==TUNIT_SEC) strcpy(tunit, "seconds");
  else if(tac->tunit==TUNIT_MIN) strcpy(tunit, "minutes");
  else strcpy(tunit, petTunit(tac->tunit));
  strcpy(cunit, petCunit(tac->cunit));
  cptr=strstr(cunit, "mL"); if(cptr==NULL) cptr=strstr(cunit, "ml");
  if(cptr!=NULL) {*cptr='c'; cptr++; *cptr='c';}

  /* Write obligatory header line */
  if(verbose>1) printf("writing obligatory title lines\n");
  if(tac->isframe==0) n=fprintf(fp, "time[%s]", tunit);
  else n=fprintf(fp, "start[%s]\tend[%s]", tunit, cunit);
  if(n<1) return TACIO_CANNOTWRITE;
  for(ri=0; ri<tac->voiNr; ri++) {
    /* write TAC names, must not include spaces */
    strReplaceChar(tac->voi[ri].id.name, ' ', '-');
    strReplaceChar(tac->voi[ri].id.name, '\t', '-');
    n=fprintf(fp, "\t%s", tac->voi[ri].id.name);
    if(n<1) return TACIO_CANNOTWRITE;
    /* write calibration unit when necessary */
    if(ri==0 && tac->isframe==0) fprintf(fp, "[%s]", cunit);
  }
  n=fprintf(fp, "\n"); if(n<1) return TACIO_CANNOTWRITE;
  
  /* Write data */
  if(verbose>1) printf("writing data table\n");
  for(fi=0; fi<tac->frameNr; fi++) {
    /* Time(s) (x, x1, x2) */
    if(tac->isframe==0) v=tac->x[fi]; else v=tac->x1[fi];
    if(isnan(v)) n=fprintf(fp, "."); else n=fprintf(fp, "%.5f", v);
    if(n<1) return TACIO_CANNOTWRITE;
    if(tac->isframe!=0) {
      v=tac->x2[fi];
      if(isnan(v)) n=fprintf(fp, "\t."); else n=fprintf(fp, "\t%.5f", v);
      if(n<1) return TACIO_CANNOTWRITE;
    }
    /* Concentrations (y values) */
    for(ri=0; ri<tac->voiNr; ri++) {
      if(isnan(tac->voi[ri].y[fi])) n=fprintf(fp, "\t.");
      else fprintf(fp, "\t%.*e", prec, tac->voi[ri].y[fi]);
      if(n<1) return TACIO_CANNOTWRITE;
    }
    n=fprintf(fp, "\n"); if(n<1) return TACIO_CANNOTWRITE;
  }
  
  return(TACIO_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read PMOD TAC file contents into specified TAC data structure.
\return Returns TACIO status.
 */
int tacReadPmod(
  /** Pointer to initiated TAC struct where TAC data will be written;
   *  any old content is deleted. */
  TAC *d,
  /** Name of file to be read */
  char *filename,
  /** Verbose level; if zero, nothing is printed into stdout */
  int verbose
) {
  int ret, voiNr=0, frameNr=0;

  if(verbose>0) printf("tacReadPmod(tac, '%s', ...)\n", filename);
  if(strlen(filename)<1 || d==NULL) return TACIO_FAULT;

  /* Delete any previous data */
  tacEmpty(d);

  /* Read file into IFT struct */
  if(verbose>1) printf("reading file\n");
  ret=iftRead(&d->ift, filename, 0);
  if(ret!=IFT_OK) {
    if(verbose>0) printf("Error in reading: %s\n", d->ift.status);
    iftEmpty(&d->ift); if(ret==IFT_CANNOTREAD) return TACIO_CANNOTOPEN;
    return TACIO_INVALIDFORMAT;
  }
  if(verbose>1) printf("%d line(s).\n", d->ift.keyNr);

  /* Get data dimensions */
  if(verbose>1) printf("reading dimensions of the data\n");
  ret=iftGetDataDimensions(&d->ift, &frameNr, &voiNr);
  if(ret!=IFT_OK) {
    if(verbose>0) printf("error %d in reading dimensions of the data\n", ret);
    tacEmpty(d); return TACIO_INVALIDFORMAT;
  }
  if(verbose>2) {
    printf("data_lines := %d\n", frameNr);
    printf("max_column_nr := %d\n", voiNr);
  }

  /* Allocate memory for TAC */
  if(verbose>1) printf("allocating memory for TACs\n");
  voiNr--; // first column should be time
  ret=tacSetmem(d, frameNr, voiNr);
  if(ret!=TACIO_OK) {tacEmpty(d); return ret;}
  
  /* Move TAC data from IFT */
  if(verbose>1) printf("moving data table from IFT\n");
  ret=tacProcessPmodData(d, verbose-10);
  if(ret!=TACIO_OK) {tacEmpty(d); return ret;}

  /* Set study number from filename, if necessary */
  if(strlen(d->studynr)==0) studynr_from_fname(filename, d->studynr);
  
  /* Set file format */
  d->format=TAC_FORMAT_PMOD;

  //return TACIO_INVALIDFORMAT;
  return TACIO_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Move DFT TAC data table from IFT inside TAC struct file.
\return Returns TACIO status.
 */
int tacProcessPmodData(
  /** Pointer to TAC struct where TAC table is in IFT;
   *  corresponding parts of IFT are deleted. */
  TAC *d,
  /** Verbose level; if zero, nothing is printed into stdout */
  int verbose
) {
  int ret, i, ri, fi, ti, tabs;
  char *cptr, *cptr2, *lptr, separs[8], tmp[256];
  double v;
  
  if(verbose>0) printf("tacProcessPmodData()\n");
  if(d==NULL) return TACIO_FAULT;
  if(verbose>3) {
    printf("_framedataNr := %d\n", d->_framedataNr);
    printf("_voidataNr := %d\n", d->_voidataNr);
    printf("keyNr := %d\n", d->ift.keyNr);
  }

  /* Go through the IFT 'lines' */
  i=0; ri=0; fi=0; ti=0;
  while(i<d->ift.keyNr && fi<d->_framedataNr) {
    if(verbose>2) printf("table line := '%s'\n", d->ift.item[i].value);
    if(d->ift.item[i].type=='#') {i++; continue;}
    if(d->ift.item[i].type==';') {i++; continue;}
    if(strlen(d->ift.item[i].value)<1) {i++; continue;}
    if(strlen(d->ift.item[i].key)>0) {i++; continue;}
    /* One title line in PMOD files */
    if(ti==0) {
      /* If tabs found, then do not use space as field separator */
      tabs=strChrCount(d->ift.item[i].value, "\t");
      if(tabs>0) strcpy(separs, "\t\n\r"); else strcpy(separs, " \t\n\r");
      /* The first token in the line tell whether we have middle times or frame
         start and end times, and time units */
      lptr=d->ift.item[i].value; cptr=strtok(lptr, separs);
      if(strncasecmp(cptr, "Time[", 5)==0) {
        d->isframe=0;
        strncpy(tmp, cptr+5, 255);
	cptr2=strchr(tmp, ']'); if(cptr2==NULL) return TACIO_INVALIDFORMAT;
	*cptr2=(char)0; d->tunit=petTunitId(tmp);
      } else if(strncasecmp(cptr, "start[", 6)==0) {
        d->isframe=1;
        strncpy(tmp, cptr+6, 255);
	cptr2=strchr(tmp, ']'); if(cptr2==NULL) return TACIO_INVALIDFORMAT;
	*cptr2=(char)0; d->tunit=petTunitId(tmp);
      } else return TACIO_INVALIDFORMAT;
      /* If frame start and end times are available, then next token
         contains calibration unit */
      if(d->isframe!=0) {
        cptr=strtok(NULL, separs);
	if(strncasecmp(cptr, "end[", 4)!=0) return TACIO_INVALIDFORMAT;
        strncpy(tmp, cptr+4, 255);
	cptr2=strchr(tmp, ']'); if(cptr2==NULL) return TACIO_INVALIDFORMAT;
	*cptr2=(char)0; d->cunit=petCunitId(tmp);
      }
      /* Next tokens are TAC names */
      ri=0;
      while((cptr=strtok(NULL, separs))!=NULL && ri<d->_voidataNr) {
        strncpy(tmp, cptr, 255);
	cptr2=strchr(tmp, '['); if(cptr2!=NULL) {*cptr2=(char)0; cptr2++;}
        if(strcmp(cptr, ".")==0) strcpy(d->voi[ri].id.name, "");
        else strncpy(d->voi[ri].id.name, tmp, MAX_TACNAME_LEN);
        d->voi[ri++].id.name[MAX_TACNAME_LEN]=(char)0;
        /* If frame middle time was given, then calibration unit must be
           read here */
        if(d->isframe==0 && cptr2!=NULL) {
          cptr=strchr(cptr2, ']');
          if(cptr2!=NULL) {*cptr=(char)0; d->cunit=petCunitId(cptr2);}
	}
      }
      /* If we did not get anything, then that is bad */
      if(ri==0) return TACIO_BADTABLE;
      /* Check that we get the same nr of y values on each line */
      d->voiNr=ri; // first sample line
      if(verbose>2) printf("voiNr based on title line := %d\n", d->voiNr);
      /* Remove this line from IFT */
      ret=iftDeleteItem(&d->ift, i);
      ti++; continue; 
    }
    if(verbose>2 && ti==1 && fi==0)
      printf("PMOD title line was read; reading TAC data table\n");
    /* The first token in the line is the sample time (x or x1) */
    lptr=d->ift.item[i].value; cptr=strtok(lptr, " \t");
    if(verbose>3) printf("  1st token := '%s'\n", cptr);
    if(strcmp(cptr, ".")==0) v=nan("");
    else if(atof_with_check(cptr, &v)!=0) return TACIO_INVALIDFORMAT;
    if(d->isframe==0) {
      d->x[fi]=v;
    } else {
      d->x1[fi]=v;
      /* The 2nd token in the line is the frame end time (x2) */
      cptr=strtok(NULL, " \t");
      if(verbose>3) printf("  2nd token := '%s'\n", cptr);
      if(strcmp(cptr, ".")==0) v=nan("");
      else if(atof_with_check(cptr, &v)!=0) return TACIO_INVALIDFORMAT;
      d->x2[fi]=v; d->x[fi]=0.5*(d->x1[fi]+d->x2[fi]);
    }
    /* Next tokens are concentration values (y) */
    ri=0;
    while((cptr=strtok(NULL, " \t"))!=NULL && ri<d->_voidataNr) {
      if(verbose>3) printf("  next token := '%s'\n", cptr);
      if(strcmp(cptr, ".")==0) {ret=0; v=nan("");}
      else ret=atof_with_check(cptr, &v);
      if(ret==0) d->voi[ri++].y[fi]=v; else return TACIO_INVALIDFORMAT;
    }
    /* If we did not get any y values, then that is bad */
    if(ri==0) return TACIO_BADTABLE;
    /* Check that we get the same nr of y values on each line */
    if(ri!=d->voiNr) return TACIO_BADTABLE;
    fi++;
    /* Remove this line from IFT */
    ret=iftDeleteItem(&d->ift, i);
  }
  d->frameNr=fi;
  /* Check the data table size */
  if(d->frameNr<1 || d->voiNr<1) return TACIO_NOTABLE;

  return TACIO_OK;
}
/*****************************************************************************/

/*****************************************************************************/
