/******************************************************************************
  Copyright (c) 2013 by Turku PET Centre

  File:        tacsel.c
  Description: Functions for selecting certain curve(s) from in TAC data.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  2013-09-26 Vesa Oikonen
       First created, based on libtpccurveio.


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/tacio.h"
/*****************************************************************************/

/*****************************************************************************/
/** Select the VOIs that have matching ID name or number.
    Sets sw=1 or sw=0.
\return Returns the number of newly selected VOIs, or <0 in case of an error.
 */
int tacSelectVOIs(
  /** Pointer to TAC data where VOIs are selected */
  TAC *d,
  /** Name or VOI number (as string) which is searched; string can be inside
   *  quotation marks. */
  const char *region_name,
  /** 1=Non-matching VOIs are deselected, 0=Old selections are preserved */
  int reset
) {
  int ri, len, match_nr=0;

  /* Check the input */
  if(d==NULL || d->voiNr<1) return(-1);
  /* Remove any quotation marks from name */
  len=strlen(region_name); if(len<1) return(-1);
  char temp[len+1];
  len=strnCopyClean(temp, region_name, len+1); if(len<1) return(-1);
  /* Reset all selections if required */
  if(reset!=0) for(ri=0; ri<d->voiNr; ri++) d->voi[ri].id.sw=0;
  /* Check each VOI */
  for(ri=0; ri<d->voiNr; ri++) {
    /* does the name match? */
    if(tacNameMatch(&d->voi[ri].id, temp, 0)!=0) {
      d->voi[ri].id.sw=1; match_nr++; continue;
    }
  }
  /* If at least one match was found, then we are ready */
  if(match_nr>0) return(match_nr);

  /* Next try VOI number, if region_name is suitable */
  int n;
  if(atoi_with_check(temp, &n)!=0) return(0);
  if(n<1 || n>d->voiNr) return(0);
  d->voi[n-1].id.sw=1; match_nr=1;
  return(match_nr);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the number of selected VOIs (sw!=0) in TAC struct.
\return Returns the number of selected VOIs.
 */
int tacSelectedVOIs(
  /** Pointer to TAC data */
  TAC *d
) {
  int ri, n;

  /* Check the input */
  if(d==NULL || d->voiNr<1) return(0);
  /* Check each VOI */
  for(ri=n=0; ri<d->voiNr; ri++) if(d->voi[ri].id.sw!=0) n++;
  return(n);
}
/*****************************************************************************/

/*****************************************************************************/
/** Select the best guess of reference region in case that several VOIs were
 *  matching with tacSelectVOIs().
\return Returns the index of best region, or <0 in case of an error.
 */
int tacSelectBestReference(
  /** Pointer to TAC struct, after using tacSelectVOIs() */
  TAC *d
) {
  int ri, len, min_len, i, n, ni;
  
  if(d==NULL || d->voiNr<1) return -1;
  if(tacSelectedVOIs(d)<1) return -2;
  for(ri=0, i=-1, min_len=9999; ri<d->voiNr; ri++) {
    if(d->voi[ri].id.sw==0) continue;
    n=tacNameSplit(&d->voi[ri].id); if(n<1) continue;
    for(ni=len=0; ni<n; ni++) {
      if(strcasecmp(d->voi[ri].id.sub[ni], "ALL")==0) continue;
      if(strcasecmp(d->voi[ri].id.sub[ni], "AVG")==0) continue;
      if(strcasecmp(d->voi[ri].id.sub[ni], "MEAN")==0) continue;
      len+=strlen(d->voi[ri].id.sub[ni]);
    }
    if(len<min_len) {min_len=len; i=ri;}
  }
  if(i<0) return -2; else return i;
}
/*****************************************************************************/

/*****************************************************************************/
