/******************************************************************************
  Copyright (c) 2013 by Turku PET Centre

  File:        tacsimp.c
  Description: Functions for TAC file in Simple format.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  2013-09-10 Vesa Oikonen
       First created.
     


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/tacio.h"
/*****************************************************************************/

/*****************************************************************************/
/** Read Simple TAC file contents into specified TAC data structure.
\return Returns TACIO status.
 */
int tacReadSimple(
  /** Pointer to initiated TAC struct where TAC data will be written;
   *  any old content is deleted. */
  TAC *d,
  /** Name of file to be read */
  char *filename,
  /** Verbose level; if zero, nothing is printed into stdout */
  int verbose
) {
  int ret, voiNr=0, frameNr=0;

  if(verbose>0) printf("tacReadSimple(tac, '%s', ...)\n", filename);
  if(strlen(filename)<1 || d==NULL) return TACIO_FAULT;

  /* Delete any previous data */
  tacEmpty(d);

  /* Read file into IFT struct */
  if(verbose>1) printf("reading file\n");
  ret=iftRead(&d->ift, filename, 0);
  if(ret!=IFT_OK) {
    if(verbose>0) printf("Error in reading: %s\n", d->ift.status);
    iftEmpty(&d->ift); if(ret==IFT_CANNOTREAD) return TACIO_CANNOTOPEN;
    return TACIO_INVALIDFORMAT;
  }
  if(verbose>1) printf("%d line(s).\n", d->ift.keyNr);

  /* Get data dimensions */
  if(verbose>1) printf("reading dimensions of the data\n");
  ret=iftGetDataDimensions(&d->ift, &frameNr, &voiNr);
  if(ret!=IFT_OK) {
    if(verbose>0) printf("error %d in reading dimensions of the data\n", ret);
    tacEmpty(d); return TACIO_INVALIDFORMAT;
  }
  if(verbose>2) {
    printf("data_lines := %d\n", frameNr);
    printf("max_column_nr := %d\n", voiNr);
  }

  /* Allocate memory for TAC */
  if(verbose>1) printf("allocating memory for TACs\n");
  voiNr--; // first column should be time
  ret=tacSetmem(d, frameNr, voiNr);
  if(ret!=TACIO_OK) {tacEmpty(d); return ret;}
  
  /* Move TAC data from IFT */
  if(verbose>1) printf("moving data table from IFT\n");
  ret=tacProcessSimpleData(d, verbose-10);
  if(ret!=TACIO_OK) {tacEmpty(d); return ret;}


  /* Set study number from filename, if necessary */
  if(strlen(d->studynr)==0) studynr_from_fname(filename, d->studynr);
  
  /* Set file format */
  d->format=TAC_FORMAT_SIMPLE;

  //return TACIO_INVALIDFORMAT;
  return TACIO_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Move Simple TAC data table from IFT inside TAC struct file.
\return Returns TACIO status.
 */
int tacProcessSimpleData(
  /** Pointer to TAC struct where TAC table is in IFT;
   *  corresponding parts of IFT are deleted. */
  TAC *d,
  /** Verbose level; if zero, nothing is printed into stdout */
  int verbose
) {
  int ret, i, n, ri, fi, tokNr;
  char tmp[256];
  double v;
  
  if(verbose>0) printf("tacProcessSimpleData()\n");
  if(d==NULL) return TACIO_FAULT;
  if(verbose>3) {
    printf("_framedataNr := %d\n", d->_framedataNr);
    printf("_voidataNr := %d\n", d->_voidataNr);
    printf("keyNr := %d\n", d->ift.keyNr);
  }
  
  /* Go through the IFT 'lines' */
  i=0; ri=0; fi=0;
  while(i<d->ift.keyNr && fi<d->_framedataNr) {
    if(verbose>2) printf("table line := '%s'\n", d->ift.item[i].value);
    if(d->ift.item[i].type=='#') {i++; continue;}
    if(d->ift.item[i].type==';') {i++; continue;}
    if(strlen(d->ift.item[i].key)>0) {i++; continue;}
    tokNr=strTokenNr(d->ift.item[i].value, " \t"); if(tokNr<2) {i++; continue;}
    /* The first token in the line is the sample time (x) */
    n=strTokenNCpy(d->ift.item[i].value, " \t", 1, tmp, 256);
    if(verbose>3) printf("  1st token := '%s'\n", tmp);
    if(n==1 && tmp[0]=='.') {ret=0; v=nan("");}
    else ret=atof_with_check(tmp, &v);
    if(ret==0) d->x[fi]=v; else return TACIO_BADTABLE;
    /* Next tokens are concentration values (y) */
    for(ri=0; ri<tokNr-1; ri++) {
      n=strTokenNCpy(d->ift.item[i].value, " \t", 2+ri, tmp, 256);
      if(verbose>3) printf("  next token := '%s'\n", tmp);
      if(n==1 && tmp[0]=='.') {ret=0; v=nan("");}
      else ret=atof_with_check(tmp, &v);
      if(ret==0) d->voi[ri].y[fi]=v; else return TACIO_BADTABLE;
    }
    /* Check that we get the same nr of y values on each line */
    if(fi==0) { // first sample line
      d->voiNr=ri;
    } else { // next sample lines
      if(ri!=d->voiNr) return TACIO_BADTABLE;
    }
    fi++;
    /* Remove this line from IFT */
    ret=iftDeleteItem(&d->ift, i);
  }
  d->frameNr=fi;
  /* Check the data table size */
  if(d->frameNr<1 || d->voiNr<1) return TACIO_NOTABLE;

  return TACIO_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Write TAC data into specified file pointer in Simple format.
    Number of decimals can be determined by changing global variable
    TAC_NR_OF_DECIMALS.
\return Returns TACIO status.
 */
int tacWriteSimple(
  /** Pointer to TAC struct, contents of which are to be written */
  TAC *tac,
  /** File pointer */
  FILE *fp,
  /** Write (1) or do not write (0) also extra header fields found in IFT */
  int extra,
  /** Verbose level; if zero, nothing extra is printed into stdout */
  int verbose
) {
  int fi, ri, n, prec;
  double v;

  if(verbose>0) printf("tacWriteSimple(tac, fp, %d, ...)\n", extra);

  /* Check that there is some data to write */
  if(tac==NULL) return TACIO_FAULT;
  if(tac->voiNr<1 || tac->frameNr<1) return TACIO_NOTABLE;

  /* Write data */
  prec=0; if(TAC_NR_OF_DECIMALS>prec) prec=TAC_NR_OF_DECIMALS;
  for(fi=0; fi<tac->frameNr; fi++) {
    /* Time (x) */
    if(tac->isframe==0) v=tac->x[fi]; else v=0.5*(tac->x1[fi]+tac->x2[fi]);
    if(isnan(v)) n=fprintf(fp, "."); else n=fprintf(fp, "%.5f", v);
    if(n<1) return TACIO_CANNOTWRITE;
    /* Concentrations (y values) */
    for(ri=0; ri<tac->voiNr; ri++) {
      if(isnan(tac->voi[ri].y[fi])) n=fprintf(fp, "\t.");
      else fprintf(fp, "\t%.*e", prec, tac->voi[ri].y[fi]);
      if(n<1) return TACIO_CANNOTWRITE;
    }
    n=fprintf(fp, "\n"); if(n<1) return TACIO_CANNOTWRITE;
  }
  
  return(TACIO_OK);
}
/*****************************************************************************/

/*****************************************************************************/
