/** @file anabyteo.c
 *  @brief Convert byte order in Analyze 7.5 images.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 *  @test Add tests.
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <time.h>
#include <sys/stat.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Show or optionally convert the byte order in Analyze 7.5 images.",
  "Without options the current byte order is shown but not changed.",
  " ",
  "Usage: @P [Options] database(s)>",
  " ",
  "Options:",
  " -little",
  "     Change byte order to little endian (PC Intel).",
  " -big",
  "     Change byte order to big endian (Sun Sparc, Motorola, PowerPC).",
  " -o=<directory>",
  "     Converted files are placed in specified directory, and original",
  "     files are not overwritten.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: ana2ecat, ecat2ana, ana_lhdr, ana_ehdr, convend",
  " ",
  "Keywords: image, format conversion, Analyze, byte order",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int       ai, help=0, version=0, verbose=1;
  int       errorNr=0;
  unsigned int byteSize;
  int       ffi=0, fileNr=0;
  int       orderTo=-1; /* 1=big, 0=little */
  char      dbname[FILENAME_MAX], datfile[FILENAME_MAX], hdrfile[FILENAME_MAX];
  char      temp[FILENAME_MAX], outpath[FILENAME_MAX], *cptr;
  char      dbname2[FILENAME_MAX], datfile2[FILENAME_MAX], hdrfile2[FILENAME_MAX];
  FILE     *fpi, *fpo;
  unsigned char buf[32];
  ANALYZE_DSR dsr;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  outpath[0]=dbname[0]=datfile[0]=hdrfile[0]=(char)0;
  dbname2[0]=datfile2[0]=hdrfile2[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strncasecmp(cptr, "LITTLE", 1)==0) {
      orderTo=0; continue;
    } else if(strncasecmp(cptr, "BIG", 1)==0) {
      orderTo=1; continue;
    } else if(strncasecmp(cptr, "O=", 2)==0) {
      cptr+=2; if(strlen(cptr)>0) {strcpy(outpath, cptr); continue;}
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break;

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(ffi<1) ffi=ai;
    /* Check that Analyze file (*.hdr and *.img) exists */
    if(anaDatabaseExists(argv[ai], hdrfile, datfile, NULL) > 0) {fileNr++; continue;}
    fprintf(stderr, "Error: Analyze file '%s' does not exist.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(fileNr<1) {
    fprintf(stderr, "Error: no Analyze files were specified.\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++) printf(" %s", argv[ai]);
    printf("\n");
    printf("orderTo := %d\n", orderTo);
    printf("outpath := %s\n", outpath);
    printf("fileNr := %d\n", fileNr);
  }
  if(verbose>3) ANALYZE_TEST=verbose-3; else ANALYZE_TEST=0;


  /*
   *  Process each Analyze image
   */
  if(verbose>1) printf("processing %d Analyze file(s)\n", fileNr);
  for(ai=ffi; ai<argc; ai++) {

    /* Check and process the given filename */
    strlcpy(dbname, argv[ai], FILENAME_MAX);
    /* If it has an Analyze extension, remove it */
    anaRemoveFNameExtension(dbname);
    if(verbose>0) fprintf(stdout, "%s : \n", dbname);

    /* Make hdr and img filenames */
    if(anaDatabaseExists(dbname, hdrfile, datfile, NULL) < 1) {
      fprintf(stderr, "  Error: Analyze file '%s' does not exist.\n", dbname);
      errorNr++; continue;
    }

    /* Read analyze header */
    if(anaReadHeader(hdrfile, &dsr)) {
      fprintf(stderr, "  Error: cannot read header file %s\n", hdrfile);
      errorNr++; continue;
    }

    /* Show the current byte order */
    if(verbose>0 || orderTo<0) {
      if(dsr.little)
        fprintf(stdout, "  currently little endian (Intel) byte order.\n");
      else
        fprintf(stdout, "  currently big endian (SUN Sparc) byte order.\n");
    }

    /* If no option was given, then do nothing else */
    if(orderTo<0) continue;

    /* If byte order is already the required one, then do nothing */
    if((dsr.little==1 && orderTo==0) || (dsr.little==0 && orderTo==1)) {
      if(verbose>0) fprintf(stdout, "  no conversion was necessary.\n");
      continue;
    }
    
    /* Make output Analyze database name */
    if(!outpath[0]) {
      strcpy(dbname2, dbname);
    } else {
      cptr=outpath+strlen(outpath)-1;
      if(*cptr=='/' || *cptr=='\\') *cptr=(char)0;
      cptr=strrchr(dbname, '/'); if(cptr==NULL) cptr=strrchr(dbname, '\\');
      if(cptr==NULL) cptr=dbname; else cptr++;
      snprintf(dbname2, FILENAME_MAX, "%s/%s", outpath, cptr);
      /* Create output directory if necessary */
      if(access(outpath, 0) == -1) {
        if(verbose>0) fprintf(stdout, "  creating subdirectory %s\n", outpath);
#ifdef WIN32
        int ret=mkdir(outpath);
#else
        int ret=mkdir(outpath, 00775);
#endif
        if(ret!=0) {
          fprintf(stderr, "  Error: cannot create subdirectory.\n");
          errorNr++;
          return(5); /* there is no reason to try the next image */
        }
      }
    }
    if(verbose>1) fprintf(stdout, "  Output dbname='%s'\n", dbname2);
    strcpy(datfile2, dbname2); strcat(datfile2, ".img");
    strcpy(hdrfile2, dbname2); strcat(hdrfile2, ".hdr");

    /* Change byte order of the image data */
    /* Get the nr of pixels and their size in bytes */
    size_t pxlNr=dsr.dime.dim[1]*dsr.dime.dim[2];
    if(dsr.dime.dim[0]>2) pxlNr*=dsr.dime.dim[3];
    if(dsr.dime.dim[0]>3) pxlNr*=dsr.dime.dim[4];
    byteSize=dsr.dime.bitpix/8;
    if(verbose>1)
      fprintf(stdout, "  %zu pixels, %d byte(s)/pixel\n", pxlNr, byteSize);
    if(pxlNr<1) {
      fprintf(stderr, "  Error: invalid image dimensions.\n");
      errorNr++; continue;
    }
    if(byteSize>8) {
      fprintf(stderr, "  Error: unsupported bitpix.\n");
      errorNr++; continue;
    } else if(byteSize<2) {
      if(verbose>0)
        fprintf(stdout, "  no conversion is needed for image file.\n");
    } else {
      /* Make input image datafile name */
      if(verbose>0) {
        if(!outpath[0]) fprintf(stdout, "  converting %s\n", datfile);
        else fprintf(stdout, "  converting %s to %s\n", datfile, datfile2);
      }
      /* Open original datafile */
      if((fpi=fopen(datfile, "rb")) == NULL) {
        fprintf(stderr, "  Error: cannot read image file %s\n", datfile);
        errorNr++; continue;
      }
      /* Open temporary datafile */
      strcpy(temp, datfile2); strcat(temp, "%%");
      if(verbose>2) printf("  temp file: %s\n", temp);
      if((fpo=fopen(temp, "wb")) == NULL) {
        fprintf(stderr, "  Error: cannot write temp file %s\n", temp);
        errorNr++; fclose(fpi); continue;
      }
      /* Read, swap, and write each pixel value */
      int ret=0;
      for(size_t i=0; i<pxlNr; i++) {
        if(fread(buf, 1, byteSize, fpi) < byteSize) {ret=1; break;}
        swap(buf, buf, byteSize);
        if(fwrite(buf, 1, byteSize, fpo)!=byteSize) {ret=2; break;}
      } /* next pixel */
      fclose(fpi); fclose(fpo);
      if(ret) {
        if(ret==1) fprintf(stderr, "  Error: cannot read %s\n", datfile);
        else fprintf(stderr, "  Error: cannot write temp file %s\n", temp);
        errorNr++; remove(temp); continue;      
      }
    }

    /* Write header with different byte order */
    if(verbose>0) {
      if(!outpath[0]) fprintf(stdout, "  converting %s\n", hdrfile);
      else fprintf(stdout, "  converting %s to %s\n", hdrfile, hdrfile2);
    }
    if(dsr.little==0) dsr.little=1; else dsr.little=0;
    if(anaWriteHeader(hdrfile2, &dsr)) {
      fprintf(stderr, "  Error: cannot write header file %s\n", hdrfile2);
      if(byteSize>1) remove(temp);
      errorNr++; continue;
    }

    /* If no errors, then replace the original image datafile by new */
    /* or just rename the temp file */
    if(byteSize>1) {
      if(verbose>2)
        printf("  remove(%s); rename(%s, %s);\n", datfile2, temp, datfile2);
      remove(datfile2); rename(temp, datfile2);      
    }

  } /* next image */

  if(errorNr>0) return(errorNr+10);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
