#!/bin/bash
#: Title      : test_fit_srtm
#: Date       : 2020-02-03
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../fit_srtm$EXT;

if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n verifying that required data exists \n"

if [ ! -f rpplasma.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f frames1.dat ] || [ ! -f frames1_smallgap.dat ] || [ ! -f frames1_biggap.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f frames1_smalloverlap.dat ] || [ ! -f frames1_bigoverlap.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f frames1_few.dat ] || [ ! -f frames2.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f correct_basic1.res ] || [ ! -f correct_basic2.res ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_basicdvr.res ] || [ ! -f correct_rpsim.res ] || [ ! -f correct_nonphys.res ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


printf "\n making data for testing \n"

if [ ! -f sim1.par ]; then
  rm -f rpsim_full.tac
  printf "# model := SER3TCM\n" > sim1.par
  printf "Parameters\tK1\tk2\tk3\tk4\tk5\tk6\tVb\tf\tfA\n" >> sim1.par
  printf "cereb__all\t0.11\t0.35\t0.0\t0.0\t0\t0\t0\t0\t0\n" >> sim1.par
  printf "putam_d_all\t0.153\t0.489\t0.303\t0.0926\t0\t0\t0\t0\t0\n" >> sim1.par
  printf "putam_s_all\t0.153\t0.489\t0.303\t0.1852\t0\t0\t0\t0\t0\n" >> sim1.par
fi

if [ ! -f rpsim_full.tac ]; then
  rm -f rpsim.tac rpsim_smallgap.tac rpsim_biggap.tac rpsim_smalloverlap.tac rpsim_bigoverlap.tac rpsim_fewframes.tac
  sim_3tcm -nosub sim1.par rpplasma.kbq rpplasma.kbq rpsim_full.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force rpsim_full.tac rpsim1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f rpsim.tac ]; then
  simframe rpsim_full.tac frames1.dat rpsim.tac C-11
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dftweigh rpsim.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f rpsim_smallgap.tac ]; then
  simframe rpsim_full.tac frames1_smallgap.dat rpsim_smallgap.tac C-11
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dftweigh rpsim_smallgap.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f rpsim_biggap.tac ]; then
  simframe rpsim_full.tac frames1_biggap.dat rpsim_biggap.tac C-11
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dftweigh rpsim_biggap.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f rpsim_smalloverlap.tac ]; then
  simframe rpsim_full.tac frames1_smalloverlap.dat rpsim_smalloverlap.tac C-11
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dftweigh rpsim_smalloverlap.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f rpsim_bigoverlap.tac ]; then
  simframe rpsim_full.tac frames1_bigoverlap.dat rpsim_bigoverlap.tac C-11
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dftweigh rpsim_bigoverlap.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f rpsim_fewframes.tac ]; then
  simframe rpsim_full.tac frames1_few.dat rpsim_fewframes.tac C-11
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dftweigh rpsim_fewframes.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f sim2.par ]; then
  rm -f basicsim_full.tac
  printf "# model := SER3TCM\n" > sim2.par
  printf "Parameters\tK1[mL/(mL*min)]\tk2[1/min]\tk3[1/min]\tk4[1/min]\tk5\tk6\tVb\tf\tfA\n" >> sim2.par
  printf "refer-lowk\t0.100\t0.200\t0.0\t0.0\t0\t0\t0\t0\t0\n" >> sim2.par
  printf "refer-highk\t0.400\t0.800\t0.0\t0.0\t0\t0\t0\t0\t0\n" >> sim2.par
  printf "tis3\t0.200\t0.333333\t0.0\t0.0\t0\t0\t0\t0\t0\n" >> sim2.par
  printf "tis4\t0.200\t0.133333\t0.0\t0.0\t0\t0\t0\t0\t0\n" >> sim2.par
  printf "tis5\t0.200\t0.036364\t0.0\t0.0\t0\t0\t0\t0\t0\n" >> sim2.par
  printf "tis6\t0.200\t0.007843\t0.0\t0.0\t0\t0\t0\t0\t0\n" >> sim2.par
  printf "tis7\t0.050\t0.020000\t0.0\t0.0\t0\t0\t0\t0\t0\n" >> sim2.par
  printf "tis8\t0.100\t0.040000\t0.0\t0.0\t0\t0\t0\t0\t0\n" >> sim2.par
  printf "tis9\t0.300\t0.120000\t0.0\t0.0\t0\t0\t0\t0\t0\n" >> sim2.par
fi

if [ ! -f basicsim_full.tac ]; then
  rm -f basicsim.tac
  sim_3tcm -nosub sim2.par rpplasma.kbq rpplasma.kbq basicsim_full.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force basicsim_full.tac sim2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f basicsim.tac ]; then
  rm -f basicsim.tac basicsim.dat
  simframe basicsim_full.tac frames2.dat basicsim.tac C-11
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dftweigh basicsim.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f basicsim.dft ]; then
  tacformat -f=DFT basicsim.tac basicsim.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f basicsim.dat ]; then
  tacformat -f=SIMPLE -hdr=no -mid basicsim.tac basicsim.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f sim3.par ] || [ ! -f nonphyssim.tac ]; then
  printf "# model := SER3TCM\n" > sim3.par
  printf "Parameters\tK1[mL/(mL*min)]\tk2[1/min]\tk3[1/min]\tk4[1/min]\tk5\tk6\tVb\tf\tfA\n" >> sim3.par
  printf "refer\t0.2\t0.4\t0.0\t0.0\t0\t0\t0\t0\t0\n" >> sim3.par
  printf "irrev\t0.2\t4.0E-07\t0.0\t0.0\t0\t0\t0\t0\t0\n" >> sim3.par
  printf "noflow\t1.0E-06\t3.33E-07\t0.0\t0.0\t0\t0\t0\t0\t0\n" >> sim3.par
  printf "fine\t0.2\t0.2\t0.0\t0.0\t0\t0\t0\t0\t0\n" >> sim3.par

  sim_3tcm -nosub sim3.par rpplasma.kbq rpplasma.kbq nonphyssim_full.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force nonphyssim_full.tac sim3
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe nonphyssim_full.tac frames2.dat nonphyssim.tac C-11
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi





printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n 1.0.0 \n"
printf " Test case 1: Fit TACs simulated with realistic range of parameters. \n"
printf " Test case 2: Reference inside TTAC file. \n"
printf " Expected result 1: Result file is created, and results are close to correct \n"
printf " Expected result 2: SVG plot is created. \n"
printf " Expected result 3: Fitted TACs should be close to original TACs.\n\n"

rm -f fit_1_0_0.res fit_1_0_0.tac fit_1_0_0.svg

$PROGRAM -fit=fit_1_0_0.tac -svg=fit_1_0_0.svg basicsim.tac "refer lowk" 120 fit_1_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

if [ ! -f fit_1_0_0.tac ] || [ ! -f fit_1_0_0.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


resmatch -abs=0.1 -header=y -param=y -regions=y -res=1 correct_basic1.res fit_1_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.05 -header=y -param=y -regions=y -res=2 correct_basic1.res fit_1_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.1 -header=y -param=y -regions=y -res=4 correct_basic1.res fit_1_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -abs=1.0 -rel=1 basicsim.tac fit_1_0_0.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case: Same as above but with different reference region. \n"
printf " Expected result 1: Result file is created, and results are close to correct \n"
printf " Expected result 2: SVG plot is created. \n"
printf " Expected result 3: Fitted TACs should be close to original TACs.\n\n"

rm -f fit_1_0_1.res fit_1_0_1.tac fit_1_0_1.svg

$PROGRAM -fit=fit_1_0_1.tac -svg=fit_1_0_1.svg basicsim.tac "refer highk" 120 fit_1_0_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

if [ ! -f fit_1_0_1.tac ] || [ ! -f fit_1_0_1.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


resmatch -abs=0.1 -header=y -param=y -regions=y -res=1 correct_basic2.res fit_1_0_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.05 -header=y -param=y -regions=y -res=2 correct_basic2.res fit_1_0_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.1 -header=y -param=y -regions=y -res=4 correct_basic2.res fit_1_0_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -abs=1.0 -rel=1 basicsim.tac fit_1_0_1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.1.0 \n"
printf " Test case 1: Fit TACs simulated with nonphysiological parameters. \n"
printf " Test case 2: Reference inside TTAC file. \n"
printf " Expected result 1: Result file is created \n"
printf " Expected result 2: SVG plot is created. \n\n"

rm -f fit_1_1_0.res fit_1_1_0.tac fit_1_1_0.svg

$PROGRAM -fit=fit_1_1_0.tac -svg=fit_1_1_0.svg nonphyssim.tac refer 120 fit_1_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

if [ ! -f fit_1_1_0.tac ] || [ ! -f fit_1_1_0.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


resmatch -abs=0.5 -header=y -param=y -regions=y -res=1 correct_nonphys.res fit_1_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.2 -header=y -param=y -regions=y -res=2 correct_nonphys.res fit_1_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=1000000 -header=y -param=y -regions=y -res=4 correct_nonphys.res fit_1_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -abs=5.0 -rel=20 nonphyssim.tac fit_1_1_0.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.2.0 \n"
printf " Test case 1: Fit TACs simulated with realistic parameters for raclopride. \n"
printf " Test case 2: Reference inside TTAC file. \n"
printf " Expected result 1: Result file is created, and results are close to correct \n"
printf " Expected result 2: SVG plot is created. \n"
printf " Expected result 3: Fitted TACs should be close to original TACs.\n\n"

rm -f fit_1_2_0.res fit_1_2_0.tac fit_1_2_0.svg

$PROGRAM -fit=fit_1_2_0.tac -svg=fit_1_2_0.svg rpsim.tac "cereb" 60 fit_1_2_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

if [ ! -f fit_1_2_0.tac ] || [ ! -f fit_1_2_0.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

resmatch -abs=0.4 -header=y -param=y -regions=y -res=1 correct_rpsim.res fit_1_2_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.3 -header=y -param=y -regions=y -res=2 correct_rpsim.res fit_1_2_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.02 -header=y -param=y -regions=y -res=4 correct_rpsim.res fit_1_2_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

taccut rpsim.tac 0 60 test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=1.0 -rel=1 test.tac fit_1_2_0.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.3.0 \n"
printf " Test case: DVR is calculated. \n"
printf " Expected result: Result file is created, and results are close to correct \n\n"

rm -f fit_1_3_0.res

$PROGRAM -dvr basicsim.tac "refer lowk" 120 fit_1_3_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

resmatch -abs=0.1 -header=y -param=y -regions=y -res=1-4 correct_basicdvr.res fit_1_3_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 1.4.0 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -nonexistingoption basicsim refer 900 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-nonexistingoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.4.1 \n"
printf " Test case: Missing argument. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM basicsim.tac refer 900 &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM basicsim.tac refer output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM basicsim.tac 999 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 2.0.0 \n"
printf " Test case 1: Fit TACs simulated with realistic parameters. \n"
printf " Test case 2: Reference regions in separate TTAC file. \n"
printf " Expected result 1: Result file is created, and results are close to correct. \n"
printf " Expected result 2: Warning about several reference regions. \n"
printf " Expected result 3: Reference TACs not saved with fitted TTACs. \n"
printf " Expected result 4: Reference regions omitted from result file. \n\n"

rm -f fit_2_0_0.res fit_2_0_0.tac fit_2_0_0.svg stdout.txt

cp -f basicsim.tac test.tac
tacadd -ovr -move refer.tac test.tac refer
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

cp -f correct_basic1.res test.res
resdel test.res refer
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"


$PROGRAM -fit=fit_2_0_0.tac -svg=fit_2_0_0.svg test.tac refer.tac 120 fit_2_0_0.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

if [ ! -f fit_2_0_0.tac ] || [ ! -f fit_2_0_0.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

iftisval fit_2_0_0.res "Reference region" "refer lowk"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "several reference regions found: refer lowk selected."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

resmatch -abs=0.1 -header=n -param=y -regions=y -res=1 test.res fit_2_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.05 -header=n -param=y -regions=y -res=2 test.res fit_2_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.1 -header=n -param=y -regions=y -res=4 test.res fit_2_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -abs=1.0 -rel=1 test.tac fit_2_0_0.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.0.1 \n"
printf " Test case: Given reference region file does not exist. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt

$PROGRAM basicsim.tac nonexisting.tac 999 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading reference input" "cannot find region"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.1.0 \n"
printf " Test case 1: Fit TACs simulated with realistic parameters. \n"
printf " Test case 2: Reference regions in the same TTAC file. \n"
printf " Test case 3: Reference name given matches with two regions. \n"
printf " Expected result 1: Warning about several reference regions. \n"
printf " Expected result 2: All TACs saved with fitted TTACs. \n"
printf " Expected result 3: Only selected reference omitted from result file. \n\n"

rm -f fit_2_1_0.res fit_2_1_0.tac fit_2_1_0.svg stdout.txt

$PROGRAM -fit=fit_2_1_0.tac -svg=fit_2_1_0.svg basicsim.tac refer 120 fit_2_1_0.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval fit_2_1_0.res "Reference region" "refer lowk"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "several reference regions found: refer lowk selected."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

resmatch -abs=0.2 -header=y -param=y -regions=y correct_basic1.res fit_2_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -abs=1.0 -rel=1 basicsim.tac fit_2_1_0.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.1 \n"
printf " Test case 1: Fit TACs simulated with realistic parameters. \n"
printf " Test case 2: Reference regions in the same TTAC file. \n"
printf " Test case 3: Precise reference name given matches with only one region. \n"
printf " Expected result 1: No warning about several reference regions. \n"
printf " Expected result 2: All TACs saved with fitted TTACs. \n"
printf " Expected result 3: Only selected reference omitted from result file. \n\n"

rm -f fit_2_1_1.res fit_2_1_1.tac fit_2_1_1.svg stdout.txt

$PROGRAM -fit=fit_2_1_1.tac basicsim.tac "refer highk" 120 fit_2_1_1.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval fit_2_1_1.res "Reference region" "refer highk"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning"
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

resmatch -abs=0.2 -header=y -param=y -regions=y correct_basic2.res fit_2_1_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -abs=1.0 -rel=1 basicsim.tac fit_2_1_1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.0 \n"
printf " Test case 1: Reference regions in the same TTAC file. \n"
printf " Test case 2: Reference number given. \n"
printf " Expected result 1: Correct reference region used. \n"
printf " Expected result 2: Existing result file can be overwritten. \n\n"

rm -f fit_2_2_0.res stdout.txt

$PROGRAM basicsim.tac 1 120 fit_2_2_0.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval fit_2_2_0.res "Reference region" "refer lowk"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" 
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

resmatch -abs=0.2 -header=y -param=y -regions=y correct_basic1.res fit_2_2_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


$PROGRAM basicsim.tac 2 120 fit_2_2_0.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval fit_2_2_0.res "Reference region" "refer highk"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" 
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

resmatch -abs=0.2 -header=y -param=y -regions=y correct_basic2.res fit_2_2_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.3.0 \n"
printf " Test case: Given reference region name does not exist. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt

$PROGRAM basicsim.tac stupid 999 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading reference input" "cannot find region"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.3.1 \n"
printf " Test case: Given reference region nr does not exist. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt

$PROGRAM basicsim.tac 30 999 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading reference input" "cannot find region"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.0.0 \n"
printf " Test case: TTAC file in DFT format. \n"
printf " Expected result 1: Result file is created, and results are close to correct. \n\n"

rm -f fit_3_0_0.res

$PROGRAM basicsim.dft "refer lowk" 120 fit_3_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval fit_3_0_0.res "Reference region" "refer lowk"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

resmatch -abs=0.2 -header=n -param=y -regions=y -res=1-4 correct_basic1.res fit_3_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.0 \n"
printf " Test case: TTAC file in simple text format without headers. \n"
printf " Expected result 1: Result file is created, and results are close to correct. \n\n"

rm -f fit_3_1_0.res

$PROGRAM basicsim.dat 2 120 fit_3_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

resmatch -abs=0.2 -header=n -param=y -regions=n -res=1-4 correct_basic2.res fit_3_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.2.0 \n"
printf " Test case: TTAC file in PMOD format with frame middle times. \n"
printf " Expected result: Result file is created, and results are close to correct. \n\n"

rm -f fit_3_2_0.res

tacformat -f=PMOD -mid basicsim.tac test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test.tac 2 120 fit_3_2_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

resmatch -abs=0.2 -header=n -param=y -regions=n -res=1-4 correct_basic2.res fit_3_2_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.3.0 \n"
printf " Test case: Result files are compatible with rescoll. \n"
printf " Expected result: rescoll processes result files without error. \n\n"

rm -f fit_3_3_0a.res fit_3_3_0b.res fit_3_3_0c.res

$PROGRAM rpsim.tac cereb 30 fit_3_3_0a.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM rpsim.tac cereb 45 fit_3_3_0b.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM rpsim.tac cereb 60 fit_3_3_0c.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

rescoll temp.html fit_3_3_0a.res fit_3_3_0b.res fit_3_3_0c.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.4.0 \n"
printf " Test case: Nonexisting data file. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt

$PROGRAM nonexistingfile.dat cereb 60 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexistingfile.dat'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 4.0.0 \n"
printf " Test case: 3 s time gap in the beginning. \n"
printf " Expected result 1: Result file is created, and results are about the same. \n"
printf " Expected result 2: No warning. \n\n"

rm -f fit_4_0_0.res stdout.txt
tactime rpsim.tac 0.05 test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM rpsim.tac "cereb" 60 test.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"


$PROGRAM -w1 test.tac cereb 60 fit_4_0_0.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt "Warning" 
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

resmatch -abs=0.1 -header=n -param=y -regions=y -res=1-4 test.res fit_4_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.0.1 \n"
printf " Test case: 6 s time gap in the beginning. \n"
printf " Expected result 1: Result file is created, and results are about the same. \n"
printf " Expected result 2: Warning. \n\n"

rm -f fit_4_0_1.res stdout.txt
tactime rpsim.tac 0.10 test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -w1 test.tac cereb 60 fit_4_0_1.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt "Warning" "TACs should start at time zero."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

resmatch -abs=0.1 -header=n -param=y -regions=y -res=1-4 test.res fit_4_0_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.0.2 \n"
printf " Test case:       1 min time gap in the beginning. \n"
printf " Expected result: Error. \n\n"

rm -f fit_4_0_2.res stdout.txt
tactime rpsim.tac 1.0 test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -w1 test.tac cereb 60 fit_4_0_2.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ -f fit_4_0_2.res ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

iftisval stdout.txt "Error" "TACs must start at time zero."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 4.1.0 \n"
printf " Test case:       Small gap between frames. \n"
printf " Expected result: Result file is created, and results are about the same. \n\n"

rm -f fit_4_1_0.res stdout.txt

$PROGRAM rpsim.tac "cereb" 60 test.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM rpsim_smallgap.tac cereb 60 fit_4_1_0.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

resmatch -abs=0.001 -header=n -param=y -regions=y -res=1-4 test.res fit_4_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 4.1.1 \n"
printf " Test case:       Larger gap between frames. \n"
printf " Expected result: Result file is created, and results are still about the same. \n\n"

rm -f fit_4_1_1.res stdout.txt

$PROGRAM rpsim_biggap.tac cereb 60 fit_4_1_1.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

resmatch -abs=0.01 -header=n -param=y -regions=y -res=1-4 test.res fit_4_1_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.2.0 \n"
printf " Test case:       Small frame overlap. \n"
printf " Expected result: Result file is created, and results are about the same. \n\n"

rm -f fit_4_2_0.res stdout.txt

$PROGRAM rpsim_smalloverlap.tac cereb 60 fit_4_2_0.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

resmatch -abs=0.001 -header=n -param=y -regions=y -res=1-4 test.res fit_4_2_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 4.2.1 \n"
printf " Test case:       Larger frame overlap. \n"
printf " Expected result: Error. \n\n"

rm -f fit_4_2_1.res stdout.txt

$PROGRAM -d rpsim_bigoverlap.tac cereb 60 fit_4_2_1.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ -f fit_4_2_1.res ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

iftisval stdout.txt "Error" "rpsim_bigoverlap.tac has overlapping frame times."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 4.3.0 \n"
printf " Test case: Reference file has same times but contains only frame middle times. \n"
printf " Expected result: Result file is created, and results are about the same. \n\n"

rm -f fit_4_3_0.res
cp -f rpsim_fewframes.tac test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd -ovr -move ref.tac test.tac cereb
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacformat -f=PMOD -mid ref.tac refm.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test.tac ref.tac 70 test.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test.tac refm.tac 70 fit_4_3_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

resmatch -abs=0.001 -header=n -param=y -regions=y -res=1-4 test.res fit_4_3_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 4.4.0 \n"
printf " Test case: Reference file has different calibration unit. \n"
printf " Expected result: Result are the same. \n\n"

rm -f fit_4_4_0.res fit_4_4_0.tac fit_4_4_0.svg
cp -f rpsim.tac test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd -ovr -move ref.tac test.tac cereb
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacunit -yconv=kBq/mL ref.tac test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

cp -f ref.tac ref2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
cp -f test.tac test2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -yconv=Bq/mL ref2.tac test2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"


$PROGRAM test.tac ref.tac 60 test.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -fit=fit_4_4_0.tac -svg=fit_4_4_0.svg test.tac ref2.tac 60 fit_4_4_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

resmatch -abs=0.001 -header=n -param=y -regions=y -res=1 test.res fit_4_4_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


$PROGRAM test2.tac ref2.tac 60 test.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test2.tac ref.tac 60 fit_4_4_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

resmatch -abs=0.001 -header=n -param=y -regions=y -res=1 test.res fit_4_4_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 4.5.0 \n"
printf " Test case: Reference file has different time unit. \n"
printf " Expected result: Result are the same. \n\n"

cp -f rpsim.tac test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd -ovr -move ref.tac test.tac cereb
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacunit -xconv=min ref.tac test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

cp -f ref.tac ref2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
cp -f test.tac test2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=sec ref2.tac test2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"


$PROGRAM test.tac ref.tac 60 test.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test.tac ref2.tac 60 fit_4_5_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

resmatch -abs=0.001 -header=n -param=y -regions=y -res=1 test.res fit_4_5_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


$PROGRAM test2.tac ref2.tac 60 test.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test2.tac ref.tac 60 fit_4_5_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

resmatch -abs=0.001 -header=n -param=y -regions=y -res=1 test.res fit_4_5_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n===================================================================\n"
printf "   Fit time \n"
printf "===================================================================\n"



printf "\n 5.0.0 \n"
printf " Test case: Fit time is set to very large value. \n"
printf " Expected result: Fit time is set to the end of the data. \n\n"

rm -f fit_5_0_0.res

$PROGRAM rpsim.tac cereb 9999 fit_5_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval fit_5_0_0.res "Data range" "0 - 120 min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 5.0.1 \n"
printf " Test case: Fit time is set to zero. \n"
printf " Expected result: Fit time is set to the end of the data. \n\n"

rm -f fit_5_0_1.res

$PROGRAM rpsim.tac cereb 0 fit_5_0_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval fit_5_0_1.res "Data range" "0 - 120 min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 5.0.2 \n"
printf " Test case: Fit time is accidentally set to letter 'O'. \n"
printf " Expected result: Error. \n\n"

rm -f fit_5_0_2.res stdout.txt

$PROGRAM rpsim.tac cereb O fit_5_0_2.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "invalid fit time: 'O'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 5.1.0 \n"
printf " Test case: Fit time is shorter than data range but reasonably long. \n"
printf " Expected result: Fit time is set to the closest time of the data. \n\n"

$PROGRAM rpsim.tac cereb 30 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Data range" "0 - 30 min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM rpsim.tac cereb 31 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Data range" "0 - 30 min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM rpsim.tac cereb 33 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Data range" "0 - 35 min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 5.1.1 \n"
printf " Test case: Fit time would cover less than four frames. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt

$PROGRAM rpsim_fewframes.tac cereb 20 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "too few data points for a decent fit."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Constraints \n"
printf "===================================================================\n"




printf "\n 6.0.0 \n"
printf " Test case: Default constraints can be written in file. \n"
printf " Expected result: Limit file is written when given as the only option. \n\n"

rm -f temp.lim

$PROGRAM -lim=temp.lim
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval temp.lim "R1_lower"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval temp.lim "R1_upper"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval temp.lim "k2_lower"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval temp.lim "k2_upper"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval temp.lim "BP_lower"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval temp.lim "BP_upper"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 6.0.1 \n"
printf " Test case: Default constraints are tried to be written in file but file exists. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt

$PROGRAM -lim=temp.lim &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "parameter constraint file temp.lim exists."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 6.1.0 \n"
printf " Test case: Valid constraint file given. \n"
printf " Expected result: Limits are used. \n\n"

printf "R1_lower := 0.9 \n"  >   test.lim
printf "k2_lower := 0.15 \n" >>  test.lim
printf "BP_lower := 1.2 \n"  >>  test.lim
printf "R1_upper := 1.5 \n"  >>  test.lim
printf "k2_upper := 0.5 \n"  >>  test.lim
printf "BP_upper := 2.9 \n"  >>  test.lim

rm -f fit_6_1_0.res stdout.txt

$PROGRAM -d6 -lim=test.lim rpsim.tac cereb 60 fit_6_1_0.res  &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt "def_pmin[1]" "0.9"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt "def_pmax[1]" "1.5"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt "def_pmin[2]" "0.15"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt "def_pmax[2]" "0.5"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt "def_pmin[3]" "1.2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt "def_pmax[3]" "2.9"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 6.1.1 \n"
printf " Test case 1: A parameter can be fixed using constraint file. \n"
printf " Test case 2: Constraint file does not need to contain limits for all parameters. \n"
printf " Expected result: Limits are used. \n\n"

printf "k2_lower := 0.5 \n"  >   test2.lim
printf "k2_upper := 0.5 \n"  >>  test2.lim

rm -f fit_6_1_1.res stdout.txt

$PROGRAM -d6 -lim=test2.lim rpsim.tac cereb 60 fit_6_1_1.res  &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt "def_pmin[2]" "0.5"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt "def_pmax[2]" "0.5"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

parformat -f=IFT fit_6_1_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval fit_6_1_1.ift "k2" "0.5 1/min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 6.2.0 \n"
printf " Test case: Invalid constraint file given. \n"
printf " Expected result: Error. \n\n"

printf "k2 := 0.5 \n"  >  test.lim

rm -f output.res stdout.txt

$PROGRAM -lim=test.lim rpsim.tac cereb 60 output.res  &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt "Error" "invalid parameter file."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 6.3.0 \n"
printf " Test case: Given constraint file does not exist. \n"
printf " Expected result: Error. \n\n"


rm -f output.res stdout.txt

$PROGRAM -lim=nonexisting.lim rpsim.tac cereb 60 output.res  &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt "Error in reading 'nonexisting.lim'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Weights \n"
printf "===================================================================\n"



printf "\n 7.0.0 \n"
printf " Test case: If file does not contain weights then by default weights are 1\n"
printf " Expected result: correct weights are printed in test mode. \n\n"

rm -f stdout.txt
cp -f rpsim_fewframes.tac test.tac
dftweigh -rm test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -d4 test.tac cereb 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Weighting" "no"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt common_data_weights "1, 1, 1, 1, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 7.0.1 \n"
printf " Test case: Datafile contains weights.\n"
printf " Expected result: correct weights are printed in test mode. \n\n"

dftweigh -wf test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -d4 test.tac cereb 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Weighting" "yes"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt common_data_weights "0.583333, 0.583333, 1.16667, 1.16667, 1.16667, 1.16667, 1.16667"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 7.0.2 \n"
printf " Test case: Datafile contains weights but set to 1 optionally.\n"
printf " Expected result: correct weights are printed in test mode. \n\n"

$PROGRAM -d4 -w1 test.tac cereb 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Weighting" "no"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt common_data_weights "1, 1, 1, 1, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 7.0.3 \n"
printf " Test case: Datafile contains weights but set based on frame lengths optionally.\n"
printf " Expected result: correct weights are printed in test mode. \n\n"

dftweigh test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -d4 -wf test.tac cereb 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Weighting" "yes"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt common_data_weights "0.583333, 0.583333, 1.16667, 1.16667, 1.16667, 1.16667, 1.16667"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 7.0.4 \n"
printf " Test case: Datafile does not contain weights but set based on frame lengths optionally.\n"
printf " Expected result: correct weights are printed in test mode. \n\n"

dftweigh -rm test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -d4 -wf test.tac cereb 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Weighting" "yes"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt common_data_weights "0.583333, 0.583333, 1.16667, 1.16667, 1.16667, 1.16667, 1.16667"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n 8.0.0 \n"
printf " Test case: Bootstrapping with options -sd=y and -cl=y. \n"
printf " Expected result: Result file contains sd and cl for parameters \n\n"

rm -f fit_8_0_0.res

$PROGRAM -sd=y -cl=y rpsim_fewframes.tac cereb 120 fit_8_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat -f=ift fit_8_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval fit_8_0_0.ift "SD"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval fit_8_0_0.ift "95%CL1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval fit_8_0_0.ift "95%CL2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 8.0.1 \n"
printf " Test case: Bootstrapping with options -sd=n and -cl=y. \n"
printf " Expected result: Result file contains cl for parameters \n\n"

rm -f fit_8_0_1.res

$PROGRAM -sd=n -cl=y rpsim_fewframes.tac cereb 120 fit_8_0_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat -f=ift fit_8_0_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval fit_8_0_1.ift "SD"
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval fit_8_0_1.ift "95%CL1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval fit_8_0_1.ift "95%CL2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 8.0.2 \n"
printf " Test case: Bootstrapping with options -sd=y and -cl=n. \n"
printf " Expected result: Result file contains sd for parameters \n\n"

rm -f fit_8_0_2.res

$PROGRAM -sd=y -cl=n rpsim_fewframes.tac cereb 120 fit_8_0_2.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat -f=ift fit_8_0_2.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval fit_8_0_2.ift "SD"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval fit_8_0_2.ift "95%CL1"
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval fit_8_0_2.ift "95%CL2"
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

