#!/bin/bash
#: Title      : test_fit_trtm
#: Date       : 2018-02-10
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../fit_trtm$EXT;

if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n verifying that required data exists \n"

if [ ! -f mpaplasma.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f frames.dat ] || [ ! -f correct_trtm1.res ] || [ ! -f correct_trtm2.res ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


printf "\n making data for testing \n"

if [ ! -f sim1.par ] || [ ! -f refinput.tac ]; then
  printf "# model := SER3TCM\n" > sim1.par
  printf "Parameters,K1,k2,k3,k4,k5,k6,Vb,f,fA\n" >> sim1.par
  printf "putam,0.5,0.1,30.0,0.0,0,0,0,0,0\n"     >> sim1.par
  rm -f trtm_full.tac putamen.tac trtm.tac
  sim_3tcm -nosub sim1.par mpaplasma.dat mpaplasma.dat refinput.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f sim2.par ]; then
  rm -f trtm_full.tac trtm.tac putamen.tac
  printf "# model := TRTM\n" > sim2.par
  printf "Parameters,R1[unitless],k2[1/min],k3[1/min]\n" >> sim2.par
  printf "cortex_1,0.8,0.08,0.01\n"                      >> sim2.par
  printf "cortex_2,0.8,0.08,0.07\n"                      >> sim2.par
  printf "cortex_3,0.8,0.08,0.14\n"                      >> sim2.par
fi

if [ ! -f trtm_full.tac ]; then
  rm -f trtm.tac putamen.tac
  sim_rtcm -nosub sim2.par refinput.tac trtm_full.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd trtm_full.tac refinput.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f trtm.tac ] || [ ! -f trtmnor.tac ] || [ ! -f putamen.tac ]; then
  simframe trtm_full.tac frames.dat trtm.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dftweigh -wf --silent trtm.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr trtmnor.tac trtm.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr -move putamen.tac trtmnor.tac putam
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=y trtm.svg trtm.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi




printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n 1.0.0 \n"
printf " Test case 1: Fit TACs simulated with realistic range of parameters. \n"
printf " Test case 2: Reference inside TTAC file. \n"
printf " Expected result 1: Result file is created, and results are close to correct \n"
printf " Expected result 2: SVG plot is created. \n"
printf " Expected result 3: Fitted TACs should be close to original TACs.\n\n"

rm -f fit_1_0_0.res fit_1_0_0.tac fit_1_0_0.svg

$PROGRAM -fit=fit_1_0_0.tac -svg=fit_1_0_0.svg trtm.tac putam 90 fit_1_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

if [ ! -f fit_1_0_0.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -abs=0.5 -rel=0.5 trtm.tac fit_1_0_0.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

resmatch -abs=0.01 -header=y -param=y -regions=y -res=1-3 correct_trtm1.res fit_1_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case 1: Fit TACs simulated with realistic range of parameters. \n"
printf " Test case 2: Reference in separate TAC file. \n"
printf " Expected result 1: Result file is created, and results are close to correct \n"
printf " Expected result 2: SVG plot is created. \n"
printf " Expected result 3: Fitted TACs should be close to original TACs.\n\n"

rm -f fit_1_1_0.res fit_1_1_0.tac fit_1_1_0.svg

$PROGRAM -fit=fit_1_1_0.tac -svg=fit_1_1_0.svg trtmnor.tac putamen.tac 90 fit_1_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

if [ ! -f fit_1_1_0.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -abs=0.5 -rel=0.5 trtmnor.tac fit_1_1_0.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


resmatch -abs=0.01 -header=y -param=y -regions=y -res=1-3 correct_trtm2.res fit_1_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.0.0 \n"
printf " Test case: Fix rk2 to given value. \n"
printf " Expected result: rk2 in results is exactly as given. \n\n"

rm -f fit_2_0_0.res fit_2_0_0.tac

$PROGRAM -rk2=0.1 -fit=fit_2_0_0.tac trtm.tac putam 90 fit_2_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.5 -rel=0.5 trtm.tac fit_2_0_0.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

resmatch -abs=0.01 -header=y -param=y -regions=y -res=1-3 correct_trtm1.res fit_2_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

parformat -format=IFT fit_2_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval fit_2_0_0.ift rk2 "0.1 1/min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.0.1 \n"
printf " Test case: Fix rk2 to somewhat wrong value. \n"
printf " Expected result: rk2 in results is still exactly as given. \n\n"

rm -f fit_2_0_1.res fit_2_0_1.tac

$PROGRAM -rk2=0.09 -fit=fit_2_0_1.tac trtm.tac putam 60 fit_2_0_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parformat -format=IFT fit_2_0_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval fit_2_0_1.ift rk2 "0.09 1/min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 2.1.0 \n"
printf " Test case: Fix rk2 to regional medium. \n"
printf " Expected result: Result file is created, and results are close to correct \n"

rm -f fit_2_1_0.res fit_2_1_0.tac

$PROGRAM -rk2=median -fit=fit_2_1_0.tac trtm.tac putam 120 fit_2_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.5 -rel=0.5 trtm.tac fit_2_1_0.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

resmatch -abs=0.01 -header=y -param=y -regions=y -res=1-3 correct_trtm1.res fit_2_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.2.0 \n"
printf " Test case: Fix rk2 to regional mean. \n"
printf " Expected result: Result file is created, and results are close to correct \n"

rm -f fit_2_2_0.res fit_2_2_0.tac

$PROGRAM -rk2=mean -fit=fit_2_2_0.tac trtm.tac putam 120 fit_2_2_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.5 -rel=0.5 trtm.tac fit_2_2_0.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

resmatch -abs=0.01 -header=y -param=y -regions=y -res=1-3 correct_trtm1.res fit_2_2_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

