/** @file e7emhdr.c
 *  @brief Edit the ECAT 7 mainheader.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Edit the information in the main header of an ECAT 7 file.",
  " ",
  "Usage: @P [Options] ecatfile fieldname := fieldvalue ",
  "   or: @P [Options] ecatfile headerfile",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Header file can contain one or more mainheader field names and values",
  "in interfile-like format, for example:",
  "  patient_name := Einstein Albert",
  "  radiopharmaceutical := O-15 WATER",
  "  study_description := brain scan",
  "  facility_name := Turku PET Centre",
  " ",
  "Examples:",
  "  @P s2345dy1.v patient_name := \"Albert Einstein\"",
  "  @P s2345dy1.v s2345dy1.header",
  " ",
  "See also: lmhdr, lshdr, e7evhdr, lmlist, imgunit, imgdecay, e63emhdr",
  " ",
  "Keywords: ECAT, image, header, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int                 ai, help=0, version=0, verbose=1;
  int                 li, ret, change_nr=0;
  char               *cptr, petfile[FILENAME_MAX], iftfile[FILENAME_MAX];
  char                keyname[FILENAME_MAX], eqsign[FILENAME_MAX],
                      keyvalue[FILENAME_MAX];
  FILE               *fp;
  ECAT7_mainheader    main_header;
  IFT                 ift;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  petfile[0]=iftfile[0]=(char)0;
  keyname[0]=eqsign[0]=keyvalue[0]=(char)0;
  iftInit(&ift);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option %s\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  /* PET filename */
  if(ai<argc) {
    strlcpy(petfile, argv[ai], FILENAME_MAX); ai++;
    /* check that file exists */
    if(access(petfile, 0) == -1) {
      fprintf(stderr, "Error: file '%s' does not exist.\n", petfile);
      return(1);
    }
  }
  /* If the next argument is also the last, then that is header filename */
  if(ai==argc-1) {
    strlcpy(iftfile, argv[ai], FILENAME_MAX); ai++;
    /* check that file exists */
    if(access(iftfile, 0) == -1) {
      fprintf(stderr, "Error: file '%s' does not exist.\n", iftfile);
      return(1);
    }
  } else {
  /* else we have header key and value */
    if(ai<argc) {strcpy(keyname, argv[ai]); ai++;}
    if(ai<argc) {
      strlcpy(eqsign, argv[ai], FILENAME_MAX); ai++;
      if(strcmp(eqsign, ":=")!=0) { /* check if eqsign is ok */
        fprintf(stderr, "Error: use ' := ' to separate field name and value.\n");
        return(1);
      }
    }
    if(ai<argc) {strlcpy(keyvalue, argv[ai], FILENAME_MAX); ai++;}
    /* if even more arguments, they probably are space-separated values */
    for(; ai<argc; ai++) {strcat(keyvalue, " "); strcat(keyvalue, argv[ai]);}
  }
  
  /* Is something missing? */
  /* Either header filename, or, both keyname and value must be given */
  if((!keyname[0] && !iftfile[0]) || (keyname[0] && !keyvalue[0])) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  /* If available, store the keyname and value */
  if(keyname[0]) iftPut(&ift, keyname, keyvalue, NULL, 0);


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("petfile := %s\n", petfile);
    if(iftfile[0]) printf("iftfile := %s\n", iftfile);
    if(keyname[0]) printf("%s := %s\n", keyname, keyvalue);
  }


  /*
   *  Read key names and values from file, if necessary
   */
  if(iftfile[0]) {
    if(verbose>1) printf("reading %s\n", iftfile);
    ret=iftRead(&ift, iftfile, 1, 0);
    if(ret) {
      fprintf(stderr, "Error: %s\n", ift.status);
      iftEmpty(&ift); return(2);
    }
  }
  if(verbose>2) iftWrite(&ift, "stdout", 0);


  /*
   *  Read main header
   */
  /* Open input ECAT file */
  if(verbose>1) printf("reading %s\n", petfile);
  if((fp=fopen(petfile, "rb")) == NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", petfile); 
    iftEmpty(&ift); return(2);
  }
  /* Read the header */
  if((ret=ecat7ReadMainheader(fp, &main_header))) {
    fprintf(stderr, "Error (%d): cannot read main header.\n", ret);
    iftEmpty(&ift); fclose(fp); return(3);
  }
  /* Close ECAT file */
  fclose(fp);
  if(verbose>9) ecat7PrintMainheader(&main_header, stdout);
  /* Check file format */
  if(strncmp(main_header.magic_number, ECAT7V_MAGICNR, 7)!=0) {
    if(verbose>0) 
      fprintf(stderr, "Warning: %s is not an ECAT 7.x file.\n", petfile);
    //iftEmpty(&ift); return(3);
    // do not give error, otherwise we cannot edit empty header
  }


  /*
   *  Change the fields
   */
  for(li=0, change_nr=0; li<ift.keyNr; li++) {
    if(verbose>1) iftWriteItem(&ift, li, stdout, 0);
    /* Change field value */
    ret=ecat7EditMHeader(&main_header, ift.item[li].key, ift.item[li].value, verbose-2);
    if(ret==1) {
      fprintf(stderr, "  warning: invalid field name: %s\n", ift.item[li].key);
    } else if(ret==2) {
      fprintf(stderr, "  warning: invalid field value: %s\n", ift.item[li].value);
    } else change_nr++;
  } /* next field */
  if(verbose>4) ecat7PrintMainheader(&main_header, stdout);
  iftEmpty(&ift);
  if(change_nr==0) {
    fprintf(stderr, "Error: no fields were changed.\n"); 
    return(8);
  }


  /*
   *  Write main header
   */
  if(verbose>1) printf("writing edited %s\n", petfile);
  /* Open input CTI file */
  if((fp=fopen(petfile, "r+b")) == NULL) {
    fprintf(stderr, "Error: cannot open %s for write.\n", petfile); 
    return(11);
  }
  /* Write header */
  if((ret=ecat7WriteMainheader(fp, &main_header))) {
    fprintf(stderr, "Error (%d): cannot write main header.\n", ret);
    fclose(fp); return(12);
  }
  /* Close ECAT file */
  fclose(fp);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
