/** @file e7evhdr.c
 *  @brief Edit the ECAT 7 image volume header.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
//#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Edit the information in the ECAT 7.x image volume subheaders.",
  " ",
  "Usage: @P [options] ecatfile headerfile",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Header file can contain one or more subheader field names and values",
  "in interfile-like format, for example:",
  "  x_pixel_size := 0.257426",
  "  y_pixel_size := 0.257426",
  "  z_pixel_size := 0.2425",
  " ",
  "See also: lshdr, e7emhdr, e7eshdr, lmlist, lmhdr, eframe",
  " ",
  "Keywords: ECAT, image, header, tool, IFT",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int                     ai, help=0, version=0, verbose=1;
  int                     li, mi, ret;
  char                   *cptr, petfile[FILENAME_MAX], iftfile[FILENAME_MAX];
  IFT                     ift;
  ECAT7_mainheader        main_header;
  ECAT7_imageheader       image_header;
  static ECAT7_MATRIXLIST mlist;
  FILE                   *fp;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  petfile[0]=iftfile[0]=(char)0;
  iftInit(&ift);
  ecat7InitMatlist(&mlist);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option %s\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!petfile[0]) {strcpy(petfile, argv[ai]); continue;}
    if(!iftfile[0]) {strcpy(iftfile, argv[ai]); continue;}
    fprintf(stderr, "Error: invalid argument '%s'\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!iftfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("petfile := %s\n", petfile);
    printf("iftfile := %s\n", iftfile);
  }


  /*
   *  Read key names and values from file, if necessary
   */
  if(verbose>1) printf("reading %s\n", iftfile);
  ret=iftRead(&ift, iftfile, 1, 0);
  if(ret) {
    fprintf(stderr, "Error: %s\n", ift.status);
    iftEmpty(&ift); return(2);
  }
  if(verbose>9) iftWrite(&ift, "stdout", 0);


  /*
   *  Open ECAT volume file
   */
  if(verbose) printf("opening %s\n", petfile);
  if((fp=fopen(petfile, "r+b")) == NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", petfile); 
    iftEmpty(&ift); return(3);
  }
  /* Read main header */
  if((ret=ecat7ReadMainheader(fp, &main_header))) {
    fprintf(stderr, "Error (%d): cannot read main header.\n", ret);
    iftEmpty(&ift); fclose(fp); return(3);
  }
  if(verbose>9) ecat7PrintMainheader(&main_header, stdout);
  /* Check file format */
  if(strncmp(main_header.magic_number, ECAT7V_MAGICNR, 7)!=0) {
    fprintf(stderr, "Error: %s is not an ECAT 7.x file.\n", petfile);
    iftEmpty(&ift); fclose(fp); return(3);
  }
  /* Check the file type */
  if(main_header.file_type!=ECAT7_VOLUME8 &&
     main_header.file_type!=ECAT7_VOLUME16) {
    fprintf(stderr, "Error: filetype %s is not supported.\n",
      ecat7filetype(main_header.file_type) );
    iftEmpty(&ift); fclose(fp); return(3);
  }

  /*
   *  Read matrix list
   */
  if(verbose>1) printf("reading matrix list\n");
  ret=ecat7ReadMatlist(fp, &mlist, verbose-1);
  if(ret) {
    fprintf(stderr, "Error (%d): cannot read matrix list.\n", ret);
    iftEmpty(&ift); fclose(fp); return(4);
  }
  if(mlist.matrixNr<=0) {
    fprintf(stderr, "Error: matrix list is empty.\n");
    iftEmpty(&ift); fclose(fp); return(4);
  }
  if(verbose>8) ecat7PrintMatlist(&mlist);


  /*
   *  Edit subheaders one at a time
   */
  if(verbose>1) printf("editing matrices\n");
  for(mi=0; mi<mlist.matrixNr; mi++) {
    /* Read subheader */
    if(verbose>2) printf("reading subheader from matrix %d\n", mi+1);
    ret=ecat7ReadImageheader(fp, mlist.matdir[mi].strtblk, &image_header);
    if(ret) {
      fprintf(stderr, "Error %d in reading subheader.\n", ret);
      iftEmpty(&ift); ecat7EmptyMatlist(&mlist); fclose(fp);
      return(5);
    }
    /* Change requested header fields */
    for(li=0; li<ift.keyNr; li++) {
      ret=ecat7EditVHeader(&image_header, ift.item[li].key, ift.item[li].value,
                           verbose-2);
      if(ret==1) 
        fprintf(stderr, "Error: invalid field name: %s\n", ift.item[li].key);
      else if(ret==2) 
        fprintf(stderr, "Error: invalid field value: %s\n", ift.item[li].value);
      if(ret==1 || ret==2) {
        iftEmpty(&ift); ecat7EmptyMatlist(&mlist); fclose(fp);
        return(6);
      }
    } // next field
    if(verbose>5) ecat7PrintImageheader(&image_header, stdout);
    /* Write subheader */
    if(verbose>2) printf("writing header\n");
    ret=ecat7WriteImageheader(fp, mlist.matdir[mi].strtblk, &image_header);
    if(ret) {
      fprintf(stderr, "Error %d in writing subheader.\n", ret);
      iftEmpty(&ift); ecat7EmptyMatlist(&mlist); fclose(fp);
      return(11);
    }
  } /* next matrix */
  if(verbose>2) printf("all matrices processed.\n");

  /*
   *  Close ECAT file
   */
  fclose(fp);

  /*
   *  Free memory
   */
  iftEmpty(&ift); ecat7EmptyMatlist(&mlist);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
