/** @file egetstrt.c
 *  @brief Get the scan_start_time in ECAT files.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Print the scan_start_time in ECAT 6 or 7 files on screen (stdout).",
  " ",
  "Usage: @P [Options] ecatfile(s)",
  " ",
  "Options:",
  " -o=<output filename>",
  "     Output is written in given file; any existing file is overwritten.",
  " -dif[=<min|sec>]",
  "     Instead of scan start times, program lists the difference to the first",
  "     one, either in seconds (sec) or in minutes (min); minutes by default.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: lmhdr, ecattime, esetstrt, tactime, injdifft, eframe",
  " ",
  "Keywords: image, ECAT, input, tool, scan start time",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int                 ai, help=0, version=0, verbose=1;
  int                 ret, fileNr=0;
//int                 fformat=63;
  int                 is_diff=0;
  int                 time_unit=TUNIT_MIN;
  char                ecatfile[FILENAME_MAX], outfile[FILENAME_MAX],
                      tmp[32], *cptr;
  FILE               *fp, *ofp;
  ECAT63_mainheader   e63mhdr;
  ECAT7_mainheader    e7mhdr;
  struct tm          *stp, tm, stm;
  time_t              timet1=0, timet2;
  double              time_diff_sec=0;
  int                 firstfile=0;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  ecatfile[0]=outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1;
    if(strncasecmp(cptr, "DIF=MIN", 5)==0) {
      time_unit=TUNIT_MIN; is_diff=1; continue;
    } else if(strncasecmp(cptr, "DIF=SEC", 5)==0) {
      time_unit=TUNIT_SEC; is_diff=1; continue;
    } else if(strcasecmp(cptr, "DIF")==0) {
      time_unit=TUNIT_MIN; is_diff=1; continue;
    } else if(strncasecmp(cptr, "D=MIN", 3)==0) {
      time_unit=TUNIT_MIN; is_diff=1; continue;
    } else if(strncasecmp(cptr, "D=SEC", 3)==0) {
      time_unit=TUNIT_SEC; is_diff=1; continue;
    } else if(strncasecmp(cptr, "O=", 2)==0) {
      strcpy(outfile, cptr+2); if(strlen(outfile)>0) continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(firstfile<1) firstfile=ai;
    /* check that file exists */
    if(access(argv[ai], 0) == -1) {
      fprintf(stderr, "Error: file '%s' does not exist.\n", argv[ai]);
      return(1);
    }
    fileNr++;
  }

  /* Is something missing? */
  if(fileNr<1) {
    fprintf(stderr, "Error: missing command-line argument; try %s --help\n",
      argv[0]);
    return(1);
  }
  /* Time difference option is not valid with just one file */
  if(fileNr==1 && is_diff) is_diff=0;

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("fileNr := %d\n", fileNr);
    printf("outfile := %s\n", outfile);
    printf("is_diff := %d\n", is_diff);
    printf("time_unit := %d\n", time_unit);
  }
  

  /*
   *  Open output file if necessary
   */
  if(outfile[0]) {
    if(access(outfile, 0) != -1)
      fprintf(stderr, "Warning: %s is overwritten.\n", outfile);
    ofp=fopen(outfile, "w"); if(ofp==NULL) {
      fprintf(stderr, "Error: cannot open file %s for writing.\n", outfile);
      return(2);
    }
  } else {
    ofp=stdout;
  }

  /*
   *  Process one file at a time
   */
  for(ai=firstfile, fileNr=0; ai<argc; ai++) {
    strcpy(ecatfile, argv[ai]);
    if(verbose>0) fprintf(ofp, "filename := %s\n", ecatfile);

    /* Open file */
    if((fp=fopen(ecatfile, "rb")) == NULL) {
      fprintf(stderr, "Error: cannot open file %s\n", ecatfile);
      if(outfile[0]) fclose(ofp); 
      return(3);
    }

    /* Try to read ECAT 7.x main header */
    ret=ecat7ReadMainheader(fp, &e7mhdr);
    if(ret) {
      fprintf(stderr, "Error (%d): cannot read main header.\n", ret);
      if(outfile[0]) fclose(ofp); 
      fclose(fp); return(3);
    }

    /* If header could be read, check for magic number */
    if(strncmp(e7mhdr.magic_number, ECAT7V_MAGICNR, 7)==0) {
      /* This is ECAT 7.x file */
      //fformat=7;
      /* All file types are supported */
      if(verbose>4) printf("  identified as an ECAT 7 file.\n");
      if(verbose>9) ecat7PrintMainheader(&e7mhdr, stdout);
      /* Get the scan_start_time */
      timet2=e7mhdr.scan_start_time; if(fileNr==0) timet1=timet2;
      if(timet2<86400 || !gmtime_r(&timet2, &tm)) {
        fprintf(stderr, "Error: %s does not contain scan start time.\n", 
                ecatfile);
        if(outfile[0]) fclose(ofp); 
        fclose(fp); return(4);
      }
      stp=&tm;
    } else { /* Try to read as ECAT 6.3 file */
      if((ret=ecat63ReadMainheader(fp, &e63mhdr))) {
        fprintf(stderr, "Error (%d): cannot read main header.\n", ret);
        if(outfile[0]) fclose(ofp); 
        fclose(fp); return(2);
      }
      if(verbose>9) ecat63PrintMainheader(&e63mhdr, stdout);
      /* This is ECAT 6.3 file */
      //fformat=63;
      if(verbose>4) printf("  identified as an ECAT 6.3 file.\n");
      if(verbose>1) {
        char tmp[20]; fprintf(stdout, "Scan start: %s\n", 
                              ecat63ScanstarttimeInt(&e63mhdr, tmp));
      }
      /* Get the scan_start_time */
      ecat63ScanstarttimeToTm(&e63mhdr, &stm); 
      timet2=timegm(&stm);
      if(timet2==-1 || timet2<86400) {
        fprintf(stderr, "Error: %s does not contain scan start time.\n", 
                ecatfile);
        if(outfile[0]) fclose(ofp); 
        fclose(fp); return(2);
      }
      if(fileNr==0) timet1=timet2;
      stp=&stm;
    }
    /* Close file */
    fclose(fp);

    /* Print either start time or time difference */
    if(is_diff==0 || fileNr==0) {
      /* Print the time */
      //strftime(tmp, 32, "%Y-%m-%d %T", stp); THIS DOES NOT WORK IN MINGW!!!
      strftime(tmp, 32, "%Y-%m-%d %H:%M:%S", stp);
      fprintf(ofp, "scan_start_time := %s\n", tmp);
    } else if(fileNr>0) {
      time_diff_sec=difftime(timet2, timet1);
      if(time_unit==TUNIT_SEC) 
        fprintf(ofp, "scan_start_time_difference := %g\n", time_diff_sec);
      else 
        fprintf(ofp, "scan_start_time_difference := %g\n", time_diff_sec/60.0);
    }

    fileNr++;
  } /* next file */

  if(outfile[0]) fclose(ofp);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
