/** @file img2scn.c
    @brief Make 2D sinogram from PET image in ECAT format.
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
    @details Based on programs written by Sakari Alenius for
    Sun UNIX workstations.
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <dirent.h>
#include <sys/stat.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcrec.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Make 2D sinogram from PET image in CTI ECAT 931 format for program testing",
  "and simulations.",
  "Image plane number is used to guess the scanner and to set sinogram size.",
  "This program cannot remove calibrations, or normalization, attenuation,",
  "or dead-time correction. Decay correction is not removed.",
  " ",
  "Usage: @P [Options] imgfile scnfile",
  " ",
  "Options:",
//" -dtc=<Y|n>",
//"     Dead-time correction is done (y, default), or not done (n).",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "  @P a2345dy1.img a2345dy1.scn",
  " ",
  "See also: ecatfbp, ecatmrp, eframe, edecay, ecalibr, flat2img, dft2img",
  " ",
  "Keywords: ECAT, sinogram, reconstruction, software testing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int                 ai, help=0, version=0, verbose=1;
  char                imgfile[FILENAME_MAX], scnfile[FILENAME_MAX];
  char               *cptr;
  int                 ret;



  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=scnfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(imgfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {strlcpy(scnfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {
    fprintf(stderr, "Error: too many arguments.\n");
    return(1);
  }

  /* Is something missing? */
  if(!scnfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("imgfile := %s\n", imgfile);
    printf("scnfile := %s\n", scnfile);
  }

  /* Check that input and output files do not have the same filename */
  if(strcasecmp(imgfile, scnfile)==0) {
    fprintf(stderr, "Error: original image must not be overwritten.\n");
    return(1);
  }


  /*
   *  Read PET image
   */
  if(verbose>0) printf("reading image %s\n", imgfile);
  IMG img; imgInit(&img);
  ret=imgRead(imgfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg);
    if(verbose>1) printf("ret := %d\n", ret);
    return(2);
  }
  if(verbose>1) { 
    printf("dimx := %d\n", img.dimx);
    printf("dimy := %d\n", img.dimy);
    printf("dimz := %d\n", img.dimz);
    printf("dimt := %d\n", img.dimt);
  }

  /*
   *  Reprojection
   */
  if(verbose>0) printf("reprojecting...\n");
  IMG scn; imgInit(&scn);
  ret=imgReprojection(&img, &scn, verbose-2);
  if(ret) {
    fprintf(stderr, "Error: cannot do reprojection.\n");
    if(verbose>1) printf("ret := %d\n", ret);
    imgEmpty(&scn); imgEmpty(&img);
    return(3);
  }

  /* Save the sinogram */
  if(verbose>0) printf("writing sinogram %s\n", scnfile);
  if(imgWrite(scnfile, &scn)) {
    fprintf(stderr, "Error: %s\n", scn.statmsg);
    imgEmpty(&img); imgEmpty(&scn);
    return(11);
  }

  imgEmpty(&img); imgEmpty(&scn);
  if(verbose>0) fprintf(stdout, "done.\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
