/** @file imgfrdif.c
 *  @brief Sum the difference between frame values in dynamic PET image.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Compute the pixel value difference between consecutive frames to get a rough",
  "estimate on variance on each pixel.",
  "Program writes an image containing sum of |frame[i+1]-frame[i]|.",
  " ",
  "Usage: @P [Options] imgfile difimgfile",
  " ",
  "Options:",
  " -norm[alize]",
  "     Differences are divided by the average of frames,",
  "     |frame[i+1]+frame[i]|/2.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: imgpeak, imgmask, imgthrs, imgfrsmo, imgdelfr, imgledif, img2tif",
  " ",
  "Keywords: image, dynamics, noise, time, mask",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int     ai, help=0, version=0, verbose=1;
  char    imgfile[FILENAME_MAX], outfile[FILENAME_MAX];
  int     dif_normalized=0;
  int     ret;
  char   *cptr; 


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strncasecmp(cptr, "NORMALIZED", 4)==0) {
      dif_normalized=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}
  
  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(imgfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(outfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Is something missing or wrong? */
  if(!outfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("imgfile := %s\n", imgfile);
    printf("outfile := %s\n", outfile);
    printf("dif_normalized := %d\n", dif_normalized);
    fflush(stdout);
  }


  /*
   *  Read dynamic image
   */
  if(verbose>0) {printf("reading dynamic image %s\n", imgfile); fflush(stdout);}
  IMG img; imgInit(&img);
  ret=imgRead(imgfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg);
    if(verbose>1) printf("ret := %d\n", ret);
    return(2);
  }
  if(img.dimt<2) {
    fprintf(stderr, "Error: %s contains only 1 time frame.\n", imgfile);
    imgEmpty(&img); return(2);
  }
  if(verbose>0) {
    fprintf(stdout, "  image contains %d frames and %d planes.\n", img.dimt, img.dimz);
    fflush(stdout);
  }


  /*
   *  Calculate the difference image
   */
  if(verbose>0) {printf("computing frame differences\n"); fflush(stdout);}
  IMG dimg; imgInit(&dimg);
  IMG mimg; imgInit(&mimg);
  if(dif_normalized==0) ret=imgGetFrameDiff(&img, &dimg, NULL, verbose-1);
  else ret=imgGetFrameDiff(&img, &dimg, &mimg, verbose-1);
  imgEmpty(&img);
  if(ret) {
    fprintf(stderr, "Error: cannot compute frame differences.\n");
    imgEmpty(&dimg); imgEmpty(&mimg);
    return(4);
  }
  /* Normalize */
  if(dif_normalized) {
    if(verbose>1) {printf("normalizing difference image\n"); fflush(stdout);}
    int zi, yi, xi;
    for(zi=0; zi<dimg.dimz; zi++)
      for(yi=0; yi<dimg.dimy; yi++)
        for(xi=0; xi<dimg.dimx; xi++) {
          if(mimg.m[zi][yi][xi][0]>1.0E-10)
            dimg.m[zi][yi][xi][0]/=mimg.m[zi][yi][xi][0];
          else
            dimg.m[zi][yi][xi][0]=0.0;
        }
  }

  /*
   *  Save result image
   */
  if(verbose>1) {printf("writing result image\n"); fflush(stdout);}
  ret=imgWrite(outfile, &dimg);
  if(ret) {
    fprintf(stderr, "Error: %s\n", mimg.statmsg);
    imgEmpty(&dimg); imgEmpty(&mimg);
    return(11);
  }
  if(verbose>0) fprintf(stdout, "Image %s saved.\n", outfile);

  imgEmpty(&dimg); imgEmpty(&mimg);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
