/** @file imgmlist.c
 *  @brief List image matrices with their min, max, and mean pixel values.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "List image matrices with their min, max, and mean pixel values.",
  "Image file in ECAT 6.3 or 7.x, NIfTI-1, and Analyze 7.5 format is accepted.",
  " ",
  "Usage: @P [Options] imgfile",
  " ",
  "Options:",
  " -xyz | -xy",
  "     List x,y,z image volumes (default), or x,y image planes.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: imgmax, imgmaxp, imghead, lmlist, img2flat",
  " ",
  "Keywords: image, max, min, matrixlist",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int     ai, help=0, version=0, verbose=1;
  int     ret;
  char    imgfile[FILENAME_MAX], *cptr=NULL;
  int     listVol=1;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strcasecmp(cptr, "XYZ")==0) {
      listVol=1; continue;
    } else if(strcasecmp(cptr, "XY")==0) {
      listVol=0; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  
  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(imgfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Did we get all the information that we need? */
  if(!imgfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("imgfile := %s\n", imgfile);
    printf("listVol := %d\n", listVol);
  }

  if(verbose>2) IMG_TEST=ECAT63_TEST=ECAT7_TEST=verbose-2;
  else IMG_TEST=ECAT63_TEST=ECAT7_TEST=0;


  /*
   *  Read the image file
   */
  if(verbose>1) fprintf(stdout, "reading %s\n", imgfile);
  IMG img; imgInit(&img);
  ret=imgRead(imgfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg);
    if(verbose>1) printf("ret := %d\n", ret);
    return(2);
  }
  /* Check if PET data is raw or image */
  if(img.type!=IMG_TYPE_IMAGE) {
    fprintf(stderr, "Error: %s is not an image.\n", imgfile);
    imgEmpty(&img); return(2);
  }
  int dimt, dimz, dimy, dimx;
  dimt=img.dimt; dimz=img.dimz; dimy=img.dimy; dimx=img.dimx;
  if(verbose>1) fprintf(stdout, "  dim[x,y,z,t] := %d, %d, %d, %d\n", dimx, dimy, dimz, dimt);


  /*
   *  List the contents
   */
  if(listVol==1) {
    fprintf(stdout, "Frame\tMinValue\tMaxValue\tMeanValue\tValidN\tInvalidN\n"); fflush(stdout);
    for(int t=0; t<dimt; t++) {
      float mmax=nan(""), mmin=nan(""), msum=0.0;
      int n=0, in=0;
      for(int z=0; z<dimz; z++)
        for(int y=0; y<dimy; y++)
          for(int x=0; x<dimx; x++) {
            if(!isfinite(img.m[z][y][x][t])) {in++; continue;}
            if(!isfinite(mmax)) mmax=img.m[z][y][x][t];
            if(!isfinite(mmin)) mmin=img.m[z][y][x][t];
            if(img.m[z][y][x][t]>mmax) mmax=img.m[z][y][x][t];
            if(img.m[z][y][x][t]<mmin) mmin=img.m[z][y][x][t];
            msum+=img.m[z][y][x][t];
            n++;
          }
      fprintf(stdout, "%d\t%.3e\t%.3e\t%.3e\t%d\t%d\n", 1+t, mmin, mmax, msum/(float)n, n, in);
      fflush(stdout);
    }
  } else {
    fprintf(stdout, "Frame\tPlane\tMinValue\tMaxValue\tMeanValue\tValidN\nInvalidN\n"); fflush(stdout);
    for(int t=0; t<dimt; t++)
      for(int z=0; z<dimz; z++) {
        float mmax=nan(""), mmin=nan(""), msum=0.0;
        int n=0, in=0;
        for(int y=0; y<dimy; y++)
          for(int x=0; x<dimx; x++) {
            if(!isfinite(img.m[z][y][x][t])) {
              if(verbose>1) printf("pixel[%d][%d][%d][%d] := %g\n", z, y, x, t, img.m[z][y][x][t]);
              in++; continue;
            }
            if(!isfinite(mmax)) mmax=img.m[z][y][x][t];
            if(!isfinite(mmin)) mmin=img.m[z][y][x][t];
            if(img.m[z][y][x][t]>mmax) mmax=img.m[z][y][x][t];
            if(img.m[z][y][x][t]<mmin) mmin=img.m[z][y][x][t];
            msum+=img.m[z][y][x][t];
            n++;
          }
        fprintf(stdout, "%d\t%d\t%.3e\t%.3e\t%.3e\t%d\t%d\n", 1+t, 1+z, mmin, mmax, msum/(float)n, n, in);
        fflush(stdout);
      }
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
