/** @file imgunit.c
 *  @brief Set or get calibration unit in PET image(s).
 *  @details Application name was previously ecatunit.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 *  @test Test other image formats than just ECAT, as far as they support units. 
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
#include <float.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Report or set data calibration units in PET images.",
  "By default, the calibration units are only shown, not changed.",
  "Notice that some image formats, like Analyze, do not contain information",
  "on the calibration units.",
  "Time frame units are determined by the image format.",
  " ",
  "Usage: @P [Options] file(s)",
  " ",
  "Options:",
  " -u=<New unit; e.g. Bq/cc or kBq/ml>",
  "     Set the unit, but does NOT change the pixel values.",
  " -us=<New unit; e.g. Bq/cc or kBq/ml>",
  "     Set the unit only if unit is not originally defined in the image.",
  "     This does NOT change the pixel values.",
  " -uc=<New unit; e.g. Bq/cc or kBq/ml>",
  "     Converts pixel values to the specified unit.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "The following unit conversions are tested (but also others may work):",
  "  Bq/cc  -> kBq/ml",
  "  nCi/ml -> kBq/ml",
  "  MBq/ml -> kBq/ml",
  "  kBq/ml -> Bq/cc",
  " ",
  "Example: convert the pixel values in ECAT images to Bq/mL",
  "  @P -uc=Bq/ml *.v",
  " ",
  "See also: tacunit, lmhdr, lshdr, e7emhdr, imgmax, imgcalc, imglkup",
  " ",
  "Keywords: image, ECAT, unit, tool, simulation",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int       ai, help=0, version=0, verbose=1;
  int       ret, fileNr=0, ffi=0;
  int       ifUnknown=0;
  int       convertUnit=0;
  char     *cptr, petfile[FILENAME_MAX], *newunit=NULL, imgunit[128];
  IMG       img;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  petfile[0]=(char)0; strcpy(imgunit, "");
  imgInit(&img);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1;
    if(strncasecmp(cptr, "US=", 3)==0 && newunit==NULL) {
      cptr+=3; ifUnknown=1;
      newunit=cptr;
      if(strlen(newunit)>12) {
        fprintf(stderr, "Error: unit '%s' too long (>12).\n", newunit);
        return(1);
      }
      if(strlen(newunit)>0) continue;
    } else if(strncasecmp(cptr, "UC=", 3)==0 && newunit==NULL) {
      cptr+=3; convertUnit=1;
      newunit=cptr;
      if(strlen(newunit)>12) {
        fprintf(stderr, "Error: unit '%s' too long (>12).\n", newunit);
        return(1);
      }
      if(strlen(newunit)>0) continue;
    } else if(strncasecmp(cptr, "U=", 2)==0 && newunit==NULL) {
      cptr+=2; ifUnknown=0;
      newunit=cptr;
      if(strlen(newunit)>12) {
        fprintf(stderr, "Error: unit '%s' too long (>12).\n", newunit);
        return(1);
      }
      if(strlen(newunit)>0) continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(ffi<1) ffi=ai; // save the position of first file
    fileNr++;
  }

  /* Is something missing? */
  if(fileNr<1) {
    fprintf(stderr, "Error: missing command-line argument; try %s --help\n",
      argv[0]);
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("imgunit := %s\n", imgunit);
    printf("fileNr := %d\n", fileNr);
    printf("ifUnknown := %d\n", ifUnknown);
    printf("convertUnit := %d\n", convertUnit);
  }


  /*
   *  Do one file at a time
   */
  fileNr=0; // increase when successfully processed
  for(ai=ffi; ai<argc; ai++) {

    strcpy(petfile, argv[ai]);

    /* Read image file */
    if(verbose>1) printf("reading %s\n", petfile);
    ret=imgRead(petfile, &img);
    if(ret) {
      fprintf(stderr, "Warning: %s\n", img.statmsg); 
      if(verbose>2) imgInfo(&img);
      continue;
    }
    if(verbose>9) imgInfo(&img);

    /* Get the current image unit */
    strcpy(imgunit, imgUnit(img.unit));
    if(verbose>2) printf("existing calibration unit := %d\n", img.unit);

    if(newunit==NULL) {
      if(verbose>2) printf("print the unit and do nothing else, as requested\n");
      fprintf(stdout, "# filename := %s\n", petfile);
      fprintf(stdout, "unit := %s\n", imgunit);
      imgEmpty(&img);
      fileNr++;
      continue;
    }

    if(ifUnknown!=0 && img.unit!=0 && convertUnit==0) {
      if(verbose>2) printf("unit exists already; do nothing, as requested\n");
      if(verbose>0)
        fprintf(stderr, "%s : unit '%s' is not replaced\n",
          petfile, imgunit);
      fileNr++;
      imgEmpty(&img); continue;
    }


    if(convertUnit!=0) {
      if(verbose>2) printf("convert concentrations to new units\n");
      /* Check that there is an old unit */
      if(img.unit==0) {
        fprintf(stderr, "%s : cannot convert unknown unit.\n", petfile);
        imgEmpty(&img); continue;
      }
      /* Try to convert it */
      if(img.unit==imgUnitId(newunit)) {
        if(verbose>0)
          fprintf(stderr, "%s: unit is already %s\n", petfile, newunit);
        imgEmpty(&img); fileNr++; 
        continue;
      }
      ret=imgConvertUnit(&img, newunit);
      if(ret==0) {
        if(verbose>0)
          fprintf(stdout, "%s : unit '%s' converted to '%s'\n",
            petfile, imgunit, newunit);
        fileNr++;
      } else {
        fprintf(stderr, "%s: warning: cannot convert unit %s to %s\n",
          petfile, imgunit, newunit);
        imgEmpty(&img); continue;
      }
    } else {
      if(verbose>2) printf("set the unit but keep the concentrations\n");
      ret=imgSetUnit(&img, newunit);
      if(ret==0) {
        if(verbose>0)
          fprintf(stdout, "%s : unit '%s' replaced by '%s'\n",
            petfile, imgunit, newunit);
        fileNr++;
      } else {
        fprintf(stderr, "%s: warning: cannot set unit.\n", petfile);
        imgEmpty(&img); continue;
      }
    }

    /* Save the modified image */
    if(verbose>0) printf("Writing image file %s\n", petfile);
    if(imgWrite(petfile, &img)) {
      fprintf(stderr, "Error: %s\n", img.statmsg);
      imgEmpty(&img); return(11);
    }
    imgEmpty(&img);

  } /* next file */
  if(fileNr==0) {
    fprintf(stderr, "Error: no file(s) could be processed.\n");
    return(3);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
