/** @file imgaumc.c
 *  @brief Calculate area under moment curve (AUMC) for every pixel in dynamic image.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Calculate area under moment curve (AUMC) for every pixel in dynamic image.",
  "Data is not extrapolated, and AUMC is therefore not complete.",
  " ",
  "Usage: @P [Options] imgfile outputfile",
  " ",
  "Image file can be in ECAT 6.3 or 7.x, NIfTI-1, or Analyze 7.5 format.",
  "If data units are available, output units will be (Bq/mL)*s^2, if possible.",
  " ",
  "Options:",
  " -MRT",
  "     Instead of AUMC, AUMC/AUC (MRT) is calculated, with units s.",
  " -log",
  "     Result will be natural log transformed.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: imgledif, imginteg, imgpeak, imginv, imgthrs",
  " ",
  "Keywords: image, peak, time, AUC",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  char  imgfile[FILENAME_MAX], outfile[FILENAME_MAX];
  int   log_transform=0;
  int   mode=0; // 0=AUMC, 1=MRT


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=outfile[0]=(char)0; 
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strcasecmp(cptr, "LOG")==0) {
      log_transform=1; continue;
    } else if(strcasecmp(cptr, "MRT")==0) {
      mode=1; continue;
    } else if(strcasecmp(cptr, "AUMC")==0) {
      mode=0; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(imgfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(outfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Did we get all the information that we need? */
  if(!outfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }


  /* In verbose mode print options */
  if(verbose>1) {
    printf("imgfile := %s\n", imgfile);
    printf("outfile := %s\n", outfile);
    printf("mode := %d\n", mode);
    printf("log_transform := %d\n", log_transform);
  }


  /*
   *  Read PET image
   */
  if(verbose>0) {printf("reading %s\n", imgfile); fflush(stdout);}
  IMG img; imgInit(&img);
  if(imgRead(imgfile, &img)) {
    fprintf(stderr, "Error: %s\n", img.statmsg);
    return(2);
  }
  if(verbose>1) {
    printf("dimt := %d\n", img.dimt);
    printf("dimx := %d\n", img.dimx);
    printf("dimy := %d\n", img.dimy);
    printf("dimz := %d\n", img.dimz);
    printf("unit := %s\n", imgUnit(img.unit));
  }
  if(img.dimt<2) {
    fprintf(stderr, "Error: dynamic image required.\n");
    imgEmpty(&img); return(3);
  }
  if(!imgExistentTimes(&img)) {
    fprintf(stderr, "Error: unknown frame times.\n");
    imgEmpty(&img); return(3);
  }
  /* Make sure that there is no overlap in image frames */ 
  {
    if(verbose>0) printf("checking frame overlap\n");
    int ret=imgFramesCheck(&img, verbose-2);
    if(ret==1 || ret==3) fprintf(stderr, "Warning: overlapping time frames.\n");
    if(ret==2 || ret==3) fprintf(stderr, "Warning: gaps between time frames.\n");
    /* Fix large gaps */
    if(imgFrameGapFill(&img, verbose-2)) {
      fprintf(stderr, "Error: cannot fix time frame gap.\n"); fflush(stderr);
      imgEmpty(&img); return(3);
    }
    /* Fix overlaps */
    if(imgDeleteFrameOverlap(&img)) {
      fprintf(stderr, "Error: cannot fix time frame overlap.\n"); fflush(stderr);
      imgEmpty(&img); return(3);
    }
    if(verbose>4) {
      fprintf(stdout, "Frame times (sec):\n");
       for(int i=0; i<img.dimt; i++)
         fprintf(stdout, "  %e %e %e\n", img.start[i], img.end[i], img.mid[i]);
    }
  }

  /* Try to convert to Bq/mL */
  if(imgConvertUnit(&img, "Bq/mL")!=0) {
    fprintf(stderr, "Warning: cannot convert units.\n"); fflush(stderr);
  }


  /*
   *  Process
   */
  if(verbose>0) printf("processing...\n");
  IMG oimg; imgInit(&oimg);
  if(mode==0) {
    if(imgAUMC(&img, &oimg, verbose-1)) {
      fprintf(stderr, "Error: cannot calculate AUMC.\n");
      imgEmpty(&img); return(6);
    }
  } else {
    if(imgMRT(&img, &oimg, verbose-1)) {
      fprintf(stderr, "Error: cannot calculate MRT.\n");
      imgEmpty(&img); return(6);
    }
  }
  imgEmpty(&img);
  if(log_transform) {
    if(verbose>0) printf("log transform...\n");
    imgLn(&oimg);
  }


  /*
   *  Save the modified image
   */
  if(verbose>0) printf("writing image %s\n", outfile);
  if(imgWrite(outfile, &oimg)) {
    fprintf(stderr, "Error: %s\n", oimg.statmsg);
    imgEmpty(&oimg); return(11);
  }
  imgEmpty(&oimg);
  if(verbose>0) printf("done.\n\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
