/** @file imgftac.c
    @brief Find those voxels in dynamic PET image which correlate best with
     user-specified TAC.
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcmodel.h"
#include "libtpccurveio.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
#include "libtpcmodext.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Find those voxels in dynamic PET image which correlate best with",
  "user-specified TAC. Program saves a static image containing the Pearson's",
  "correlation coefficients.",
  " ",
  "Usage: @P [Options] image tac ccimage",
  " ",
  "Options:",
  " -slope=<filename>",
  "     Save the slopes of lines fitted to the concentration values as",
  "     static image.",
  " -ic=<filename>",
  "     Save the y axis intercepts of lines fitted to the concentrations as",
  "     static image.",
  " -integral",
  "     Integrals from 0 to each sample time are used in the correlation.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: imgthrs, imgqntls, imgslim, imgcalc, pxl2tac, imgdysmo",
  " ",
  "Keywords: image, TAC, correlation, mask, threshold",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int      ai, help=0, version=0, verbose=1;
  char     imgfile[FILENAME_MAX], tacfile[FILENAME_MAX], outfile[FILENAME_MAX];
  char     slopefile[FILENAME_MAX], icfile[FILENAME_MAX];
  int      useIntegralCurve=0;
  char    *cptr, tmp[128];
  int      ret;
  
  
  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=tacfile[0]=outfile[0]=slopefile[0]=icfile[0]=(char)0;
  /* Get options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(strncasecmp(cptr, "SLOPE=", 6)==0) {
      strlcpy(slopefile, cptr+6, FILENAME_MAX); continue;
    } else if(strncasecmp(cptr, "IC=", 3)==0) {
      strlcpy(icfile, cptr+3, FILENAME_MAX); continue;
    } else if(strncasecmp(cptr, "INTEGRAL", 1)==0) {
      useIntegralCurve=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(imgfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {strlcpy(tacfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {strlcpy(outfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Did we get all the information that we need? */
  if(!outfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("imgfile := %s\n", imgfile);
    printf("tacfile := %s\n", tacfile);
    printf("outfile := %s\n", outfile);
    if(slopefile[0]) printf("slopefile := %s\n", slopefile);
    if(icfile[0]) printf("icfile := %s\n", icfile);
    printf("useIntegralCurve := %d\n", useIntegralCurve);
    fflush(stdout);
  }
  if(verbose>9) IMG_TEST=verbose-9; else IMG_TEST=0;


  /*
   *  Read PET image and TAC
   */
  if(verbose>0) printf("reading data files\n");
  IMG img; DFT tac;  dftInit(&tac); imgInit(&img);
  double scanlen=1.0E+020;
  int frameNr=0;
  ret=imgReadModelingData(
    imgfile, NULL, tacfile, NULL, NULL, &scanlen, &frameNr, &img,
    NULL, &tac, 0, stdout, verbose-2, tmp);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot read the data.\n");
    fprintf(stderr, "%s.\n", tmp);
    if(verbose>1) printf("  ret := %d\n", ret);
    return(2);
  }
  if(verbose>1) {
    printf("scanlen := %g\n", scanlen);
    printf("frameNr := %d\n", frameNr);
  }
  /* Check that image is dynamic */
  if(frameNr<3) {
    fprintf(stderr, "Error: too few time frames.\n");
    imgEmpty(&img); dftEmpty(&tac); return(2);
  }
  /* Check that TAC data contains just one TAC */
  if(tac.voiNr>1) {
    fprintf(stderr, "Warning: only the first TAC is used.\n");
  }


  /*
   *  Allocate memory for output image data
   */
  if(verbose>1) fprintf(stdout, "allocating memory\n");
  IMG ccimg; imgInit(&ccimg);
  IMG kimg; imgInit(&kimg);
  IMG bimg; imgInit(&bimg);
  ret=imgAllocateWithHeader(&ccimg, img.dimz, img.dimy, img.dimx, 1, &img);
  if(ret) {
    fprintf(stderr, "Error: cannot allocate memory for new image.\n"); 
    imgEmpty(&img);
    return(3);
  }
  /* Copy data from the original image */
  if(verbose>2) fprintf(stdout, "copying data\n");
  ccimg.start[0]=img.start[0];
  ccimg.end[0]=img.end[img.dimt-1];
  ccimg.mid[0]=0.5*(ccimg.start[0]+ccimg.end[0]);
  /* Slope data */
  if(slopefile[0]) {
    ret=imgAllocateWithHeader(&kimg, img.dimz, img.dimy, img.dimx, 1, &img);
    if(ret) {
      fprintf(stderr, "Error: cannot allocate memory for new image.\n"); 
      imgEmpty(&img); imgEmpty(&ccimg);
      return(4);
    }
    /* Copy data from the original image */
    if(verbose>2) fprintf(stdout, "copying data\n");
    kimg.start[0]=img.start[0];
    kimg.end[0]=img.end[img.dimt-1];
    kimg.mid[0]=0.5*(kimg.start[0]+kimg.end[0]);
  }
  /* Intercept data */
  if(icfile[0]) {
    ret=imgAllocateWithHeader(&bimg, img.dimz, img.dimy, img.dimx, 1, &img);
    if(ret) {
      fprintf(stderr, "Error: cannot allocate memory for new image.\n"); 
      imgEmpty(&img); imgEmpty(&ccimg); imgEmpty(&kimg);
      return(4);
    }
    /* Copy data from the original image */
    if(verbose>2) fprintf(stdout, "copying data\n");
    bimg.start[0]=img.start[0];
    bimg.end[0]=img.end[img.dimt-1];
    bimg.mid[0]=0.5*(bimg.start[0]+bimg.end[0]);
  }


  /*
   *  Compute correlation coefficients
   */
  int zi, yi, xi, ti;
  long long okNr=0;
  double *x, y[img.dimt], k, kSD, b, bSD, r, ySD;
  for(zi=0; zi<img.dimz; zi++) {
    for(yi=0; yi<img.dimy; yi++) for(xi=0; xi<img.dimx; xi++) {
      /* Set data for correlation calculation */
      if(useIntegralCurve==0) {
        x=tac.voi[0].y;
        for(ti=0; ti<img.dimt; ti++) y[ti]=img.m[zi][yi][xi][ti];
      } else {
        x=tac.voi[0].y2;
        float ytemp[img.dimt];
        ret=fpetintegral(img.start, img.end, img.m[zi][yi][xi], img.dimt, ytemp, NULL);
        if(ret==0) for(ti=0; ti<img.dimt; ti++) y[ti]=ytemp[ti];
        else for(ti=0; ti<img.dimt; ti++) y[ti]=0.0;
      }
      ret=pearson3(x, y, img.dimt, &k, &kSD, &b, &bSD, &r, &ySD);
      if(ret==0) {
        okNr++;
      } else {
        k=kSD=b=bSD=r=ySD=0.0;
      }
      ccimg.m[zi][yi][xi][0]=(float)r;
      if(slopefile[0]) {
        if(k>1.0E+03) kimg.m[zi][yi][xi][0]=1.0E+03;
        else if(k<-1.0E+03) kimg.m[zi][yi][xi][0]=-1.0E+03;
        else kimg.m[zi][yi][xi][0]=(float)k;
      }
      if(icfile[0]) bimg.m[zi][yi][xi][0]=(float)b;
    }
  }
  imgEmpty(&img);

  /*
   *  Write output image data
   */
  if(verbose>1) fprintf(stdout, "writing correlation image...\n");
  ret=imgWrite(outfile, &ccimg);
  if(ret) {
    fprintf(stderr, "Error: %s\n", ccimg.statmsg);
    imgEmpty(&ccimg); imgEmpty(&kimg); imgEmpty(&bimg); return(11);
  }
  if(verbose>0) fprintf(stdout, "%s written.\n", outfile);
  imgEmpty(&ccimg);
  /* Slope data */
  if(slopefile[0]) {
    if(verbose>1) fprintf(stdout, "writing slope image...\n");
    ret=imgWrite(slopefile, &kimg);
    if(ret) {
      fprintf(stderr, "Error: %s\n", kimg.statmsg);
      imgEmpty(&kimg); imgEmpty(&bimg); return(11);
    }
    if(verbose>0) fprintf(stdout, "%s written.\n", slopefile);
    imgEmpty(&kimg);
  }
  /* Intercept data */
  if(icfile[0]) {
    if(verbose>1) fprintf(stdout, "writing intercept image...\n");
    ret=imgWrite(icfile, &bimg);
    if(ret) {
      fprintf(stderr, "Error: %s\n", bimg.statmsg);
      imgEmpty(&bimg); return(11);
    }
    if(verbose>0) fprintf(stdout, "%s written.\n", icfile);
    imgEmpty(&bimg);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
