/** @file imgzavg.c
 *  @brief Image average over z dimension (planes).
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Calculate average of 3D or 4D image over z dimension (image planes).",
  " ",
  "Usage: @P [Options] imgfile outputfile",
  " ",
  "Image file can be in ECAT 6.3 or 7.x, NIfTI-1, or Analyze 7.5 format.",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: imgdelpl, imgshrink, imginteg, imgslim, esplit, imgflip",
  " ",
  "Keywords: image, cropping",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  char  imgfile[FILENAME_MAX], outfile[FILENAME_MAX];


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=outfile[0]=(char)0; 
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(imgfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(outfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Did we get all the information that we need? */
  if(!outfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }


  /* In verbose mode print options */
  if(verbose>1) {
    printf("imgfile := %s\n", imgfile);
    printf("outfile := %s\n", outfile);
  }


  /*
   *  Read PET image
   */
  if(verbose>0) {printf("reading %s\n", imgfile); fflush(stdout);}
  IMG img; imgInit(&img);
  if(imgRead(imgfile, &img)) {
    fprintf(stderr, "Error: %s\n", img.statmsg);
    return(2);
  }
  if(verbose>1) {
    printf("dimt := %d\n", img.dimt);
    printf("dimx := %d\n", img.dimx);
    printf("dimy := %d\n", img.dimy);
    printf("dimz := %d\n", img.dimz);
  }
  if(img.dimz<2) {
    // just a copy will be made
    fprintf(stderr, "Warning: image z dimension is 1.\n");
  }


  /*
   *  Process
   */
  if(verbose>0) printf("processing...\n");
  IMG oimg; imgInit(&oimg);
  if(imgMeanZ(&img, &oimg)) {
    fprintf(stderr, "Error: cannot calculate mean over z dimension.\n");
    imgEmpty(&img); return(6);
  }
  imgEmpty(&img);


  /*
   *  Save the modified image
   */
  if(verbose>0) printf("writing image %s\n", outfile);
  if(imgWrite(outfile, &oimg)) {
    fprintf(stderr, "Error: %s\n", oimg.statmsg);
    imgEmpty(&oimg); return(11);
  }
  imgEmpty(&oimg);
  if(verbose>0) printf("done.\n\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
