/** @file p2blood.c
 *  @brief Convert PET plasma TAC to blood TAC.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "libtpccurveio.h"
#include "libtpcmisc.h"
/*****************************************************************************/
#include "pbconv.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Compute the plasma-to-blood ratio curve used in b2plasma and p2blood",
  "programs.",
  "The hematocrit (HCr, HCT) is normally between 0.40-0.51 in men and 0.36-0.47",
  "in women. Note that HCr is lower in small vessels in tissue.",
  "Sample times are autointerpolated from 0 to endtime (min)",
  "with decreasing sample frequency.",
  " ",
  "Usage: @P [Options] tracer HCr endtime ratiofile",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "The following PET tracers for human studies are supported:",
  "  AH690",
  "     ratio is assumed to follow a population average curve.",
  "  AH691",
  "     ratio is assumed to follow a population average curve.",
  "  Carfentanil",
  "     ratio is assumed to follow a population average curve",
  "     (unpublished measurements from eight 70-min PET studies).",
  "  FBPA",
  "     ratio is assumed to rise from zero with slope 0.00888",
  "     (unpublished measurement from 10 subjects).",
  "  FDG",
  "     ratio is assumed to be 0.8 in the beginning and to rise with",
  "     slope 0.0012/min (Phelps ME et al. Ann Neurol 1979;6:371-388).",
  "  FDOPA",
  "     ratio is assumed to follow equation R(t)=(Rmax*t)/(Th+t),",
  "     where Rmax=1.446 and Th=83.56 (unpublished results).",
  "  Flu[mazenil]",
  "     ratio is assumed to follow an average curve based on two",
  "     subjects with assumed HCR=0.43.",
  "  FLUORIDE",
  "     ratio is assumed to be 1.23 in the beginning and to decrease slowly",
  "     with slope 0.00123/min (Hawkins et al., JNM 1992;33:633-642.).",
  "  FMPEP-D2",
  "     ratio is assumed to follow a population average curve.",
  "  FTHA",
  "     concentration in red blood cells is assumed to be zero.",
  "  MEAIB",
  "     ratio is assumed to rise from zero with slope 0.00398",
  "     (unpublished measurement from 7 subjects).",
  "  Metomidate or MTO",
  "     concentration is the same in RBC and plasma water.",
  "  ORM-B",
  "     ratio is assumed to follow a population average curve",
  "     (unpublished measurement from 6 subjects studied twice).",
  "  Palm[itate]",
  "     ratio is assumed to follow a population average curve",
  "     (unpublished measurement from 8 subjects).",
  "  PBR28HAB, PBR28MAB, PBR28LAB",
  "     ratio is assumed to follow a binding-type specific population average",
  "     curve (unpublished measurements from 90-min PET studies).",
  "  PE2I",
  "     ratio is assumed to follow a population average curve",
  "     (unpublished measurements from ten 70-min PET studies).",
  "  PIB or 6OHBTA1",
  "     ratio is assumed to follow a population average curve,",
  "     currently based on 15 subjects.",
  "  PK11195",
  "     ratio is assumed to follow a curve based on one",
  "     subject with measured HCr.",
  "  SMW139",
  "     ratio is assumed to follow a population average curve,",
  "     currently based on 11 subjects.",
  "For mice and rats (these conversion do not utilize hematocrit):",
  "  MOUSEFDG1",
  "     plasma-to-blood ratio is assumed to follow the function published by",
  "     Yu AS et al. J Nucl Med 2009;50(6):966-973.",
  "  MOUSEFDG2",
  "     plasma-to-blood ratio is assumed to follow the function published by",
  "     Huang et al. J Nucl Med 2017;58(4):611-616.",
  "  RATFDG",
  "     plasma-to-blood ratio is assumed to follow the function published by",
  "     Weber et al. Eur J Nucl Med 2002;29(3):319-s323.",
  " ",
  "Codes for tracers that have below mentioned properties:",
  "  norbc",
  "     concentration in red blood cells is assumed to be zero.",
  "  inwater",
  "     concentration is the same in RBC and plasma water.",
  " ",
  " ",
  "Example: estimate blood curve from measured FDG plasma TAC:",
  "     @P FDG i3344ap.kbq 0.38 90 i3344ab.kbq", 
  " ",
  "See also: b2plasma, p2blood, b2rbc, bpr2cpr, tacinv, tac2svg, tacformat",
  " ",
  "Keywords: simulation, blood, plasma, software testing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int        ai, help=0, version=0, verbose=1;
  int        fi, ret;
  int        tracer=-1;
  char       ofile[FILENAME_MAX], *cptr;
  char       tmp[1024];
  DFT        data;
  double     HCR=-1.0;
  int        ratio_type=0; // 0=RBC/plasma; 1=plasma/blood
  double     aiendtime=nan("");


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  ofile[0]=(char)0;
  dftInit(&data);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Next argument must be the tracer code */
  if(ai<argc) {
    tracer=tracer_code(argv[ai]);
    if(tracer<0) {
      fprintf(stderr, "Error: invalid tracer code '%s'.\n", argv[ai]);
      return(1);
    }
    ai++;
  }

  /* Next argument must be the hematocrit */
  if(ai<argc) {
    HCR=atof_dpi(argv[ai]); if(HCR>1.0) HCR/=100.0;
    if(HCR<0.01 || HCR>1.0) {
      fprintf(stderr, "Error: bad HCr %g.\n", HCR); return(1);}
    ai++;
  }

  /* Next argument must be the last sample time */
  if(ai<argc) {
    aiendtime=atof_dpi(argv[ai]);
    if(aiendtime<=0.0) {
      fprintf(stderr, "Error: invalid end time %s.\n", argv[ai]); return(1);}
    ai++;
  }

  /* Next argument must be the output file */
  if(ai<argc) {strcpy(ofile, argv[ai]); ai++;}

  /* Is something extra? */
  if(ai<argc) {
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!ofile[0]) {
    fprintf(stderr, "Error: missing command-line argument; try %s --help\n",
      argv[0]);
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("aiendtime := %g\n", aiendtime);
    printf("ofile := %s\n", ofile);
    printf("HCR := %.2f\n", HCR);
    printf("tracer := %d\n", tracer);
  }

  /*
   *  Make data for calculated function values
   */
  {
    double c, d;
    int rnr;
    /* calculate the nr of interpolated data points */
    c=0.0; d=0.02; rnr=1;
    while(c+d<aiendtime && rnr<9999) {c+=d; d*=1.05; rnr++;}
    d=aiendtime-c; c=aiendtime; rnr++;
    /* allocate memory */
    if(dftSetmem(&data, rnr, 1)) {
      fprintf(stderr, "Memory allocation error.\n");
      return(3);
    }
    data.voiNr=1; data.frameNr=rnr;
    /* set times */
    data.timetype=DFT_TIME_MIDDLE;
    c=0.0; d=0.02; fi=0; data.x[fi++]=c;
    while(c+d<aiendtime && rnr<9999) {c+=d; d*=1.05; data.x[fi++]=c;}
    d=aiendtime-c; c=aiendtime; data.x[fi++]=c; data.frameNr=fi;
  }
  data.timeunit=TUNIT_MIN;
  dftUnitToDFT(&data, CUNIT_UNITLESS);
  /* Set TAC name */
  strcpy(data.voi[0].voiname, "P2B"); 
  strcpy(data.voi[0].name, data.voi[0].voiname);


  /*
   *  Calculate the ratio curve
   */
  if(verbose>1) printf("calculating ratio curve\n");
  /* Try first RBC/plasma curve */
  ratio_type=0; if(verbose>2) printf("trying RBC/plasma ratio\n");
  ret=rbc_plasma_ratio(tracer, data.x, data.voi[0].y2, data.frameNr);
  /* If tracer was not identified, then try plasma/blood ratio */
  if(ret==-1) {
    ratio_type=1; if(verbose>2) printf("trying plasma/blood ratio\n");
    ret=plasma_to_blood_ratio(tracer, data.x, data.voi[0].y2, data.frameNr);
  }
  if(ret) {
    fprintf(stderr, "Error (%d): cannot calculate ratio.\n", ret);
    dftEmpty(&data); return(5);
  }
  if(verbose>2) printf("ratio_type := %d\n", ratio_type);
  if(ratio_type==1 && verbose>1)
    fprintf(stderr, "Note: HCr is not used in conversion.\n"); 
  if(verbose>5) {
    if(ratio_type==0) strcpy(tmp, "RBC-to-plasma");
    else strcpy(tmp, "plasma-to-blood");
    for(fi=0; fi<data.frameNr; fi++)
      printf("%s[%g] := %.3f\n", tmp, data.x[fi], data.voi[0].y2[fi]);
  }    
  /* If RBC-to-plasma ratio was computed, then convert it to plasma-to-blood */
  /* Otherwise just copy */
  if(ratio_type==0) {
    if(verbose>1) printf("calculating plasma/blood curve\n");
    for(fi=0; fi<data.frameNr; fi++) 
      data.voi[0].y[fi]=1.0/(1.0-HCR*(1.0-data.voi[0].y2[fi]));
  } else {
    for(fi=0; fi<data.frameNr; fi++) 
      data.voi[0].y[fi]=data.voi[0].y2[fi];
  }


  /*
   *  Save data
   */
  if(verbose>1) printf("writing %s\n", ofile);
  data._type=DFT_FORMAT_PMOD;
  DFT_NR_OF_DECIMALS=6;
  if(dftWrite(&data, ofile)) {
    fprintf(stderr, "Error in writing '%s': %s\n", ofile, dfterrmsg);
    dftEmpty(&data); return(11);
  }
  if(verbose>0) printf("Plasma-to-blood curve written in %s\n", ofile);

  /* Free memory */
  dftEmpty(&data);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
