/// @file heartcorr.c
/// @brief Functions for correcting spill-over and recovery
///        errors in myocardial LV muscle and cavity.
/// @author Vesa Oikonen
///
/*****************************************************************************/
#include "libtpcidi.h"
/*****************************************************************************/

/*****************************************************************************/
/// @cond
/** Upcoming function.
   @return Nothing yet.
 */
int heartcorr(
  /** Verbose level */
  int verbose
) {
  if(verbose>0) printf("heartcorr()\n");

  return(0);
}
/// @endcond
/*****************************************************************************/

/*****************************************************************************/
/** Calculate recovery and spillover correction coefficients.

    Based on
    Henze E, Huang S-C, Ratib O, Hoffman E, Phelps ME, Schelbert HR.
    Measurements of regional tissue and blood-pool radiotracer concentrations
    from serial tomographic images of the heart. J Nucl Med. 1983;24:987-996.

    @sa rcPeakPET, simMyocDiameterCurve, imgSimulateSphere
    @return Returns 0 if successful, or <> 0 in case of an error.
 */
int heartRecoverySpilloverCorrectionFactors(
  /** radius of cavity and circular ROI (mm). */
  double R,
  /** thickness of myocardium (mm). */
  double d,
  /** spatial resolution (mm); s = FWHM/(2*SQRT(2*LN(2))). */
  double s,
  /** Vascular volume fraction in myocardium; Henze et al assumed 0.1. */
  double Vb,
  /** Pointer to resulting correction coefficient FMM, see the article. */
  double *FMM,
  /** Pointer to resulting correction coefficient FMB, see the article. */
  double *FMB,
  /** Pointer to resulting correction coefficient FBM, see the article. */
  double *FBM,
  /** Pointer to resulting correction coefficient FBB, see the article. */
  double *FBB
) {
  double hp;

  if(R<=0.0 || d<=0.0 || s<=0.0) return(1);
  if(FMM==NULL || FMB==NULL || FBM==NULL || FBB==NULL) return(2);
  if(Vb<0.0 || Vb>=1.0) return(3);
  hp = exp(-R*R/(2.0*s*s));
  *FMB = hp - exp(-(R+d)*(R+d)/(2.0*s*s));
  *FMM = erf( (d/2.0) / (M_SQRT2*s) );
  *FBB = 1.0 - hp;
  *FBM = Vb + 0.5*(1.0 - *FMM);  //0.6 - 0.5 * *FMM;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
