/******************************************************************************
 * This file is not compiled into the library, but it contains main()
 * which is compiled to an executable, used to test the library functions. 
 *****************************************************************************/
//#pragma pack(1)
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/
#include <unistd.h>
/*****************************************************************************/

/*****************************************************************************/
#ifndef MAXVAL
#define MAXVAL 100000;
#endif
/*****************************************************************************/

/*****************************************************************************/
/* Test function declarations: */
int test_libtpcimgio_struct_sizes(int verbose);
int test_create_img(IMG *img, int dim_x, int dim_y, int dim_z, int dim_t);
int test_polarmap();
int test_create_polarmap(
   IMG *img, int num_rings, int *sectors_per_ring, 
   float *ring_position, short int *ring_angle, short int start_angle
);
int test_img_io(int VERBOSE);
int test_polarmap_io(IMG *img);
int test_pxlGet();
int test_pxlMakeRoom();
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Usage: @P [options]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " -t, --test",
  "     Run all tests for library functions.",
  0};
/*****************************************************************************/

/*****************************************************************************/
/** Run unit tests to the library functions
 *  @author Vesa Oikonen
 *  @return 0 if all tests pass, otherwise >0.
 * */
int main(
  /** Nr of arguments */
  int argc,
  /** Pointer to arrays of argument string */
  char *argv[ ]
) {
  int i, help=0, version=0, verbose=1, error=0, test=0;
  int ret;
  char *cptr;

  if(argc==1) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  for(i=1; i<argc; i++) {
    if(tpcProcessStdOptions(argv[i], &help, &version, &verbose)==0) continue;
    cptr=argv[i]; if(*cptr=='-') cptr++; if(*cptr=='-') cptr++;
    if(strncasecmp(cptr, "TEST", 1)==0) {
      test=1; continue;
    } else {
      error++; break;
    }
  }
  if(error>0) {
    fprintf(stderr, "Error: specify --help for usage.\n");
    return(1);
  }
  /* Print help or version? */
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  if(test==0) return(0);

  if(verbose>0) printf("running tests for library functions...\n");

  i=10;
  /* Struct sizes */
  if((ret=test_libtpcimgio_struct_sizes(verbose-1))!=0) {
    fprintf(stderr, "failed (%d).\n", ret); return(i);}

  /* img i/o */
  i++; ret=test_img_io(verbose);
  if(ret!=0) {fprintf(stderr, "failed (%d).\n", ret); return(i);}

  /* polarmap i/o */
  i++; if((ret=test_polarmap())!=0) {fprintf(stderr, "failed (%d).\n", ret); return(i);}

  /* pixel */
  i++; if((ret=test_pxlGet(verbose-1))!=0) {fprintf(stderr, "failed (%d).\n", ret); return(i);}
  i++; if((ret=test_pxlMakeRoom(verbose-1))!=0) {fprintf(stderr, "failed (%d).\n", ret); return(i);}


  if(verbose>0) printf("\nAll tests passed.\n\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Test that certain data structs are of correct size.
    @return 0 in case of success.
 */
int test_libtpcimgio_struct_sizes(
  /** Verbose level; enter 0 to print nothing. */ 
  int verbose
) {
  if(verbose>0) printf("\nTesting libtpcimgio struct sizes...\n\n");

  size_t n;

  /* ECAT7 */

  n=sizeof(ECAT7_mainheader);
  if(verbose>1) printf("sizeof(ECAT7_mainheader) := %zu\n", n);
  // Matches only if struct is packed (not aligned)
  //if(n!=(size_t)512) return(10);

  n=sizeof(ECAT7_imageheader);
  if(verbose>1) printf("sizeof(ECAT7_imageheader) := %zu\n", n);
  // Matches only if struct is packed (not aligned)
  //if(n!=(size_t)512) return(11);

  n=sizeof(ECAT7_scanheader);
  if(verbose>1) printf("sizeof(ECAT7_scanheader) := %zu\n", n);
  // Matches only if struct is packed (not aligned)
  //if(n!=(size_t)1024) return(12);

  n=sizeof(ECAT7_2Dscanheader);
  if(verbose>1) printf("sizeof(ECAT7_2Dscanheader) := %zu\n", n);
  // Matches only if struct is packed (not aligned)
  //if(n!=(size_t)512) return(13);

  n=sizeof(ECAT7_2Dnormheader);
  if(verbose>1) printf("sizeof(ECAT7_2Dnormheader) := %zu\n", n);
  // Matches only if struct is packed (not aligned)
  //if(n!=(size_t)512) return(14);

  n=sizeof(ECAT7_attenheader);
  if(verbose>1) printf("sizeof(ECAT7_attenheader) := %zu\n", n);
  // Matches only if struct is packed (not aligned)
  //if(n!=(size_t)512) return(15);

  n=sizeof(ECAT7_normheader);
  if(verbose>1) printf("sizeof(ECAT7_normheader) := %zu\n", n);
  // Matches only if struct is packed (not aligned)
  //if(n!=(size_t)512) return(16);

  n=sizeof(ECAT7_polmapheader);
  if(verbose>1) printf("sizeof(ECAT7_polmapheader) := %zu\n", n);
  // Matches only if struct is packed (not aligned)
  //if(n!=(size_t)512) return(17);


  /* ECAT 6.3 */

  n=sizeof(ECAT63_mainheader);
  if(verbose>1) printf("sizeof(ECAT63_mainheader) := %zu\n", n);
  // Matches only if struct is packed (not aligned)
  //if(n!=(size_t)512) return(20);

  n=sizeof(ECAT63_imageheader);
  if(verbose>1) printf("sizeof(ECAT63_imageheader) := %zu\n", n);
  // Matches only if struct is packed (not aligned)
  //if(n!=(size_t)512) return(21);

  n=sizeof(ECAT63_scanheader);
  if(verbose>1) printf("sizeof(ECAT63_scanheader) := %zu\n", n);
  // Matches only if struct is packed (not aligned)
  //if(n!=(size_t)512) return(22);

  n=sizeof(ECAT63_attnheader);
  if(verbose>1) printf("sizeof(ECAT63_attnheader) := %zu\n", n);
  // Matches only if struct is packed (not aligned)
  //if(n!=(size_t)512) return(23);

  n=sizeof(ECAT63_normheader);
  if(verbose>1) printf("sizeof(ECAT63_normheader) := %zu\n", n);
  // Matches only if struct is packed (not aligned)
  //if(n!=(size_t)512) return(24);


  if(verbose>0) printf("\n  ok\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/*!
 * Create IMG for testing purposes.
 *
 * @param img	Pointer to an initiated but not allocated IMG
 * @param dim_x	X dimension (nr of columns)
 * @param dim_y	Y dimension (nr of rows)
 * @param dim_z	Z dimension (nr of planes)
 * @param dim_t time dimension (nr of time frames, or gates)
 * @returns 0 if IMG could be created, or nonzero in case of error.
 */
int test_create_img(IMG *img, int dim_x, int dim_y, int dim_z, int dim_t) {
  int zi, xi, yi, fi;
  float f=-100.0;

  if(img==NULL || img->status!=IMG_STATUS_INITIALIZED) return(1);
  if(imgAllocate(img, dim_z, dim_y, dim_x, dim_t)){
    printf("    \nError in image allocation: imgAllocate() @ img.c\n");
    return(1);
  }
  //printf("  allocated\n");
  img->dimt=dim_t; img->dimx=dim_x; img->dimy=dim_y; img->dimz=dim_z;
  for(zi=0; zi<img->dimz; zi++) img->planeNumber[zi]=zi+1;
  img->type = IMG_TYPE_IMAGE;
  for(fi=0; fi<dim_t; fi++) {
    if(fi==0) img->start[fi]=0.0; else img->start[fi]=img->end[fi-1];
    img->end[fi]=img->start[fi]+6.0;
    img->mid[fi]=0.5*(img->start[fi]+img->end[fi]);
  }
  //printf("  frames set\n");
  for(zi=0; zi<dim_z; zi++) {
    //printf("  zi=%d\n", zi);
    f+=(float)(3*zi);
    for(yi=0; yi<dim_y; yi++) {
      f+=(float)(2*yi);
      for(xi=0; xi<dim_x; xi++) {
        f+=(float)(xi);
        if(f>10000.0) f=-100.0; // prevent error when floats are scaled to short ints
        for(fi=0; fi<dim_t; fi++) {
          img->m[zi][yi][xi][fi]=f+(float)fi;
        }
      }
    }
  }
  //printf("  pixel values set\n"); fflush(stdout);
  img->unit=CUNIT_KBQ_PER_ML;
  img->scanStart=time(NULL);
  img->decayCorrection=IMG_DC_CORRECTED;
  img->isotopeHalflife=HL_C11*60.;
  imgSetDecayCorrFactors(img, 1);
  //printf("  decay correction factors set\n");

  img->xform[0]=NIFTI_XFORM_UNKNOWN; // qform
  img->xform[1]=NIFTI_XFORM_SCANNER_ANAT; // sform

  return(0);
}
/******************************************************************************/
/*!
 * Create POLARMAP IMG for testing purposes.
 *
 * @param img 			Pointer to an initiated but not allocated IMG
 * @param num_rings		Number of polar map rings
 * @param sectors_per_ring      Sectors per ring
 * @param ring_position		Ring positions
 * @param ring_angle		Ring angles
 * @param start_angle		Polar map start angle for rings, as defined in
 *                              ECAT 7 header
 * @returns 0 if IMG could be created, or nonzero in case of error.
 */
int test_create_polarmap(
  IMG *img, int num_rings, int *sectors_per_ring, float *ring_position,
  short int *ring_angle, short int start_angle
) {
  int si, fi, xi, dim_x, dim_t=1;
  float f=1.0;

  if(img==NULL || img->status!=IMG_STATUS_INITIALIZED) {
    printf("    \nError: invalid test parameters.\n");
    return(1);
  }
  for(si=dim_x=0; si<num_rings; si++) dim_x+=sectors_per_ring[si];
  if(imgAllocate(img, 1, 1, dim_x, dim_t)){
    printf("    \nError in image allocation: imgAllocate() @ img.c\n");
    return(1);
  }
  img->dimt=dim_t; img->dimx=dim_x; img->dimy=1; img->dimz=1;
  img->type = IMG_TYPE_POLARMAP;
  for(fi=0; fi<dim_t; fi++) {
    if(fi==0) img->start[fi]=0.0; else img->start[fi]=img->end[fi-1];
    img->end[fi]=(fi+1)*60.0;
    img->mid[fi]=0.5*(img->start[fi]+img->end[fi]);
  }
  for(xi=0; xi<dim_x; xi++) {
    f+=(float)(xi);
    for(fi=0; fi<dim_t; fi++) {
      img->m[0][0][xi][fi]=f*(float)(1+fi);
    }
  }
  img->unit=CUNIT_KBQ_PER_ML;
  img->polarmap_num_rings=num_rings;
  for(si=0; si<num_rings; si++) {
    img->polarmap_sectors_per_ring[si]=sectors_per_ring[si];
    img->polarmap_ring_position[si]=ring_position[si];
    img->polarmap_ring_angle[si]=ring_angle[si];
  }
  img->polarmap_start_angle=start_angle;

  printf("   polarmap IMG created.\n");
  return(0);
}
/******************************************************************************/

/******************************************************************************/
/*!
   Testbench for image IO functions. Tests include:<br>
   ECAT7 image write and read<br>
   ECAT7 image values write and read<br>
   ECAT6 image volume write and read<br>
   Analyze image volume write and read<br>
   NIfTI-1 image volume write and read<br>

   This functions writes and reads images of small size.
   Function in file test_libtpcimgiobig.c writes huge files.
  
   @return 0 in case of success
 */
int test_img_io(
  /** Verbose level; enter 0 to print nothing. */ 
  int VERBOSE
) {
  int xdim, ydim, zdim, fdim;
  long long pxlNr;
  char *bfname="test_img_io";
  char fname[FILENAME_MAX];
  int ret;
  IMG img, img2;

  if(VERBOSE>0) printf("\n Testing IMG I/O functions\n");
  if(VERBOSE>3) IMG_TEST=VERBOSE-3;
  imgInit(&img); imgInit(&img2);

  /* Create test image for ECAT 7 image formats;
     note that ECAT matrix list limits plane number to < 256 and frame number to < 512 */
  xdim=32; ydim=32; zdim=10; fdim=5; // 1 310 720 000 pixels
  pxlNr=xdim*ydim*zdim*fdim;
  if(VERBOSE>1) printf("test image size is %dx%dx%dx%d = %lld\n", xdim, ydim, zdim, fdim, pxlNr);
  if(test_create_img(&img, xdim, ydim, zdim, fdim)) {
    fprintf(stderr, "Error: cannot create test data.\n");
    return(1);
  }
  img.type=IMG_TYPE_IMAGE;


  /* ECAT7 image write and read */
  sprintf(fname, "%s.v", bfname);
  img._dataType=ECAT7_SUNI2;

  if(VERBOSE>1) printf("\n  testing 3D ECAT 7 format\n");
  img._fileFormat=IMG_E7;
  ret=imgWrite(fname, &img); if(ret) {imgEmpty(&img); return(11);}
  ret=imgRead(fname, &img2); if(ret) {imgEmpty(&img); imgEmpty(&img2); return(12);}
  ret=imgMatch(&img, &img2, 0.001);
  if(ret!=0) {
    fprintf(stderr, "Error: Ecat7 image is not the same after write/read (%d)\n", ret);
    if(ret<400 && VERBOSE>2) {
      printf("\n--------\nBefore:\n"); imgInfo(&img);
      printf("\n--------\nAfter:\n"); imgInfo(&img2);
    }
    imgEmpty(&img); imgEmpty(&img2); return(13);
  }
  remove(fname); imgEmpty(&img2);

  if(VERBOSE>1) printf("\n  testing 2D ECAT 7 format\n");
  img._fileFormat=IMG_E7_2D;
  ret=imgWrite(fname, &img); if(ret) {imgEmpty(&img); return(14);}
  ret=imgRead(fname, &img2); if(ret) {imgEmpty(&img); imgEmpty(&img2); return(15);}
  ret=imgMatch(&img, &img2, 0.001);
  if(ret!=0) {
    fprintf(stderr, "Error: Ecat7 image is not the same after write/read (%d)\n", ret);
    if(ret<400 && VERBOSE>2) {
      printf("\n--------\nBefore:\n"); imgInfo(&img);
      printf("\n--------\nAfter:\n"); imgInfo(&img2);
    }
    imgEmpty(&img); imgEmpty(&img2); return(16);
  }
  remove(fname); imgEmpty(&img2);

  imgEmpty(&img); 


  /* Create test image for ECAT 6.3 image formats; at the time, files were much smaller,
     and code does not support large files. */
  xdim=32; ydim=32; zdim=15; fdim=10;
  pxlNr=xdim*ydim*zdim*fdim;
  if(VERBOSE>1) printf("test image size is %dx%dx%dx%d = %lld\n", xdim, ydim, zdim, fdim, pxlNr);
  if(test_create_img(&img, xdim, ydim, zdim, fdim)) {
    fprintf(stderr, "Error: cannot create test data.\n");
    return(1);
  }
  img.type=IMG_TYPE_IMAGE;

  if(VERBOSE>1) printf("  testing ECAT 6.3 format\n");
  sprintf(fname, "%s.img", bfname);
  img._fileFormat=IMG_E63;
  img._dataType=ECAT7_SUNI2;
  img.scanner=SCANNER_ECAT931;
  ret=imgWrite(fname, &img); if(ret) return(21);
  ret=imgRead(fname, &img2); if(ret) return(22);
  /* ECAT6 image values write and read testing */
  ret=imgMatch(&img, &img2, 0.001);
  if(ret!=0) {
    fprintf(stderr, "Error: Ecat6 image is not the same after write/read (%d)\n", ret);
    if(ret<400 && VERBOSE>2) {
      printf("\n--------\nBefore:\n"); imgInfo(&img);
      printf("\n--------\nAfter:\n"); imgInfo(&img2);
    }
    imgEmpty(&img); imgEmpty(&img2);
    return(23);
  }
  remove(fname); imgEmpty(&img2);

  imgEmpty(&img); 



  /* Create a test image for NIfTI and Analyze image formats */
  xdim=32; ydim=32; zdim=32; fdim=10; // 5.2 Gb NIfTI file, 2.5 Gb Analyze file
  pxlNr=xdim*ydim*zdim*fdim;
  if(VERBOSE>1) printf("test image size is %dx%dx%dx%d = %lld\n", xdim, ydim, zdim, fdim, pxlNr);
  if(test_create_img(&img, xdim, ydim, zdim, fdim)) {
    fprintf(stderr, "Error: cannot create test data.\n");
    return(1);
  }
  img.type=IMG_TYPE_IMAGE;
  img._dataType=0; // Not stored in NIfTI and Analyze headers
  img.unit=0; // Not stored in NIfTI and Analyze headers
  img.scanner=SCANNER_UNKNOWN; // Not stored in NIfTI and Analyze headers


  /* NIfTI single file format write and read */
  niftiRemove(fname, 0, 0);
  if(VERBOSE>1) printf("  testing NIFTI single file format\n");
  strcpy(fname, bfname);
  img._fileFormat=IMG_NIFTI_1S;
  ret=imgWrite(fname, &img); if(ret) return(41);
  int file_format;
  ret=imgFormatDetermine(fname, NULL, NULL, NULL, NULL,
         &file_format, NULL, NULL, NULL, 0);
  if(ret!=0) return(42);
  if(file_format!=IMG_NIFTI_1S) return(43);
  ret=imgRead(fname, &img2); if(ret) return(44);
  niftiRemove(fname, 0, 0);
  ret=imgMatch(&img, &img2, 0.001);
  if(ret!=0) {
    fprintf(stderr, "Error: NIfTI 1S image is not the same after write/read (%d)\n", ret);
    if(ret<400 && VERBOSE>2) {
      printf("\n--------\nBefore:\n"); imgInfo(&img);
      printf("\n--------\nAfter:\n"); imgInfo(&img2);
    }
    imgEmpty(&img); imgEmpty(&img2);
    return(23);
  }
  remove(fname); imgEmpty(&img2);


  /* NIfTI dual file format write and read */
  if(VERBOSE>1) printf("  testing NIFTI dual file format\n");
  strcpy(fname, bfname);
  img._fileFormat=IMG_NIFTI_1D;
  ret=imgWrite(fname, &img); if(ret) return(46);
  ret=imgFormatDetermine(fname, NULL, NULL, NULL, NULL,
         &file_format, NULL, NULL, NULL, 0);
  if(ret!=0) return(46);
  if(file_format!=IMG_NIFTI_1D) return(47);
  ret=imgRead(fname, &img2); if(ret) return(48);
  niftiRemove(fname, 0, 0);
  ret=imgMatch(&img, &img2, 0.001);
  if(ret!=0) {
    fprintf(stderr, "Error: NIfTI 1D is not the same after w/r (%d)\n", ret);
    if(ret<400 && VERBOSE>2) {
      printf("\n--------\nBefore:\n"); imgInfo(&img);
      printf("\n--------\nAfter:\n"); imgInfo(&img2);
    }
    imgEmpty(&img); imgEmpty(&img2);
    return(49);
  }
  imgEmpty(&img2);


  /* Analyze image volume write and read (Big endian) */
  if(VERBOSE>1) printf("  testing Analyze big endian format\n");
  strcpy(fname, bfname);
  img._fileFormat=IMG_ANA; /*ANALYZE_TEST=IMG_TEST=4;*/
  img._dataType=0;
  img.unit=0;
  img.scanner=SCANNER_UNKNOWN;
  ret=imgWrite(fname, &img); if(ret) return(31);
  ret=imgRead(fname, &img2); if(ret) return(32);
  ret=imgMatch(&img, &img2, 0.001);
  niftiRemove(fname, IMG_NIFTI_1D, 0);
  if(ret!=0) {
    fprintf(stderr, "Error: Analyze image (big endian) is not the same after w/r (%d)\n", ret);
    if(ret<400 && VERBOSE>2) {
      printf("\n--------\nBefore:\n"); imgInfo(&img);
      printf("\n--------\nAfter:\n"); imgInfo(&img2);
    }
    imgEmpty(&img); imgEmpty(&img2);
    return(33);
  }
  imgEmpty(&img2);

  /* Analyze image volume write and read (Little endian) */
  if(VERBOSE>1) printf("  testing Analyze little endian format\n");
  strcpy(fname, bfname);
  img._fileFormat=IMG_ANA_L;
  ret=imgWrite(fname, &img); if(ret) return(34);
  ret=imgRead(fname, &img2); if(ret) return(35);
  ret=imgMatch(&img, &img2, 0.001);
  niftiRemove(fname, IMG_NIFTI_1D, 0);
  if(ret!=0) {
    fprintf(stderr, "Error: Analyze image 2 is not the same after w/r (%d)\n", ret);
    if(ret<400 && VERBOSE>2) {
      printf("\n--------\nBefore:\n"); imgInfo(&img);
      printf("\n--------\nAfter:\n"); imgInfo(&img2);
    }
    imgEmpty(&img); imgEmpty(&img2);
    return(36);
  }
  imgEmpty(&img2);



  imgEmpty(&img); 

  if(VERBOSE>0) printf("  ok\n");
  return(0);
}
/******************************************************************************/

/*!
 * Test bench for polar maps
 */
int test_polarmap() {
  int ret;
  IMG polarmap;

  printf("\n Testing POLARMAP IMG functions\n");
  int num_rings=4;
  int sectors_per_ring[]={6,6,4,1};
  float ring_position[]={0,0.2,0.4,0.6};
  short int ring_angle[]={90,90,45,0};
  short int start_angle=0;

  imgInit(&polarmap);
  if(test_create_polarmap(&polarmap, num_rings, sectors_per_ring, 
                          ring_position, ring_angle, start_angle))
  {
     printf("cannot create test data.\n");
     return(1);
  }
  ret = test_polarmap_io(&polarmap);
  imgEmpty(&polarmap);

  return(ret);
}

/*!
 * Testbench for polar map images
 *
 * @param img img structure for testing
 * @return 0 if ok
 */
int test_polarmap_io(IMG *img) {
  char *fname="test_polarmap_io";
  int ret;
  IMG img2;

  printf("\n Testing POLARMAP IMG I/O functions\n");
  img->type=IMG_TYPE_POLARMAP;
  /*IMG_TEST=2;*/
  imgInit(&img2);
  printf("\tfname := %s\n", fname); 

  /* ECAT7 image write and read */
  img->_fileFormat=IMG_POLARMAP;
  printf("    writing %s\n", fname); 
  ret=imgWrite(fname, img);
  if(ret) printf("    error %d\n", ret); 
  if(ret) return(11);
  printf("    reading %s\n", fname); 
  ret=imgRead(fname, &img2); 
  if(ret) printf("    error %d\n", ret); 
  if(ret) return(12);
  remove(fname); imgEmpty(&img2);

  return(0);
}

/*****************************************************************************/

/*****************************************************************************/
/** Tests for pxlGet() */
int test_pxlGet(
  /** Verbose level; enter 0 to print nothing. */ 
  int verbose
) {
  if(verbose>0) printf("\n Testing pxlGet()\n");

  IMG_PIXELS pxls;
  pxlInit(&pxls);
  IMG_PIXEL pxl, pxl2;

  pxl.x=1; pxl.y=11; pxl.z=101;
  if(pxlAdd(&pxls, &pxl)!=0) {pxlFree(&pxls); return(1);}
  if(pxlGet(&pxls, 0, &pxl2)!=0) {pxlFree(&pxls); return(2);}
  if(pxl.x!=pxl2.x || pxl.y!=pxl2.y || pxl.z!=pxl2.z) {pxlFree(&pxls); return(3);}

  pxl.x=2; pxl.y=12; pxl.z=102;
  if(pxlAdd(&pxls, &pxl)!=0) {pxlFree(&pxls); return(11);}
  if(pxlGet(&pxls, 1, &pxl2)!=0) {pxlFree(&pxls); return(12);}
  if(pxl.x!=pxl2.x || pxl.y!=pxl2.y || pxl.z!=pxl2.z) {pxlFree(&pxls); return(13);}

  if(pxlGet(&pxls, 300, &pxl2)==0) {pxlFree(&pxls); return(21);}
  if(pxlGet(&pxls, -1, &pxl2)==0) {pxlFree(&pxls); return(22);}

  if(verbose>0) printf("   ok\n");

  pxlFree(&pxls);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Tests for pxlmakeRoom() */
int test_pxlMakeRoom(
  /** Verbose level; enter 0 to print nothing. */ 
  int verbose
) {
  if(verbose>0) printf("\n Testing pxlMakeRoom()\n");

  if(pxlMakeRoom(NULL, 0, 0)==0) return(1);

  IMG_PIXELS pxls;
  pxlInit(&pxls);

  if(pxlMakeRoom(&pxls, 0, 0)!=0) return(2);

  IMG_PIXEL pxl;
  pxl.x=0; pxl.y=10; pxl.z=100;
  long long int n=50;
  for(long long int i=1; i<=n; i++) {
    pxl.x++; pxl.y++; pxl.z++;
    if(pxlAdd(&pxls, &pxl)!=0) {pxlFree(&pxls); return(11);}
  }
  if(pxls.pxlNr!=n) {pxlFree(&pxls); return(12);}

  int ret=0;

  if(pxlMakeRoom(&pxls, n+1, 1)==0) {pxlFree(&pxls); return(21);}
  if(pxlMakeRoom(&pxls, n, 1)!=0) {pxlFree(&pxls); return(22);}
  if(pxls.pxlNr!=n || pxls._pxlNr<n+1) {pxlFree(&pxls); return(23);}
  for(long long int i=1; i<=pxls.pxlNr; i++) {
    if(pxls.p[i-1].x!=i || pxls.p[i-1].y!=10+i || pxls.p[i-1].z!=100+i) ret++;
  }
  if(ret>0) {pxlFree(&pxls); return(24);}

  if(pxlMakeRoom(&pxls, 0, 2)!=0) {pxlFree(&pxls); return(32);}
  if(verbose>1) printf("  pxlNr=%lld _pxlNr=%lld\n", pxls.pxlNr, pxls._pxlNr);
  if(pxls.pxlNr!=n+2 || pxls._pxlNr<n+2) {pxlFree(&pxls); return(33);}
  pxls.p[0].x=pxls.p[0].y=pxls.p[0].z=0;
  pxls.p[1].x=pxls.p[1].y=pxls.p[1].z=1;
  for(long long int i=1; i<=n; i++) {
    if(pxls.p[i+1].x!=i || pxls.p[i+1].y!=10+i || pxls.p[i+1].z!=100+i) ret++;
  }
  if(ret>0) {pxlFree(&pxls); return(34);}

  if(verbose>0) printf("   ok\n");

  pxlFree(&pxls);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
