/** @file maskinv.c
 *  @brief Invert mask image.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Invert mask image in ECAT 6.3 or 7.x, NIfTI-1, or Analyze 7.5 format.",
  "Pixels with value zero are set to 1, and other pixels are set to 0.",
  " ",
  "Usage: @P [Options] maskfile [invertedfile]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: maskconj, maskdila, maskeros, imgmask, imgthrs, masklbl, img2dft",
  " ",
  "Keywords: image, mask, invert",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int     ai, help=0, version=0, verbose=1;
  int     ret;
  char    maskfile[FILENAME_MAX], outfile[FILENAME_MAX];
  char   *cptr=NULL;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  maskfile[0]=outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(maskfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(outfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Did we get all the information that we need? */
  if(!maskfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  if(!outfile[0]) {
    strcpy(outfile, maskfile);
  }


  /* In verbose mode print options */
  if(verbose>1) {
    printf("maskfile := %s\n", maskfile);
    printf("outfile := %s\n", outfile);
    fflush(stdout);
  }


  /*
   *  Read mask, and optional PET image
   */
  IMG mask; imgInit(&mask);

  if(verbose>0) {printf("reading %s\n", maskfile); fflush(stdout);}
  ret=imgRead(maskfile, &mask);
  if(ret) {
    fprintf(stderr, "Error: %s\n", mask.statmsg);
    if(verbose>1) printf("ret := %d\n", ret);
    return(2);
  }
  if(mask.dimt>1) {
    fprintf(stderr, "Error: mask cannot be dynamic image.\n");
    imgEmpty(&mask); return(2);
  }
  /* Check whether mask has any voxels to begin with */
  unsigned int mn;
  mn=imgMaskCount(&mask);
  if(mn==0) {
    fprintf(stderr, "Warning: initial mask contains no positive voxels.\n");
    imgEmpty(&mask);
    return(0);
  }
  if(verbose>1) {
    printf("initial_nr_of_positive_voxels := %u\n", mn);
    fflush(stdout);
  }


  /*
   *  Invert the mask
   */
  if(verbose>0) {printf("inverting\n"); fflush(stdout);}
  imgMaskInvert(&mask);
  /* Check whether mask has any voxels left */
  mn=imgMaskCount(&mask);
  if(mn==0) {
    fprintf(stderr, "Warning: all voxels are zeroes; empty mask not saved.\n");
    imgEmpty(&mask);
    return(0);
  }
  if(verbose>1) {
    printf("nr_of_positive_voxels := %u\n", mn);
  }


  /*
   *  Save the modified mask
   */
  if(verbose>2) printf("writing mask\n");
  ret=imgWrite(outfile, &mask);
  if(ret) {
    fprintf(stderr, "Error: %s\n", mask.statmsg);
    imgEmpty(&mask); return(11);
  }
  imgEmpty(&mask);
  if(verbose>0) printf("mask written.\n\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
