/** @file niftiift.c
 *  @brief Procedures for reading and writing NIfTI header with IFT struct.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/
#include "niftiift.h"
/*****************************************************************************/

/*****************************************************************************/
/** Copy NIfTI-1 header contents from IFT struct into header struct.
 *  @return Returns STATUS_OK (0) when successful, STATUS_INVALIDHEADER if
 *  no fields were changed, or STATUS_UNSUPPORTED if field value is not valid.
 */
int niftiHeaderFromIFT(
  /** Pointer to target NIfTI-1 header struct. Fields that are not given
   *  in IFT will not be changed.*/
  NIFTI_DSR *dsr,
  /** Pointer to source IFT struct; contents are copied to NIfTI header struct. */
  IFT *ift,
  /** Verbose level */
  int verbose
) {
  int i, n, ne=0, ret;
  char keyname[256];
  float f;

  /* Check input */
  if(verbose>0) printf("%s(dsr, ift)\n", __func__);
  if(dsr==NULL || ift==NULL) return STATUS_FAULT;
  if(ift->keyNr<1) return STATUS_INVALIDHEADER;
  if(verbose>5) iftWrite(ift, "stdout", 0);

  strcpy(keyname, "byte_order");
  if((i=iftGet(ift, keyname, 0))>=0) {
    if(strcasecmp(ift->item[i].value, "big")==0) {
      dsr->byte_order=0; ne++;
    } else if(strcasecmp(ift->item[i].value, "little")==0) {
      dsr->byte_order=1; ne++;
    }
  }

  strcpy(keyname, "sizeof_hdr");
  ret=iftGetIntValue(ift, 0, keyname, &i, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && i>=0) {dsr->h.sizeof_hdr=i; ne++;}

  strcpy(keyname, "data_type");
  if((i=iftGet(ift, keyname, 0))>=0) {
    memcpy(dsr->h.data_type, ift->item[i].value, 10); ne++;}

  strcpy(keyname, "db_name");
  if((i=iftGet(ift, keyname, 0))>=0) {
    memcpy(dsr->h.db_name, ift->item[i].value, 18); ne++;}

  strcpy(keyname, "extents");
  ret=iftGetIntValue(ift, 0, keyname, &i, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && i>=0) {dsr->h.extents=i; ne++;}

  strcpy(keyname, "session_error");
  ret=iftGetIntValue(ift, 0, keyname, &i, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && i>=0) {dsr->h.session_error=i; ne++;}

  strcpy(keyname, "regular");
  ret=iftGetIntValue(ift, 0, keyname, &i, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && i>=0) {dsr->h.regular=i; ne++;}

  strcpy(keyname, "dim_info");
  ret=iftGetIntValue(ift, 0, keyname, &i, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && i>=0) {dsr->h.dim_info=i; ne++;}

  /* Data array dimensions */
  strcpy(keyname, "dim");
  if((i=iftGet(ift, keyname, 0))>=0) {
    n=sscanf(ift->item[i].value, "%hd %hd %hd %hd %hd %hd %hd %hd",
             &dsr->h.dim[0], &dsr->h.dim[1], &dsr->h.dim[2], &dsr->h.dim[3],
             &dsr->h.dim[4], &dsr->h.dim[5], &dsr->h.dim[6], &dsr->h.dim[7]);
    if(n!=8) return STATUS_UNSUPPORTED;
    ne++;
  }

  /* Intent parameters */
  strcpy(keyname, "intent_p1");
  ret=iftGetFloatValue(ift, 0, keyname, &f, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && !isnan(f)) {dsr->h.intent_p1=f; ne++;}
  strcpy(keyname, "intent_p2");
  ret=iftGetFloatValue(ift, 0, keyname, &f, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && !isnan(f)) {dsr->h.intent_p2=f; ne++;}
  strcpy(keyname, "intent_p3");
  ret=iftGetFloatValue(ift, 0, keyname, &f, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && !isnan(f)) {dsr->h.intent_p3=f; ne++;}
  strcpy(keyname, "intent_code");
  ret=iftGetIntValue(ift, 0, keyname, &i, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && i>=0) {dsr->h.intent_code=i; ne++;}

  strcpy(keyname, "datatype");
  ret=iftGetIntValue(ift, 0, keyname, &i, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && i>=0) {dsr->h.datatype=i; ne++;}

  strcpy(keyname, "bitpix");
  ret=iftGetIntValue(ift, 0, keyname, &i, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && i>=0) {dsr->h.bitpix=i; ne++;}

  strcpy(keyname, "slice_start");
  ret=iftGetIntValue(ift, 0, keyname, &i, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && i>=0) {dsr->h.slice_start=i; ne++;}

  strcpy(keyname, "pixdim");
  if((i=iftGet(ift, keyname, 0))>=0) {
    n=sscanf(ift->item[i].value, "%f %f %f %f %f %f %f %f",
             &dsr->h.pixdim[0], &dsr->h.pixdim[1], &dsr->h.pixdim[2],
             &dsr->h.pixdim[3], &dsr->h.pixdim[4], &dsr->h.pixdim[5],
             &dsr->h.pixdim[6], &dsr->h.pixdim[7]);
    if(n!=8) return STATUS_UNSUPPORTED;
    ne++;
  }

  strcpy(keyname, "vox_offset");
  ret=iftGetFloatValue(ift, 0, keyname, &f, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && !isnan(f)) {dsr->h.vox_offset=f; ne++;}

  strcpy(keyname, "scl_slope");
  ret=iftGetFloatValue(ift, 0, keyname, &f, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && !isnan(f)) {dsr->h.scl_slope=f; ne++;}

  strcpy(keyname, "scl_inter");
  ret=iftGetFloatValue(ift, 0, keyname, &f, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && !isnan(f)) {dsr->h.scl_inter=f; ne++;}

  strcpy(keyname, "slice_end");
  ret=iftGetIntValue(ift, 0, keyname, &i, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && i>=0) {dsr->h.slice_end=i; ne++;}

  strcpy(keyname, "slice_code");
  ret=iftGetIntValue(ift, 0, keyname, &i, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && i>=0) {dsr->h.slice_code=i; ne++;}

  strcpy(keyname, "zyzt_units");
  ret=iftGetIntValue(ift, 0, keyname, &i, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && i>=0) {dsr->h.xyzt_units=i; ne++;}

  strcpy(keyname, "cal_max");
  ret=iftGetFloatValue(ift, 0, keyname, &f, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && !isnan(f)) {dsr->h.cal_max=f; ne++;}

  strcpy(keyname, "cal_min");
  ret=iftGetFloatValue(ift, 0, keyname, &f, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && !isnan(f)) {dsr->h.cal_min=f; ne++;}

  strcpy(keyname, "slice_duration");
  ret=iftGetFloatValue(ift, 0, keyname, &f, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && !isnan(f)) {dsr->h.slice_duration=f; ne++;}

  strcpy(keyname, "toffset");
  ret=iftGetFloatValue(ift, 0, keyname, &f, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && !isnan(f)) {dsr->h.toffset=f; ne++;}

  strcpy(keyname, "glmax");
  ret=iftGetIntValue(ift, 0, keyname, &i, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && i>=0) {dsr->h.glmax=i; ne++;}

  strcpy(keyname, "glmin");
  ret=iftGetIntValue(ift, 0, keyname, &i, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && i>=0) {dsr->h.glmin=i; ne++;}

  strcpy(keyname, "descrip");
  if((i=iftGet(ift, keyname, 0))>=0) {
    memcpy(dsr->h.descrip, ift->item[i].value, 80); ne++;}

  strcpy(keyname, "aux_file");
  if((i=iftGet(ift, keyname, 0))>=0) {
    memcpy(dsr->h.aux_file, ift->item[i].value, 24); ne++;}

  /* Transformation parameters */
  strcpy(keyname, "qform_code");
  ret=iftGetIntValue(ift, 0, keyname, &i, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && i>=0) {dsr->h.qform_code=i; ne++;}
  strcpy(keyname, "sform_code");
  ret=iftGetIntValue(ift, 0, keyname, &i, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && i>=0) {dsr->h.sform_code=i; ne++;}
  strcpy(keyname, "quatern_b");
  ret=iftGetFloatValue(ift, 0, keyname, &f, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && !isnan(f)) {dsr->h.quatern_b=f; ne++;}
  strcpy(keyname, "quatern_c");
  ret=iftGetFloatValue(ift, 0, keyname, &f, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && !isnan(f)) {dsr->h.quatern_c=f; ne++;}
  strcpy(keyname, "quatern_d");
  ret=iftGetFloatValue(ift, 0, keyname, &f, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && !isnan(f)) {dsr->h.quatern_d=f; ne++;}
  strcpy(keyname, "qoffset_x");
  ret=iftGetFloatValue(ift, 0, keyname, &f, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && !isnan(f)) {dsr->h.qoffset_x=f; ne++;}
  strcpy(keyname, "qoffset_y");
  ret=iftGetFloatValue(ift, 0, keyname, &f, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && !isnan(f)) {dsr->h.qoffset_y=f; ne++;}
  strcpy(keyname, "qoffset_z");
  ret=iftGetFloatValue(ift, 0, keyname, &f, 0); if(ret<-1) return STATUS_UNSUPPORTED;
  if(ret>=0 && !isnan(f)) {dsr->h.qoffset_z=f; ne++;}

  strcpy(keyname, "srow_x");
  if((i=iftGet(ift, keyname, 0))>=0) {
    n=sscanf(ift->item[i].value, "%f %f %f %f",
             &dsr->h.srow_x[0], &dsr->h.srow_x[1],
             &dsr->h.srow_x[2], &dsr->h.srow_x[3]);
    if(n!=4) return STATUS_UNSUPPORTED;
    ne++;
  }
  strcpy(keyname, "srow_y");
  if((i=iftGet(ift, keyname, 0))>=0) {
    n=sscanf(ift->item[i].value, "%f %f %f %f",
             &dsr->h.srow_y[0], &dsr->h.srow_y[1],
             &dsr->h.srow_y[2], &dsr->h.srow_y[3]);
    if(n!=4) return STATUS_UNSUPPORTED;
    ne++;
  }
  strcpy(keyname, "srow_z");
  if((i=iftGet(ift, keyname, 0))>=0) {
    n=sscanf(ift->item[i].value, "%f %f %f %f",
             &dsr->h.srow_z[0], &dsr->h.srow_z[1],
             &dsr->h.srow_z[2], &dsr->h.srow_z[3]);
    if(n!=4) return STATUS_UNSUPPORTED;
    ne++;
  }

  strcpy(keyname, "intent_name");
  if((i=iftGet(ift, keyname, 0))>=0) {
    memcpy(dsr->h.intent_name, ift->item[i].value, 16); ne++;}

  strcpy(keyname, "magic");
  if((i=iftGet(ift, keyname, 0))>=0) {
    memcpy(dsr->h.magic, ift->item[i].value, 4); ne++;}

  strcpy(keyname, "extension");
  if((i=iftGet(ift, keyname, 0))>=0) {
    memcpy(dsr->e.extension, ift->item[i].value, 4); ne++;}

  if(verbose>1) printf("%d field(s) edited.\n", ne);
  if(ne>0) return STATUS_OK;
  return STATUS_INVALIDHEADER; // no fields were changed
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy NIfTI-1 header contents into IFT struct.
 *  @return Returns STATUS_OK (0) when succesful..
 */
int niftiHeaderToIFT(
  /** Pointer to source NIfTI-1 header struct. */
  NIFTI_DSR *dsr,
  /** Pointer to target IFT struct. Must be initiated. 
   *  Any previous contents are deleted. */
  IFT *ift,
  /** Verbose level */
  int verbose
) {
  int i, ret;
  char *cptr, tmp[1024], tmp2[256];


  /* Check input */
  if(verbose>0) printf("niftiHeaderToIFT(dsr, ift)\n");
  if(dsr==NULL || ift==NULL) return STATUS_FAULT;

  /* Empty IFT */
  iftEmpty(ift);

  /*
   *  Fill IFT 
   */

  /* Byte order */
  if(dsr->byte_order==0) strcpy(tmp, "big"); else strcpy(tmp, "little");
  ret=iftPut(ift, "byte_order", tmp, NULL, 0);
  if(ret!=0) return STATUS_UNSUPPORTED;

  /* sizeof_hdr */
  sprintf(tmp, "%d", dsr->h.sizeof_hdr);
  ret=iftPut(ift, "sizeof_hdr", tmp, NULL, 0);
  if(ret!=0) return STATUS_UNSUPPORTED;

  /* data_type */
  strlcpy(tmp, dsr->h.data_type, 10);
  cptr=tmp; while(*cptr) {if(!isprint(cptr[0])) cptr[0]=' '; cptr++;}
  ret=iftPut(ift, "data_type", tmp, NULL, 0);
  if(ret!=0) return STATUS_UNSUPPORTED;

  /* db_name */
  strlcpy(tmp, dsr->h.db_name, 18);
  cptr=tmp; while(*cptr) {if(!isprint(*cptr)) *cptr=' '; cptr++;}
  ret=iftPut(ift, "db_name", tmp, NULL, 0);
  if(ret!=0) return STATUS_UNSUPPORTED;

  /* extents */
  sprintf(tmp, "%d", dsr->h.extents);
  ret=iftPut(ift, "extents", tmp, NULL, 0);
  if(ret!=0) return STATUS_UNSUPPORTED;

  /* session_error */
  sprintf(tmp, "%d", dsr->h.session_error);
  ret=iftPut(ift, "session_error", tmp, NULL, 0);
  if(ret!=0) return STATUS_UNSUPPORTED;

  /* regular */
  sprintf(tmp, "%d", dsr->h.regular);
  ret=iftPut(ift, "regular", tmp, NULL, 0);
  if(ret!=0) return STATUS_UNSUPPORTED;

  /* dim_info */
  sprintf(tmp, "%d", dsr->h.dim_info);
  ret=iftPut(ift, "dim_info", tmp, NULL, 0);
  if(ret!=0) return STATUS_UNSUPPORTED;

  /* Data array dimensions */
  i=0; sprintf(tmp, "%d", dsr->h.dim[i]);
  for(i=1; i<8; i++) {
    sprintf(tmp2, " %d", dsr->h.dim[i]);
    strcat(tmp, tmp2);
  }
  ret=iftPut(ift, "dim", tmp, NULL, 0);
  if(ret!=0) return STATUS_UNSUPPORTED;

  /* Intent parameters */
  sprintf(tmp, "%g", dsr->h.intent_p1);
  ret=iftPut(ift, "intent_p1", tmp, NULL, 0); if(ret!=0) return STATUS_UNSUPPORTED;
  sprintf(tmp, "%g", dsr->h.intent_p2);
  ret=iftPut(ift, "intent_p2", tmp, NULL, 0); if(ret!=0) return STATUS_UNSUPPORTED;
  sprintf(tmp, "%g", dsr->h.intent_p3);
  ret=iftPut(ift, "intent_p3", tmp, NULL, 0); if(ret!=0) return STATUS_UNSUPPORTED;
  sprintf(tmp, "%d", dsr->h.intent_code);
  ret=iftPut(ift, "intent_code", tmp, NULL, 0); if(ret!=0) return STATUS_UNSUPPORTED;

  /* datatype */
  sprintf(tmp, "%d", dsr->h.datatype);
  if(iftPut(ift, "datatype", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* bitpix */
  sprintf(tmp, "%d", dsr->h.bitpix);
  if(iftPut(ift, "bitpix", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* slice_start */
  sprintf(tmp, "%d", dsr->h.slice_start);
  if(iftPut(ift, "slice_start", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* pixdim */
  i=0; sprintf(tmp, "%g", dsr->h.pixdim[i]);
  for(i=1; i<8; i++) {
    sprintf(tmp2, " %g", dsr->h.pixdim[i]);
    strcat(tmp, tmp2);
  }
  ret=iftPut(ift, "pixdim", tmp, NULL, 0);
  if(ret!=0) return STATUS_UNSUPPORTED;

  /* vox_offset */
  sprintf(tmp, "%g", dsr->h.vox_offset);
  if(iftPut(ift, "vox_offset", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* scl_slope */
  sprintf(tmp, "%g", dsr->h.scl_slope);
  if(iftPut(ift, "scl_slope", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* scl_inter */
  sprintf(tmp, "%g", dsr->h.scl_inter);
  if(iftPut(ift, "scl_inter", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* slice_end */
  sprintf(tmp, "%d", dsr->h.slice_end);
  if(iftPut(ift, "slice_end", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* slice_code */
  sprintf(tmp, "%d", dsr->h.slice_code);
  if(iftPut(ift, "slice_code", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* xyzt_units */
  sprintf(tmp, "%d", dsr->h.xyzt_units);
  if(iftPut(ift, "xyzt_units", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* cal_max */
  sprintf(tmp, "%g", dsr->h.cal_max);
  if(iftPut(ift, "cal_max", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* cal_min */
  sprintf(tmp, "%g", dsr->h.cal_min);
  if(iftPut(ift, "cal_min", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* slice_duration */
  sprintf(tmp, "%g", dsr->h.slice_duration);
  if(iftPut(ift, "slice_duration", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* toffset */
  sprintf(tmp, "%g", dsr->h.toffset);
  if(iftPut(ift, "toffset", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* glmax */
  sprintf(tmp, "%d", dsr->h.glmax);
  if(iftPut(ift, "glmax", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* glmin */
  sprintf(tmp, "%d", dsr->h.glmin);
  if(iftPut(ift, "glmin", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* study description */
  strlcpy(tmp, dsr->h.descrip, 80);
  cptr=tmp; while(*cptr) {if(!isprint(*cptr)) *cptr=' '; cptr++;}
  if(iftPut(ift, "descrip", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* aux_file */
  strlcpy(tmp, dsr->h.aux_file, 24);
  cptr=tmp; while(*cptr) {if(!isprint(*cptr)) *cptr=' '; cptr++;}
  if(iftPut(ift, "aux_file", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* Transformation parameters */
  sprintf(tmp, "%d", dsr->h.qform_code);
  if(iftPut(ift, "qform_code", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;
  sprintf(tmp, "%d", dsr->h.sform_code);
  if(iftPut(ift, "sform_code", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;
  sprintf(tmp, "%g", dsr->h.quatern_b);
  if(iftPut(ift, "quatern_b", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;
  sprintf(tmp, "%g", dsr->h.quatern_c);
  if(iftPut(ift, "quatern_c", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;
  sprintf(tmp, "%g", dsr->h.quatern_d);
  if(iftPut(ift, "quatern_d", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;
  sprintf(tmp, "%g", dsr->h.qoffset_x);
  if(iftPut(ift, "qoffset_x", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;
  sprintf(tmp, "%g", dsr->h.qoffset_y);
  if(iftPut(ift, "qoffset_y", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;
  sprintf(tmp, "%g", dsr->h.qoffset_z);
  if(iftPut(ift, "qoffset_z", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  i=0; sprintf(tmp, "%g", dsr->h.srow_x[i]);
  for(i=1; i<4; i++) {
    sprintf(tmp2, " %g", dsr->h.srow_x[i]); strcat(tmp, tmp2);}
  if(iftPut(ift, "srow_x", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  i=0; sprintf(tmp, "%g", dsr->h.srow_y[i]);
  for(i=1; i<4; i++) {
    sprintf(tmp2, " %g", dsr->h.srow_y[i]); strcat(tmp, tmp2);}
  if(iftPut(ift, "srow_y", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  i=0; sprintf(tmp, "%g", dsr->h.srow_z[i]);
  for(i=1; i<4; i++) {
    sprintf(tmp2, " %g", dsr->h.srow_z[i]); strcat(tmp, tmp2);}
  if(iftPut(ift, "srow_z", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* intent_name */
  strlcpy(tmp, dsr->h.intent_name, 16);
  cptr=tmp; while(*cptr) {if(!isprint(*cptr)) *cptr=' '; cptr++;}
  if(iftPut(ift, "intent_name", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* Nifti magic number */
  if(iftPut(ift, "magic", dsr->h.magic, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  /* Nifti header extender */
  i=0; sprintf(tmp, "%d", dsr->e.extension[i]);
  for(i=1; i<4; i++) {
    sprintf(tmp2, " %d", dsr->e.extension[i]); strcat(tmp, tmp2);}
  if(iftPut(ift, "extension", tmp, NULL, 0)!=0) return STATUS_UNSUPPORTED;

  return STATUS_OK;
}
/*****************************************************************************/

/*****************************************************************************/
