/** @file nii2ecat.c
 *  @brief Convert NIfTI-1 PET image to ECAT 7 format.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
/* Local functions */
int imgNiftiToEcat(
  char *dbname, char *ecatfile, int scanner_type, float zoom, int verbose);
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Conversion of NIfTI-1 PET image database(s) to ECAT 7 image volume",
  "image format.",
  "Conversion can also be done using ImageConverter (.NET application).",
  " ",
  "Usage: @P [Options] database",
  " ",
  "Options:",
  " -O=<output path>",
  "     Data directory for ECAT images; by default current working directory.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "NIfTI database can be specified as a path, containing one or more databases,",
  "or filename of one NIfTI database without extension.",
  "NIfTI file(s) must be named as *.nii (single format) or *.hdr and *.img",
  "(dual format).",
  "Program reads time frame information from SIF, if SIF is located in",
  "the database directory and if file is named with *.sif extension.",
  " ",
  "Example:",
  "Conversion of all NIfTI images in directory S:\\temp\\neuro to ECAT 7",
  "images in directory C:\\data in PC/Windows:",
  "   C:",
  "   cd \\data",
  "   @P s:\\temp\\neuro",
  " ",
  "See also: ecat2nii, eframe, sif2ecat, e7emhdr, e7evhdr, flat2img, ana2ecat",
  " ",
  "Keywords: image, format conversion, ECAT, NIfTI",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int      ai, help=0, version=0, verbose=1;
  int      ret, n, errorNr=0, niiNr=0;
  int      scanner_type=0;
  float    zoom=-1.0;
  char     ecatdir[FILENAME_MAX], dbdir[FILENAME_MAX], ecatfile[FILENAME_MAX];
  char     temp[FILENAME_MAX], dbname[FILENAME_MAX], *cptr;
  DIR     *dp;
  struct dirent *de;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  ecatdir[0]=dbdir[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strncasecmp(cptr, "ZOOM=", 5)==0) {
      cptr+=5; zoom=atof_dpi(cptr); if(zoom>0.0) continue;
    } else if(strncasecmp(cptr, "SCANNER=", 8)==0) {
      cptr+=8;
      if(*cptr=='9') {scanner_type=SCANNER_ECAT931; continue;}
      if(*cptr=='A' || *cptr=='a') {scanner_type=SCANNER_ADVANCE; continue;}
      if(strcasecmp(cptr, "HRRT")==0) {
        scanner_type=SCANNER_HRRT;
        fprintf(stderr, "Warning: HRRT not fully supported.\n");
        continue;
      }
      if(strncasecmp(cptr, "HR", 2)==0) {scanner_type=SCANNER_HRPLUS; continue;}
    } else if(strncasecmp(cptr, "O=", 2)==0) {
      cptr+=2;
      if(strlen(cptr)>0) strcpy(ecatdir, cptr); else strcpy(ecatdir, ".");
      continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);

  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!dbdir[0]) {
      strcpy(dbdir, argv[ai]); continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!dbdir[0]) {
    fprintf(stderr, "Error: NIfTI image was not given."); 
    return(1);
  }
  /* Ouput path is current working path by default */
  if(!ecatdir[0]) strcpy(ecatdir, ".");

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("program := %s\n", argv[0]);
    printf("zoom := %g\n", zoom);
    printf("scanner_type := %d\n", scanner_type);
    printf("ecatdir := %s\n", ecatdir);
  }
  if(verbose>2) IMG_TEST=verbose-3; else IMG_TEST=0;


  /*
   *  Create output directory, if necessary
   */
  if(ecatdir[0]) {
    if(access(ecatdir, 0) == -1) {
      if(verbose>0) fprintf(stdout, "Creating subdirectory %s\n", ecatdir);
#ifdef WIN32
      ret=mkdir(ecatdir);
#else
      ret=mkdir(ecatdir, 00775);
#endif
      if(ret!=0) {
        fprintf(stderr, "Error: cannot created subdirectory.\n");
        return(2);
      }
    }
  } else strcpy(ecatdir, ".");


  /*
   *  Read and convert NIfTI images
   */
  dp=opendir(dbdir);
  
  if(dp==NULL) {
    /* This is not a directory, or there is no read permission */
    /* Check if this is an NIfTI database name with or without extension */
    strcpy(dbname, dbdir); niftiRemoveFNameExtension(dbname);
    ret=niftiExists(dbname, NULL, NULL, NULL, NULL, verbose-2, temp);
    if(ret==0) {
      fprintf(stderr, "Error: cannot open NIfTI image or directory %s (%s)\n",
              dbdir, temp);
      return(3);
    } else if(ret==1)
      fprintf(stderr, "Warning: no SIF for %s; frame times not available.\n",
              dbname);
    /* It seems to be an NIfTI database, therefore convert it */

    /*
     *  Make ECAT filename
     */
    cptr=strrchr(dbname, '/'); if(cptr==NULL) cptr=strrchr(dbname, '\\');
    if(cptr!=NULL) cptr++; else cptr=dbname;
#ifdef WIN32
    snprintf(ecatfile, FILENAME_MAX, "%s\\%s", ecatdir, cptr);
#else
    snprintf(ecatfile, FILENAME_MAX, "%s/%s", ecatdir, cptr);
#endif
    strcat(ecatfile, ".v");
    if(verbose>2) printf("  ECAT filename: '%s'\n", ecatfile);
    /* Remove existing ECAT file */
    if(access(ecatfile, 0)!=-1 && remove(ecatfile)!=0) {
      fprintf(stderr, "Error: cannot overwrite %s\n", ecatfile);
      errorNr++; return(5);      
    }


    /*
     *  Read NIfTI file and write ECAT image frame-by-frame
     */
    if(verbose>0) fprintf(stdout, "%s :\n", dbname);
    niiNr++;
    if(verbose>0) printf("  processing NIfTI image\n");
    ret=imgNiftiToEcat(dbname, ecatfile, scanner_type, zoom, verbose-1);
    if(ret!=STATUS_OK) {
      fprintf(stderr, "Error: %s\n", imgStatus(ret));
      return(11);
    }
    if(verbose>0)
      fprintf(stdout, "  Image saved in ECAT format in %s\n", ecatfile);
    return(0); // finished.
  }


  /* Directory was previously opened succesfully: */
  /* Conversion of all NIfTI images in the specified path */
  while((de=readdir(dp))!=NULL) {
    if(verbose>6) printf("'%s'\n", de->d_name);
    /* Remove trailing slashes */
    ret=strlen(dbdir)-1;
    if(dbdir[ret]=='/' || dbdir[ret]=='\\') dbdir[ret]=(char)0;

    /*
     *  Make NIfTI database name
     */

    /* Check that filename extension is .hdr or .nii */
    n=strlen(de->d_name);
    if(n<5) continue;
    cptr=(de->d_name)+n-4;
    if(strcasecmp(cptr, ".hdr")!=0 && strcasecmp(cptr, ".nii")!=0) continue;
    /* Make database name */
    strcpy(temp, de->d_name); temp[n-4]=(char)0;
#ifdef WIN32
    snprintf(dbname, FILENAME_MAX, "%s\\%s", dbdir, temp);
#else
    snprintf(dbname, FILENAME_MAX, "%s/%s", dbdir, temp);
#endif
    /* Check that NIfTI database exists */
    ret=niftiExists(dbname, NULL, NULL, NULL, NULL, verbose-2, NULL);
    if(ret==0) continue;
    if(verbose>0) fprintf(stdout, "%s :\n", dbname); 
    niiNr++;
    if(ret==2) fprintf(stderr,"  Warning: no SIF.\n");
    fflush(stdout); fflush(stderr);

    /*
     *  Make ECAT filename
     */
#ifdef WIN32
    snprintf(ecatfile, FILENAME_MAX-2, "%s\\%s", ecatdir, temp);
#else
    snprintf(ecatfile, FILENAME_MAX-2, "%s/%s", ecatdir, temp);
#endif
    strcat(ecatfile, ".v");
    if(verbose>1) printf("  ECAT filename: '%s'\n", ecatfile);
    
    /* Remove existing ECAT file */
    if(access(ecatfile, 0)!=-1 && remove(ecatfile)!=0) {
      fprintf(stderr, "Error: cannot overwrite %s\n", ecatfile);
      errorNr++; continue;      
    }

    /*
     *  Read NIfTI file and write ECAT image frame-by-frame
     */
    if(verbose>0) printf("  processing NIfTI image\n");
    ret=imgNiftiToEcat(dbname, ecatfile, scanner_type, zoom, verbose-1);
    if(ret!=STATUS_OK) {
      fprintf(stderr, "Error: %s\n", imgStatus(ret));
      errorNr++; continue;
    }
    if(verbose>0) fprintf(stdout, "  Image saved in ECAT format in %s\n",
                          ecatfile);

  } /* next NIfTI image */
  closedir(dp);
  if(niiNr==0) {
    fprintf(stderr, "Error: no NIfTI images were found in %s\n", dbdir);
    return(2);
  }

  return(errorNr);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
/** Read NIfTI-1 image database and write ECAT7 image frame-by-frame
\return Returns errstatus, which is STATUS_OK (0) when call was successful,
    and >0 in case of an error.
*/
int imgNiftiToEcat(
  /** NIfTI database name */
  char *dbname,
  /** ECAT image filename, output extension must be .v */
  char *ecatfile,
  /** Scanner type, 0 if not set */
  int scanner_type,
  /** recon_zoom, 0.0 or 1.0, if not known */
  float zoom,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout */
  int verbose
) {
  IMG img;
  int zi, fi, ret, file_format=0;

  if(verbose>0) printf("\nimgNiftiToEcat(%s, %s)\n", dbname, ecatfile);
  /* Check the arguments */
  if(dbname==NULL || ecatfile==NULL) return STATUS_FAULT;

  /* Read the first frame */
  imgInit(&img); fi=0;
  if(verbose>1) printf("  reading the first frame\n");
  ret=imgReadNiftiFirstFrame(dbname, &img, verbose-1);
  if(ret!=STATUS_OK) {imgEmpty(&img); return ret;}
  /* ... and then the rest of the frames */
  do {
    fi++;
    if(verbose>1) printf("  frame %d\n", fi);
    /* Set header information */
    if(fi==0)  // must be saved for reading next frame
      file_format=img._fileFormat;
    if(scanner_type>0) imgSetScanner(&img, scanner_type);
    if(zoom>0.0) img.zoom=zoom;
    for(zi=0; zi<img.dimz; zi++) img.planeNumber[zi]=zi+1;
    if(verbose>15) imgInfo(&img);
    /* Write the frame in ECAT 7 format */
    img._fileFormat=IMG_E7;
    ret=imgWriteFrame(ecatfile, fi, &img, 0); //printf("ret := %d\n", ret);
    if(ret!=STATUS_OK) break;
    if(verbose>1) printf("  frame written.\n");
    /* Try to read the next frame */
    img._fileFormat=file_format;
    ret=imgReadNiftiFrame(dbname, fi+1, &img, 0, verbose-1);
  } while(ret==0);
  imgEmpty(&img);
  if(verbose>0 && ret==STATUS_NOMATRIX) {
    fprintf(stdout, "  %d frame(s) processed.\n", fi);
  }
  if(ret!=STATUS_OK && ret!=STATUS_NOMATRIX) {
    remove(ecatfile); return ret;
  }
  /*if(verbose>2)
    fprintf(stdout, "  Image saved in ECAT format in %s\n", ecatfile);*/
  return STATUS_OK;
}
/*****************************************************************************/

/*****************************************************************************/
